% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CBA_ruleset.R
\name{CBA_ruleset}
\alias{CBA_ruleset}
\title{Constructor for Objects for Classifiers Based on Association Rules}
\usage{
CBA_ruleset(
  formula,
  rules,
  default,
  method = "first",
  weights = NULL,
  bias = NULL,
  model = NULL,
  discretization = NULL,
  description = "Custom rule set",
  ...
)
}
\arguments{
\item{formula}{A symbolic description of the model to be fitted. Has to be
of form \code{class ~ .}. The class is the variable name (part of the item
label before \code{=}).}

\item{rules}{A set of class association rules mined with \code{\link[=mineCARs]{mineCARs()}} or
\code{\link[arules:apriori]{arules::apriori()}} (from \pkg{arules}).}

\item{default}{Default class. If not specified then objects that are
not matched by rules are classified as \code{NA}.}

\item{method}{Classification method \code{"first"} found rule or \code{"majority"}.}

\item{weights}{Rule weights for the majority voting method. Either a quality measure
available in the classification rule set or a numeric vector of the same length are
the classification rule set can be specified. If missing, then equal weights are used}

\item{bias}{Class bias vector.}

\item{model}{An optional list with model information (e.g., parameters).}

\item{discretization}{A list with discretization information used by \code{\link[=predict]{predict()}} to discretize data
supplied as a \code{data.frame}.}

\item{description}{Description field used when the classifier is printed.}

\item{\dots}{Additional arguments added as list elements to the CBA object.}
}
\value{
A object of class \code{CBA} representing the trained classifier with fields:
\item{formula}{used formula.}
\item{rules}{the classifier rule base.}
\item{default}{default class label (uses partial matching against the class labels).}
\item{method}{classification method.}
\item{weights}{rule weights.}
\item{bias}{class bias vector if available.}
\item{model}{list with model description.}
\item{discretization}{discretization information.}
\item{description}{description in human readable form.}

\code{rules} returns the rule base.
}
\description{
Objects for classifiers based on association rules have class \code{CBA}.
A creator function \code{CBA_ruleset()} and several methods are provided.
}
\details{
\code{CBA_ruleset()} creates a new object of class \code{CBA} using the
provides rules as the rule base.  For method \code{"first"}, the user needs
to make sure that the rules are predictive and sorted from most to least
predictive.
}
\examples{
## Example 1: create a first-matching-rule classifier with non-redundant rules
##  sorted by confidence.
data("iris")

# discretize and create transactions
iris.disc <- discretizeDF.supervised(Species ~., iris)
trans <- as(iris.disc, "transactions")

# create rule base with CARs
cars <- mineCARs(Species ~ ., trans, parameter = list(support = .01, confidence = .8))

cars <- cars[!is.redundant(cars)]
cars <- sort(cars, by = "conf")

# create classifier and use the majority class as the default if no rule matches.
cl <- CBA_ruleset(Species ~ .,
  rules = cars,
  default = uncoveredMajorityClass(Species ~ ., trans, cars),
  method = "first")
cl

# look at the rule base
inspect(cl$rules)

# make predictions
prediction <- predict(cl, trans)
table(prediction, response(Species ~ ., trans))
accuracy(prediction, response(Species ~ ., trans))

# Example 2: use weighted majority voting.
cl <- CBA_ruleset(Species ~ .,
  rules = cars,
  default = uncoveredMajorityClass(Species ~ ., trans, cars),
  method = "majority", weights = "lift")
cl

prediction <- predict(cl, trans)
table(prediction, response(Species ~ ., trans))
accuracy(prediction, response(Species ~ ., trans))

## Example 3: Create a classifier with no rules that always predicts
##  the majority class. Note, we need cars for the structure and subset it
##  to leave no rules.
cl <- CBA_ruleset(Species ~ .,
  rules = cars[NULL],
  default = majorityClass(Species ~ ., trans))
cl

prediction <- predict(cl, trans)
table(prediction, response(Species ~ ., trans))
accuracy(prediction, response(Species ~ ., trans))
}
\seealso{
\code{\link[=mineCARs]{mineCARs()}}

Other classifiers: 
\code{\link{CBA}()},
\code{\link{CBA_helpers}},
\code{\link{FOIL}()},
\code{\link{LUCS_KDD_CBA}},
\code{\link{RCAR}()},
\code{\link{RWeka_CBA}}

Other preparation: 
\code{\link{discretizeDF.supervised}()},
\code{\link{mineCARs}()},
\code{\link{prepareTransactions}()},
\code{\link{transactions2DF}()}
}
\author{
Michael Hahsler
}
\concept{classifiers}
\concept{preparation}
