% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filter_joined.R
\name{filter_joined}
\alias{filter_joined}
\title{Filter Observations Taking Other Observations into Account}
\usage{
filter_joined(
  dataset,
  dataset_add,
  by_vars,
  join_vars,
  join_type,
  first_cond_lower = NULL,
  first_cond_upper = NULL,
  order,
  tmp_obs_nr_var = NULL,
  filter_add = NULL,
  filter_join,
  check_type = "warning"
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by the \code{by_vars} and \code{order} arguments are expected to be in the dataset.}

\item{dataset_add}{Additional dataset

The variables specified for \code{by_vars}, \code{join_vars}, and \code{order} are
expected.}

\item{by_vars}{By variables

The specified variables are used as by variables for joining the input
dataset with itself.

\emph{Permitted Values}: list of variables created by \code{exprs()}
e.g. \code{exprs(USUBJID, VISIT)}}

\item{join_vars}{Variables to keep from joined dataset

The variables needed from the other observations should be specified for
this parameter. The specified variables are added to the joined dataset
with suffix ".join". For example to select all observations with \code{AVALC == "Y"} and \code{AVALC == "Y"} for at least one subsequent visit \code{join_vars = exprs(AVALC, AVISITN)} and \code{filter_join = AVALC == "Y" & AVALC.join == "Y" & AVISITN < AVISITN.join} could be specified.

The \verb{*.join} variables are not included in the output dataset.}

\item{join_type}{Observations to keep after joining

The argument determines which of the joined observations are kept with
respect to the original observation. For example, if \code{join_type = "after"}
is specified all observations after the original observations are kept.

For example for confirmed response or BOR in the oncology setting or
confirmed deterioration in questionnaires the confirmatory assessment must
be after the assessment. Thus \code{join_type = "after"} could be used.

Whereas, sometimes you might allow for confirmatory observations to occur
prior to the observation. For example, to identify AEs occurring on or
after seven days before a COVID AE. Thus \code{join_type = "all"} could be used.

\emph{Permitted Values:} \code{"before"}, \code{"after"}, \code{"all"}}

\item{first_cond_lower}{Condition for selecting range of data (before)

If this argument is specified, the other observations are restricted from
the first observation before the current observation where the specified
condition is fulfilled up to the current observation. If the condition is
not fulfilled for any of the other observations, no observations are
considered, i.e., the observation is not flagged.

This parameter should be specified if \code{filter_join} contains summary
functions which should not apply to all observations but only from a
certain observation before the current observation up to the current
observation. For an example see the last example below.}

\item{first_cond_upper}{Condition for selecting range of data (after)

If this argument is specified, the other observations are restricted up to
the first observation where the specified condition is fulfilled. If the
condition is not fulfilled for any of the other observations, no
observations are considered, i.e., the observation is not flagged.

This parameter should be specified if \code{filter_join} contains summary
functions which should not apply to all observations but only up to the
confirmation assessment. For an example see the last example below.}

\item{order}{Order

The observations are ordered by the specified order.

For handling of \code{NA}s in sorting variables see \href{../articles/generic.html#sort_order}{Sort Order}.

\emph{Permitted Values:} list of expressions created by \code{exprs()}, e.g.,
\code{exprs(ADT, desc(AVAL))}}

\item{tmp_obs_nr_var}{Temporary observation number

The specified variable is added to the input dataset (\code{dataset}) and the
additional dataset (\code{dataset_add}). It is set to the observation number
with respect to \code{order}. For each by group (\code{by_vars}) the observation
number starts with \code{1}. The variable can be used in the conditions
(\code{filter_join}, \code{first_cond_upper}, \code{first_cond_lower}). It is not included
in the output dataset. It can also be used to select consecutive
observations or the last observation (see example below).}

\item{filter_add}{Filter for additional dataset (\code{dataset_add})

Only observations from \code{dataset_add} fulfilling the specified condition are
joined to the input dataset. If the argument is not specified, all
observations are joined.

Variables created by the \code{order} argument can be used in the condition.

The condition can include summary functions. The additional dataset is
grouped by the by variables (\code{by_vars}).}

\item{filter_join}{Condition for selecting observations

The filter is applied to the joined dataset for selecting the confirmed
observations. The condition can include summary functions like \code{all()} or
\code{any()}. The joined dataset is grouped by the original observations. I.e.,
the summary function are applied to all observations up to the confirmation
observation. For example in the oncology setting when using this function
for confirmed best overall response,  \code{filter_join = AVALC == "CR" & all(AVALC.join \%in\% c("CR", "NE")) & count_vals(var = AVALC.join, val = "NE") <= 1} selects observations with response "CR" and for all
observations up to the confirmation observation the response is "CR" or
"NE" and there is at most one "NE".}

\item{check_type}{Check uniqueness?

If \code{"warning"} or \code{"error"} is specified, the specified message is issued
if the observations of the input dataset are not unique with respect to the
by variables and the order.

\emph{Permitted Values:} \code{"none"}, \code{"warning"}, \code{"error"}}
}
\value{
A subset of the observations of the input dataset. All variables of
the input dataset are included in the output dataset.
}
\description{
The function filters observation using a condition taking other observations
into account. For example, it could select all observations with \code{AVALC == "Y"} and \code{AVALC == "Y"} for at least one subsequent observation. The input
dataset is joined with itself to enable conditions taking variables from both
the current observation and the other observations into account. The suffix
".join" is added to the variables from the subsequent observations.

An example usage might be checking if a patient received two required
medications within a certain timeframe of each other.

In the oncology setting, for example, we use such processing to check if a
response value can be confirmed by a subsequent assessment. This is commonly
used in endpoints such as best overall response.
}
\details{
The following steps are performed to produce the output dataset.
\subsection{Step 1}{
\itemize{
\item The variables specified by \code{order} are added to the additional dataset
(\code{dataset_add}).
\item The variables specified by \code{join_vars} are added to the additional dataset
(\code{dataset_add}).
\item The records from the additional dataset (\code{dataset_add}) are restricted to
those matching the \code{filter_add} condition.
}

Then the  input dataset (\code{dataset}) is joined with the restricted
additional dataset by the variables specified for \code{by_vars}. From the
additional dataset only the variables specified for \code{join_vars} are kept.
The suffix ".join" is added to those variables which are also present in
the input dataset.

For example, for \code{by_vars = USUBJID}, \code{join_vars = exprs(AVISITN, AVALC)}
and input dataset and additional dataset

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# A tibble: 2 x 4
USUBJID AVISITN AVALC  AVAL
<chr>     <dbl> <chr> <dbl>
1             1 Y         1
1             2 N         0
}\if{html}{\out{</div>}}

the joined dataset is

\if{html}{\out{<div class="sourceCode r">}}\preformatted{A tibble: 4 x 6
USUBJID AVISITN AVALC  AVAL AVISITN.join AVALC.join
<chr>     <dbl> <chr> <dbl>        <dbl> <chr>
1             1 Y         1            1 Y
1             1 Y         1            2 N
1             2 N         0            1 Y
1             2 N         0            2 N
}\if{html}{\out{</div>}}
}

\subsection{Step 2}{

The joined dataset is restricted to observations with respect to
\code{join_type} and \code{order}.

The dataset from the example in the previous step with \code{join_type = "after"} and \code{order = exprs(AVISITN)} is restricted to

\if{html}{\out{<div class="sourceCode r">}}\preformatted{A tibble: 4 x 6
USUBJID AVISITN AVALC  AVAL AVISITN.join AVALC.join
<chr>     <dbl> <chr> <dbl>        <dbl> <chr>
1             1 Y         1            2 N
}\if{html}{\out{</div>}}
}

\subsection{Step 3}{

If \code{first_cond_lower} is specified, for each observation of the input
dataset the joined dataset is restricted to observations from the first
observation where \code{first_cond_lower} is fulfilled (the observation
fulfilling the condition is included) up to the observation of the input
dataset. If for an observation of the input dataset the condition is not
fulfilled, the observation is removed.

If \code{first_cond_upper} is specified, for each observation of the input
dataset the joined dataset is restricted to observations up to the first
observation where \code{first_cond_upper} is fulfilled (the observation
fulfilling the condition is included). If for an observation of the input
dataset the condition is not fulfilled, the observation is removed.

For an example see the last example in the "Examples" section.
}

\subsection{Step 4}{

The joined dataset is grouped by the observations from the input dataset
and restricted to the observations fulfilling the condition specified by
\code{filter_join}.
}

\subsection{Step 5}{

The first observation of each group is selected and the \verb{*.join} variables
are dropped.

\strong{Note:} This function creates temporary datasets which may be much bigger
than the input datasets. If this causes memory issues, please try setting
the admiral option \code{save_memory} to \code{TRUE} (see \code{set_admiral_options()}).
This reduces the memory consumption but increases the run-time.
}
}
\examples{

library(tibble)
library(admiral)

# filter observations with a duration longer than 30 and
# on or after 7 days before a COVID AE (ACOVFL == "Y")
adae <- tribble(
  ~USUBJID, ~ADY, ~ACOVFL, ~ADURN,
  "1",        10, "N",          1,
  "1",        21, "N",         50,
  "1",        23, "Y",         14,
  "1",        32, "N",         31,
  "1",        42, "N",         20,
  "2",        11, "Y",         13,
  "2",        23, "N",          2,
  "3",        13, "Y",         12,
  "4",        14, "N",         32,
  "4",        21, "N",         41
)

filter_joined(
  adae,
  dataset_add = adae,
  by_vars = exprs(USUBJID),
  join_vars = exprs(ACOVFL, ADY),
  join_type = "all",
  order = exprs(ADY),
  filter_join = ADURN > 30 & ACOVFL.join == "Y" & ADY >= ADY.join - 7
)

# filter observations with AVALC == "Y" and AVALC == "Y" at a subsequent visit
data <- tribble(
  ~USUBJID, ~AVISITN, ~AVALC,
  "1",      1,        "Y",
  "1",      2,        "N",
  "1",      3,        "Y",
  "1",      4,        "N",
  "2",      1,        "Y",
  "2",      2,        "N",
  "3",      1,        "Y",
  "4",      1,        "N",
  "4",      2,        "N",
)

filter_joined(
  data,
  dataset_add = data,
  by_vars = exprs(USUBJID),
  join_vars = exprs(AVALC, AVISITN),
  join_type = "after",
  order = exprs(AVISITN),
  filter_join = AVALC == "Y" & AVALC.join == "Y" & AVISITN < AVISITN.join
)

# select observations with AVALC == "CR", AVALC == "CR" at a subsequent visit,
# only "CR" or "NE" in between, and at most one "NE" in between
data <- tribble(
  ~USUBJID, ~AVISITN, ~AVALC,
  "1",      1,        "PR",
  "1",      2,        "CR",
  "1",      3,        "NE",
  "1",      4,        "CR",
  "1",      5,        "NE",
  "2",      1,        "CR",
  "2",      2,        "PR",
  "2",      3,        "CR",
  "3",      1,        "CR",
  "4",      1,        "CR",
  "4",      2,        "NE",
  "4",      3,        "NE",
  "4",      4,        "CR",
  "4",      5,        "PR"
)

filter_joined(
  data,
  dataset_add = data,
  by_vars = exprs(USUBJID),
  join_vars = exprs(AVALC),
  join_type = "after",
  order = exprs(AVISITN),
  first_cond_upper = AVALC.join == "CR",
  filter_join = AVALC == "CR" & all(AVALC.join \%in\% c("CR", "NE")) &
    count_vals(var = AVALC.join, val = "NE") <= 1
)

# select observations with AVALC == "PR", AVALC == "CR" or AVALC == "PR"
# at a subsequent visit at least 20 days later, only "CR", "PR", or "NE"
# in between, at most one "NE" in between, and "CR" is not followed by "PR"
data <- tribble(
  ~USUBJID, ~ADY, ~AVALC,
  "1",         6, "PR",
  "1",        12, "CR",
  "1",        24, "NE",
  "1",        32, "CR",
  "1",        48, "PR",
  "2",         3, "PR",
  "2",        21, "CR",
  "2",        33, "PR",
  "3",        11, "PR",
  "4",         7, "PR",
  "4",        12, "NE",
  "4",        24, "NE",
  "4",        32, "PR",
  "4",        55, "PR"
)

filter_joined(
  data,
  dataset_add = data,
  by_vars = exprs(USUBJID),
  join_vars = exprs(AVALC, ADY),
  join_type = "after",
  order = exprs(ADY),
  first_cond_upper = AVALC.join \%in\% c("CR", "PR") & ADY.join - ADY >= 20,
  filter_join = AVALC == "PR" &
    all(AVALC.join \%in\% c("CR", "PR", "NE")) &
    count_vals(var = AVALC.join, val = "NE") <= 1 &
    (
      min_cond(var = ADY.join, cond = AVALC.join == "CR") >
        max_cond(var = ADY.join, cond = AVALC.join == "PR") |
        count_vals(var = AVALC.join, val = "CR") == 0
    )
)

# select observations with CRIT1FL == "Y" at two consecutive visits or at the last visit
data <- tribble(
  ~USUBJID, ~AVISITN, ~CRIT1FL,
  "1",      1,        "Y",
  "1",      2,        "N",
  "1",      3,        "Y",
  "1",      5,        "N",
  "2",      1,        "Y",
  "2",      3,        "Y",
  "2",      5,        "N",
  "3",      1,        "Y",
  "4",      1,        "Y",
  "4",      2,        "N",
)

filter_joined(
  data,
  dataset_add = data,
  by_vars = exprs(USUBJID),
  tmp_obs_nr_var = tmp_obs_nr,
  join_vars = exprs(CRIT1FL),
  join_type = "all",
  order = exprs(AVISITN),
  filter_join = CRIT1FL == "Y" & CRIT1FL.join == "Y" &
    (tmp_obs_nr + 1 == tmp_obs_nr.join | tmp_obs_nr == max(tmp_obs_nr.join))
)

# first_cond_lower and first_cond_upper argument
myd <- tribble(
  ~subj, ~day, ~val,
  "1",      1, "++",
  "1",      2, "-",
  "1",      3, "0",
  "1",      4, "+",
  "1",      5, "++",
  "1",      6, "-",
  "2",      1, "-",
  "2",      2, "++",
  "2",      3, "+",
  "2",      4, "0",
  "2",      5, "-",
  "2",      6, "++"
)

# select "0" where all results from the first "++" before the "0" up to the "0"
# (excluding the "0") are "+" or "++"
filter_joined(
  myd,
  dataset_add = myd,
  by_vars = exprs(subj),
  order = exprs(day),
  join_vars = exprs(val),
  join_type = "before",
  first_cond_lower = val.join == "++",
  filter_join = val == "0" & all(val.join \%in\% c("+", "++"))
)

# select "0" where all results from the "0" (excluding the "0") up to the first
# "++" after the "0" are "+" or "++"
filter_joined(
  myd,
  dataset_add = myd,
  by_vars = exprs(subj),
  order = exprs(day),
  join_vars = exprs(val),
  join_type = "after",
  first_cond_upper = val.join == "++",
  filter_join = val == "0" & all(val.join \%in\% c("+", "++"))
)
}
\seealso{
\code{\link[=count_vals]{count_vals()}}, \code{\link[=min_cond]{min_cond()}}, \code{\link[=max_cond]{max_cond()}}

Utilities for Filtering Observations: 
\code{\link{count_vals}()},
\code{\link{filter_exist}()},
\code{\link{filter_extreme}()},
\code{\link{filter_not_exist}()},
\code{\link{filter_relative}()},
\code{\link{max_cond}()},
\code{\link{min_cond}()}
}
\concept{utils_fil}
\keyword{utils_fil}
