# Generated by using Rcpp::compileAttributes() -> do not edit by hand
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

#' Accelerometer Artifact Correction
#'
#' Corrects abnormally high count values in accelerometer data by replacing 
#' such values with the average of neighboring count values. Returns integer 
#' vector despite the average calculation often producing a decimal; this 
#' follows the convention used in the NCI's SAS programs 
#' (\url{http://riskfactor.cancer.gov/tools/nhanes_pam}). 
#' 
#' 
#' @param counts Integer vector with accelerometer count values.
#' @param thresh Integer value specifying the smallest count value that should 
#' be considered an artifact.
#' @param counts_classify Integer vector with accelerometer count values to 
#' base artifact classification on, but not to adjust. Mainly included for 
#' triaxial data, where you might want to define artifacts based on 
#' vertical-axis counts but then actually adjust the triaxial sum or vector 
#' magnitude counts.
#'
#'
#' @return Integer vector equivalent to \code{counts} except where artifacts 
#' were adjusted.
#' 
#' 
#' @references 
#' National Cancer Institute. Risk factor monitoring and methods: SAS programs 
#' for analyzing NHANES 2003-2004 accelerometer data. Available at: 
#' \url{http://riskfactor.cancer.gov/tools/nhanes_pam}. Accessed Aug. 19, 2018.
#'
#'
#' @examples
#' # Load accelerometer data for first 5 participants in NHANES 2003-2004
#' data(unidata)
#' 
#' # Get data from ID number 21007
#' counts.part3 <- unidata[unidata[, "seqn"] == 21007, "paxinten"]
#' 
#' # Replace counts > 10,000 with average of neighboring values
#' counts.part3.corrected <- artifacts(counts = counts.part3, thresh = 10000)
#'
#'@export
artifacts <- function(counts, thresh, counts_classify = NULL) {
    .Call(`_accelerometry_artifacts`, counts, thresh, counts_classify)
}

blockaves_i <- function(x, window) {
    .Call(`_accelerometry_blockaves_i`, x, window)
}

blockaves_i_max <- function(x, window) {
    .Call(`_accelerometry_blockaves_i_max`, x, window)
}

blockaves_n <- function(x, window) {
    .Call(`_accelerometry_blockaves_n`, x, window)
}

blockaves_n_max <- function(x, window) {
    .Call(`_accelerometry_blockaves_n_max`, x, window)
}

blocksums_i <- function(x, window) {
    .Call(`_accelerometry_blocksums_i`, x, window)
}

blocksums_i_max <- function(x, window) {
    .Call(`_accelerometry_blocksums_i_max`, x, window)
}

blocksums_n <- function(x, window) {
    .Call(`_accelerometry_blocksums_n`, x, window)
}

blocksums_n_max <- function(x, window) {
    .Call(`_accelerometry_blocksums_n_max`, x, window)
}

#' Physical Activity Bout Detection
#' 
#' Identify bouts of physical activity based on a vector of accelerometer count 
#' values.
#' 
#' If \code{nci = FALSE}, the algorithm uses a moving window to go through 
#' every possible interval of length \code{bout_length} in \code{counts}. Any 
#' interval in which all counts are >= \code{tol_lower} and <= 
#' \code{tol_upper}, and no more than \code{tol} counts are less than 
#' \code{thresh_lower} or greater than \code{thresh_upper}, is classified as an 
#' activity bout.
#' 
#' If \code{nci = TRUE}, activity bouts are classified according to the 
#' algorithm used in the NCI's SAS programs. Briefly, this algorithm defines an 
#' activity bout as an interval of length \code{bout_length} that starts with a 
#' count value in \code{[thresh_lower, thresh_upper]} and has no more than 
#' \code{tol} counts outside of that range. If these criteria are met, the bout 
#' continues until there are \code{(tol + 1)} consecutive minutes outside of 
#' \code{[thresh_lower, thresh_upper]}. The parameters \code{tol_lower} and 
#' \code{tol_upper} are not used.
#' 
#' If the user allows for a tolerance (e.g. \code{tol = 2}) and does not use 
#' the NCI algorithm (i.e. \code{nci = FALSE}), specifying a non-zero value for 
#' \code{tol_lower} is highly recommended. Otherwise the algorithm will tend to 
#' classify minutes immediately before and after an activity bout as being part 
#' of the bout.
#' 
#' Specifying \code{thresh_lower} while using an arbitrarily large value for 
#' \code{thresh_upper} is generally recommended. Specifying both of these 
#' parameters can be overly restrictive in that the algorithm may miss bouts of 
#' activity in which counts are consistently high, but not exclusively in one 
#' intensity range.
#' 
#' 
#' @inheritParams weartime
#' 
#' @param weartime Integer vector with 1's for wear time minutes and 0's for 
#' non-wear time minutes.
#' 
#' @param bout_length Integer value specifying minimum length of an activity 
#' bout.
#' 
#' @param thresh_lower Integer value specifying lower bound for count values to 
#' be included for the intensity level. 
#' 
#' @param thresh_upper Integer value specifying upper bound for count values to 
#' be included for the intensity level.
#' 
#' @param tol Integer value specifying number of minutes with count values 
#' outside of [\code{thresh_lower}, \code{thresh_upper}] to allow during an 
#' activity bout.
#' 
#' @param tol_lower Integer value specifying lower cut-off for count values 
#' outside of intensity range during an activity bout.
#' 
#' @param tol_upper Integer value specifying upper cut-off for count values 
#' outside of intensity range during an activity bout.
#' 
#' @param days_distinct Logical value for whether to treat each day of data as 
#' distinct, i.e. identify non-wear time and activity bouts for day 1, then day 
#' 2, etc. If \code{FALSE}, algorithm is applied to full monitoring period 
#' continuously. If protocol has participants remove accelerometer for sleep, 
#' strongly recommend setting to \code{FALSE} to capture non-wear periods that 
#' start between 11 pm and midnight. Function assumes that first 1440 data 
#' points are day 1, next 1440 are day 2, and so on.
#' 
#' 
#' @return Integer vector with 1's for minutes that are part of an activity 
#' bout and 0's for minutes that are not.
#' 
#' 
#' @references 
#' National Cancer Institute. Risk factor monitoring and methods: SAS programs 
#' for analyzing NHANES 2003-2004 accelerometer data. Available at: 
#' \url{http://riskfactor.cancer.gov/tools/nhanes_pam}. Accessed Aug. 19, 2018.
#' 
#' Acknowledgment: This material is based upon work supported by the National 
#' Science Foundation Graduate Research Fellowship under Grant No. DGE-0940903.
#' 
#' 
#' @examples
#' # Load accelerometer data for first 5 participants in NHANES 2003-2004
#' data(unidata)
#' 
#' # Get data from ID number 21005
#' counts.part1 <- unidata[unidata[, "seqn"] == 21005, "paxinten"]
#' 
#' # Identify periods of valid wear time
#' wear.part1 <- weartime(counts = counts.part1)
#' 
#' # Identify moderate-to-vigorous activity bouts
#' mvpa.bouts <- bouts(counts = counts.part1, weartime = wear.part1, 
#'                     thresh_lower = 2020)
#' 
#' 
#' @export
bouts <- function(counts, weartime = NULL, bout_length = 10L, thresh_lower = 0L, thresh_upper = 100000L, tol = 0L, tol_lower = 0L, tol_upper = 100000L, nci = FALSE, days_distinct = FALSE) {
    .Call(`_accelerometry_bouts`, counts, weartime, bout_length, thresh_lower, thresh_upper, tol, tol_lower, tol_upper, nci, days_distinct)
}

#' Cut Count Values into Intensity Ranges
#' 
#' Given a vector of accelerometer count values, classifies each count value 
#' into intensity level 1, 2, 3, 4, or 5 (typically representing sedentary, 
#' light, lifestyle, moderate, and vigorous).
#' 
#' 
#' @inheritParams artifacts
#' 
#' @param int_cuts Numeric vector with four cutpoints from which five intensity 
#' ranges are derived. For example, \code{int_cuts = c(100, 760, 2020, 5999)} 
#' creates: 0-99 = intensity 1; 100-759 = intensity level 2; 760-2019 = 
#' intensity 3; 2020-5998 = intensity 4; >= 5999 = intensity 5.
#' 
#' 
#' @return Integer vector.
#' 
#' 
#' @examples
#' # Load accelerometer data for first 5 participants in NHANES 2003-2004
#' data(unidata)
#' 
#' # Get data from ID number 21005
#' counts.part1 <- unidata[unidata[, "seqn"] == 21005, "paxinten"]
#' 
#' # Cut into 5 intensity levels and plot
#' intensity.part1 <- cut_counts(counts = counts.part1)
#' plot(intensity.part1)
#' 
#' 
#' @export
cut_counts <- function(counts, int_cuts = as.integer( c(100, 760, 2020, 5999))) {
    .Call(`_accelerometry_cut_counts`, counts, int_cuts)
}

#' Physical Activity Intensities
#' 
#' Given a vector of accelerometer count values, calculates time spent in 5 
#' mutually exclusive user-defined intensity levels (typically representing
#' sedentary, light, lifestyle, moderate, and vigorous) as well as the total 
#' counts accumulated in various intensities. Non-wear time should be removed 
#' from \code{counts} before calling \code{intensities} to avoid overestimating 
#' sedentary time.
#' 
#' 
#' @inheritParams artifacts
#' 
#' @param int_cuts Numeric vector with four cutpoints from which five intensity 
#' ranges are derived. For example, \code{int_cuts = c(100, 760, 2020, 5999)} 
#' creates: 0-99 = intensity 1; 100-759 = intensity level 2; 760-2019 = 
#' intensity 3; 2020-5998 = intensity 4; >= 5999 = intensity 5.
#' 
#' 
#' @return Integer vector of length 16 in which the first eight values are 
#' minutes in intensities 1, 2, 3, 4, 5, 2-3, 4-5, and 2-5, and the next eight 
#' are counts accumulated during time spent in each of those intensities.
#' 
#' 
#' @examples
#' # Load accelerometer data for first 5 participants in NHANES 2003-2004
#' data(unidata)
#' 
#' # Get data from ID number 21005
#' counts.part1 <- unidata[unidata[, "seqn"] == 21005, "paxinten"]
#' 
#' # Create vector of counts during valid wear time only
#' counts.part1.wear <- counts.part1[weartime(counts = counts.part1) == 1]
#' 
#' # Calculate physical activity intensity variables
#' intensity.variables <- intensities(counts = counts.part1.wear)
#' 
#' 
#' @export
intensities <- function(counts, int_cuts = as.integer( c(100, 760, 2020, 5999))) {
    .Call(`_accelerometry_intensities`, counts, int_cuts)
}

movingaves_i <- function(x, window) {
    .Call(`_accelerometry_movingaves_i`, x, window)
}

movingaves_i_max <- function(x, window) {
    .Call(`_accelerometry_movingaves_i_max`, x, window)
}

movingaves_n <- function(x, window) {
    .Call(`_accelerometry_movingaves_n`, x, window)
}

movingaves_n_max <- function(x, window) {
    .Call(`_accelerometry_movingaves_n_max`, x, window)
}

#' Calculating Daily Averages for Physical Activity Variables
#' 
#' Not intended for direct use.
#' 
#' @param dayvars Numeric matrix with daily physical activity variables.
#' 
#' @param rows Numeric value specifying number of rows in the matrix to be 
#' returned.
#' 
#' @param days Integer value specifying minimum number of valid days a 
#' participant must have to be included.
#' 
#' @param wk Integer value specifying minimum number of valid weekdays a 
#' participant must have to be included.
#' 
#' @param we Integer value specifying minimum number of valid weekend days a 
#' participant must have to be included.
#' 
#' 
#' @return
#' Numeric matrix.
#' 
#' 
#' @export
personvars <- function(dayvars, rows, days, wk, we) {
    .Call(`_accelerometry_personvars`, dayvars, rows, days, wk, we)
}

rle2_i <- function(x, indices) {
    .Call(`_accelerometry_rle2_i`, x, indices)
}

rle2_n <- function(x, indices) {
    .Call(`_accelerometry_rle2_n`, x, indices)
}

sedbreaks <- function(counts, weartime, thresh) {
    .Call(`_accelerometry_sedbreaks`, counts, weartime, thresh)
}

sedbreaks_flags <- function(counts, weartime, thresh) {
    .Call(`_accelerometry_sedbreaks_flags`, counts, weartime, thresh)
}

#' Wear Time Classification
#' 
#' Classifies wear time vs. non-wear time based on a vector of accelerometer 
#' count values.
#' 
#' If \code{nci = FALSE}, the algorithm uses a moving window to go through 
#' every possible interval of length \code{window} in \code{counts}. Any 
#' interval in which no more than \code{tol} counts are non-zero, and those 
#' are still < \code{tol.upper}, is classified as non-wear time. 
#' 
#' If \code{nci = TRUE}, non-wear time is classified according to the algorithm 
#' used in the NCI's SAS programs. Briefly, this algorithm defines a non-wear 
#' period as an interval of length \code{window} that starts with a count value 
#' of 0, does not contain any periods with \code{(tol + 1)} consecutive 
#' non-zero count values, and does not contain any counts > \code{tol.upper}. 
#' If these criteria are met, the non-wear period continues until there are 
#' \code{(tol + 1)} consecutive non-zero count values or a single count value > 
#' \code{tol.upper}.
#' 
#' 
#' @param counts Integer vector with accelerometer count values.
#'
#' @param window Integer value specifying minimum length of a non-wear 
#' period.
#' 
#' @param tol Integer value specifying tolerance for non-wear algorithm, i.e. 
#' number of seconds/minutes with non-zero counts allowed during a non-wear 
#' interval.
#' 
#' @param tol_upper Integer value specifying maximum count value for a 
#' second/minute with non-zero counts during a non-wear interval.
#' 
#' @param nci Logical value for whether to use algorithm from NCI's SAS 
#' programs. See \bold{Details}.
#' 
#' @param days_distinct Logical value for whether to treat each day of data as 
#' distinct, as opposed to analyzing the entire monitoring period as one 
#' continuous segment. For minute-to-minute counts, strongly recommend setting 
#' to \code{FALSE} to correctly classify time near midnight.
#' 
#' @param units_day Integer value specifying how many data point are in a day. 
#' Typically either 1440 or 86400 depending on whether count values are 
#' minute-to-minute or second-to-second.
#' 
#' 
#' @return Integer vector with 1's for valid wear time and 0's for non-wear 
#' time.
#' 
#' 
#' @references 
#' National Cancer Institute. Risk factor monitoring and methods: SAS programs 
#' for analyzing NHANES 2003-2004 accelerometer data. Available at: 
#' \url{http://riskfactor.cancer.gov/tools/nhanes_pam}. Accessed Aug. 19, 2018.
#' 
#' Acknowledgment: This material is based upon work supported by the National 
#' Science Foundation Graduate Research Fellowship under Grant No. DGE-0940903.
#' 
#' 
#' @examples
#' # Load accelerometer data for first 5 participants in NHANES 2003-2004
#' data(unidata)
#' 
#' # Get data from ID number 21005
#' counts.part1 <- unidata[unidata[, "seqn"] == 21005, "paxinten"]
#' 
#' # Identify periods of valid wear time
#' weartime.flag <- weartime(counts = counts.part1)
#' 
#' 
#' @export
weartime <- function(counts, window = 60L, tol = 0L, tol_upper = 99L, nci = FALSE, days_distinct = FALSE, units_day = 1440L) {
    .Call(`_accelerometry_weartime`, counts, window, tol, tol_upper, nci, days_distinct, units_day)
}

