# This function generates noise
.noise <- function(dims, n = 100, type = c("artsy-knn", "knn", "svm", "rf", "perlin", "cubic", "simplex", "worley"),
                   k = 20, limits = c(0, 1)) {
  type <- match.arg(type)
  if (type == "artsy-knn") {
    if (length(dims) == 1) {
      vec <- expand.grid(limits[1], seq(limits[1], limits[2], length.out = dims))
    } else if (length(dims) == 2) {
      vec <- expand.grid(seq(limits[1], limits[2], length.out = dims[1]), seq(limits[1], limits[2], length.out = dims[2]))
    }
    z <- c_noise_knn(stats::runif(n), stats::runif(n), stats::runif(n), vec[, 1], vec[, 2], k, n)
  } else if (type == "svm") {
    train <- data.frame(
      x = stats::runif(n, limits[1], limits[2]),
      y = stats::runif(n, limits[1], limits[2]),
      z = stats::runif(n, limits[1], limits[2])
    )
    fit <- e1071::svm(formula = z ~ x + y, data = train)
    xsequence <- seq(limits[1], limits[2], length = dims[1])
    ysequence <- seq(limits[1], limits[2], length = dims[2])
    canvas <- expand.grid(xsequence, ysequence)
    colnames(canvas) <- c("x", "y")
    z <- predict(fit, newdata = canvas)
  } else if (type == "knn") {
    train <- data.frame(
      x = stats::runif(n, limits[1], limits[2]),
      y = stats::runif(n, limits[1], limits[2]),
      z = stats::runif(n, limits[1], limits[2])
    )
    fit <- kknn::train.kknn(formula = z ~ x + y, data = train, kmax = k)
    xsequence <- seq(limits[1], limits[2], length = dims[1])
    ysequence <- seq(limits[1], limits[2], length = dims[2])
    canvas <- expand.grid(xsequence, ysequence)
    colnames(canvas) <- c("x", "y")
    z <- predict(fit, newdata = canvas)
  } else if (type == "rf") {
    train <- data.frame(
      x = stats::runif(n, limits[1], limits[2]),
      y = stats::runif(n, limits[1], limits[2]),
      z = stats::runif(n, limits[1], limits[2])
    )
    fit <- randomForest::randomForest(formula = z ~ x + y, data = train)
    xsequence <- seq(limits[1], limits[2], length = dims[1])
    ysequence <- seq(limits[1], limits[2], length = dims[2])
    canvas <- expand.grid(xsequence, ysequence)
    colnames(canvas) <- c("x", "y")
    z <- predict(fit, newdata = canvas)
  } else if (type == "perlin") {
    z <- ambient::noise_perlin(dims, frequency = stats::runif(1, 0.001, 0.04))
    z <- (z - range(z)[1])/diff(range(z)) * diff(limits) + limits[1]
  } else if (type == "cubic") {
    z <- ambient::noise_cubic(dims, frequency = stats::runif(1, 0.001, 0.04))
    z <- (z - range(z)[1])/diff(range(z)) * diff(limits) + limits[1]
  } else if (type == "simplex") {
    z <- ambient::noise_simplex(dims, frequency = stats::runif(1, 0.001, 0.04))
    z <- (z - range(z)[1])/diff(range(z)) * diff(limits) + limits[1]
  } else if (type == "worley") {
    z <- ambient::noise_worley(dims)
    z <- (z - range(z)[1])/diff(range(z)) * diff(limits) + limits[1]
  }
  return(matrix(z, nrow = dims[1], ncol = dims[2]))
}

# This function performs validation checks on the standardized input arguments of a function
.checkUserInput <- function(background = NULL, resolution = NULL, iterations = NULL) {
  if (!is.null(background) && length(background) != 1) {
    stop("'background' must be a single character")
  }
  if (!is.null(resolution) && (resolution < 1 || resolution %% 1 != 0)) {
    stop("'resolution' must be a single value > 0")
  }
  if (!is.null(iterations) && (iterations < 1 || iterations %% 1 != 0)) {
    stop("'iterations' must be a single integer > 0")
  }
}

# This function turns a matrix into a data frame with columns x, y, and z
.unraster <- function(x, names) {
  newx <- data.frame(x = rep(1:ncol(x), times = ncol(x)), y = rep(1:nrow(x), each = nrow(x)), z = c(x))
  colnames(newx) <- names
  return(newx)
}

# This function takes a point (x, y) and returns a warped point (x, y)
.warp <- function(p, warpDistance, resolution, angles = NULL, distances = NULL) {
  if (is.null(angles)) {
    angles <- .noise(c(resolution, resolution), type = sample(c("svm", "perlin", "cubic", "simplex"), size = 1), limits = c(-pi, pi))
  } else if (is.character(angles)) {
    angles <- .noise(c(resolution, resolution), type = angles, limits = c(-pi, pi))
  } else if (is.matrix(angles)) {
    if (nrow(angles) != resolution || ncol(angles) != resolution) {
      stop(paste0("'angles' should be a ", resolution, " x ", resolution, " matrix"))
    }
  }
  if (is.null(distances)) {
    distances <- .noise(c(resolution, resolution), type = sample(c("knn", "perlin", "cubic", "simplex"), size = 1), limits = c(0, warpDistance))
  } else if (is.character(distances)) {
    distances <- .noise(c(resolution, resolution), type = distances, limits = c(0, warpDistance))
  } else if (is.matrix(distances)) {
    if (nrow(distances) != resolution || ncol(distances) != resolution) {
      stop(paste0("'distances' should be a ", resolution, " x ", resolution, " matrix"))
    }
  }
  return(matrix(c(p[, 1] + c(cos(angles)) * c(distances), p[, 2] + c(sin(angles)) * c(distances)), ncol = 2))
}
