% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nlopt.ui.general.R
\name{nlopt.ui.general}
\alias{nlopt.ui.general}
\title{Function for the determination of the population thresholds an inconclusive interval for bi-normal distributed test scores.}
\usage{
nlopt.ui.general(Se = 0.55, Sp = 0.55, distribution = "norm",
  parameters.d0 = c(mean = 0, sd = 1), parameters.d1 = c(mean = 1, sd = 1),
  overlap.interval = NULL, intersection = NULL, start = NULL,
  print.level = 0)
}
\arguments{
\item{Se}{(default = .55). Desired sensitivity of the test scores within the uncertain interval. A value <= .5 is not allowed, while a value larger than .6 is not recommended.}

\item{Sp}{(default = .55). Desired specificity of the test scores within the uncertain interval. A value <= .5 is not allowed, while a value larger than .6 is not recommended.}

\item{distribution}{Name of the continuous distribution, exact as used in R package stats. Equal to density function minus d. For instance 'norm'.}

\item{parameters.d0}{Named vector of population values or estimates of the parameters of the distribution of the test scores of the persons without the targeted condition. For instance \code{c(mean = 0, sd = 1)}. This distribution should have the lower values.}

\item{parameters.d1}{Named vector of population values or estimates of the parameters of the distribution of the test scores of the persons with the targeted condition. For instance \code{c(mean = 1, sd = 1)}. The test scores of d1 should have higher values than d1. If not, use -(test scores). This distribution should have the higher values.}

\item{overlap.interval}{A vector with a raw estimate of the lower and upper relevant of the overlap of the two distributions. If NULL, set to quantile .001 of the distribution of persons with the targeted condition and quantile .999 of the distribution of persons without the condition. Please check whether this is a good estimate of the relevant overlap.}

\item{intersection}{Default NULL. If not null, the supplied value is used as the estimate of the intersection of the two bi-normal distributions. Otherwise, it is calculated.}

\item{start}{Default NULL. If not null, the first two values of the supplied vector are used as the starting values for the \code{nloptr} optimization function.}

\item{print.level}{Default is 0. The option print_level controls how much output is shown during the optimization process. Possible values: 0 (default)    no output; 1    show iteration number and value of objective function; 2    1 + show value of (in)equalities; 3    2 + show value of controls.}
}
\value{
List of values:
\describe{
  \item{$status: }{Integer value with the status of the optimization (0 is success).}
  \item{$message: }{More informative message with the status of the optimization}
  \item{$results: }{Vector with the following values:}
     \itemize{
      \item{exp.Sp.ui: }{The population value of the specificity in the Uncertain Interval, given mu0, sd0, mu1 and sd1. This value should be very near the supplied value of Sp.}
      \item{exp.Sp.ui: }{The population value of the sensitivity in the Uncertain Interval, given mu0, sd0, mu1 and sd1. This value should be very near the supplied value of Se.}
      \item{vector of parameter values of d0}{The values that have been supplied for d0.}
      \item{vector of parameter values of d1}{The values that have been supplied for d1.}
      }
  \item{$solution: }{Vector with the following values:}
     \itemize{
      \item{L: }{The population value of the lower threshold of the uncertain interval.}
      \item{U: }{The population value of the upper threshold of the uncertain interval.}
      }
}
}
\description{
Function for the determination of the population thresholds an inconclusive interval for bi-normal distributed test scores.
}
\details{
The function can be used to determinate the uncertain interval of the two continuous distributions.
The Uncertain Interval is defined as an interval below and above the intersection of the two distributions, with a sensitivity and specificity below a desired value (default .55).

Only a single intersection is assumed (or an second intersection where the overlap is negligible).
}
\examples{
# A simple test model:
nlopt.ui.general(Se = .55, Sp = .55,
                 distribution = 'norm',
                 parameters.d0 = c(mean = 0, sd = 1),
                 parameters.d1 = c(mean = 1, sd = 1),
                 overlap.interval=c(-2,3))
# Standard procedure when using a continous distribution:
nlopt.ui.general(parameters.d0 = c(mean = 0, sd = 1),
                 parameters.d1 = c(mean = 1.6, sd = 2))
emp.AUC <- function(norm, abnorm) {
  o = outer(abnorm, norm, "-")
  mean((o > 0) + .5 * (o == 0))
}

library(MASS)
library(car)
# gamma distributed data
set.seed(4)
d0 = rgamma(100, shape=2, rate=.5)
d1 = rgamma(100, shape=7.5, rate=1)
# 1. obtain parameters
parameters.d0=fitdistr(d0, 'gamma')$estimate
parameters.d1=fitdistr(d1, 'gamma')$estimate
# 2. test if supposed distributions (gamma) is fitting
qqPlot(d0, distribution='gamma', shape=parameters.d0['shape'])
qqPlot(d1, distribution='gamma', shape=parameters.d1['shape'])
# 3. draw curves and determine overlap
curve(dgamma(x, shape=parameters.d0['shape'], rate=parameters.d0['rate']), from=0, to=16)
curve(dgamma(x, shape=parameters.d1['shape'], rate=parameters.d1['rate']), from=0, to=16, add=TRUE)
overlap.interval=c(1, 15) # ignore intersection at 0; observe large overlap
# 4. get empirical AUC
emp.AUC(d0, d1)
# about .65 --> Poor
# .90-1 = excellent (A)
# .80-.90 = good (B)
# .70-.80 = fair (C)
# .60-.70 = poor (D)
# .50-.60 = fail (F)
# 5. Get uncertain interval
(res=nlopt.ui.general (Se = .57,
                       Sp = .57,
                       distribution = 'gamma',
                       parameters.d0 = parameters.d0,
                       parameters.d1 = parameters.d1,
                       overlap.interval,
                       intersection = NULL,
                       start = NULL,
                       print.level = 0))
abline(v=c(res$intersection, res$solution))
# 6. Assess improvement when diagnosing outside the uncertain interval
sel.d0 = d0 < res$solution[1] |  d0 > res$solution[2]
sel.d1 = d1 < res$solution[1] |  d1 > res$solution[2]
(percentage.selected.d0 = sum(sel.d0) / length(d0))
(percentage.selected.d1 = sum(sel.d1) / length(d1))
emp.AUC(d0[sel.d0], d1[sel.d1])
# AUC for selected scores outside the uncertain interval
emp.AUC(d0[!sel.d0], d1[!sel.d1])
# AUC for deselected scores; worst are deselected
# weibull distributed data
set.seed(4)
d0 = rweibull(100, shape=3, scale=50)
d1 = rweibull(100, shape=3, scale=70)
# 1. obtain parameters
parameters.d0=fitdistr(d0, 'weibull')$estimate
parameters.d1=fitdistr(d1, 'weibull')$estimate
# 2. test if supposed distributions (gamma) is fitting
qqPlot(d0, distribution='weibull', shape=parameters.d0['shape'])
qqPlot(d1, distribution='weibull', shape=parameters.d1['shape'])
# 3. draw curves and determine overlap
curve(dweibull(x, shape=parameters.d0['shape'],
      scale=parameters.d0['scale']), from=0, to=150)
curve(dweibull(x, shape=parameters.d1['shape'],
      scale=parameters.d1['scale']), from=0, to=150, add=TRUE)
overlap.interval=c(1, 100) # ignore intersection at 0; observe overlap
# 4. get empirical AUC
emp.AUC(d0, d1)
# about .65 --> Poor
# .90-1 = excellent (A)
# .80-.90 = good (B)
# .70-.80 = fair (C)
# .60-.70 = poor (D)
# .50-.60 = fail (F)
# 5. Get uncertain interval
(res=nlopt.ui.general (Se = .55,
                       Sp = .55,
                       distribution = 'weibull',
                       parameters.d0 = parameters.d0,
                       parameters.d1 = parameters.d1,
                       overlap.interval,
                       intersection = NULL,
                       start = NULL,
                       print.level = 0))
abline(v=c(res$intersection, res$solution))
# 6. Assess improvement when diagnosing outside the uncertain interval
sel.d0 = d0 < res$solution[1] |  d0 > res$solution[2]
sel.d1 = d1 < res$solution[1] |  d1 > res$solution[2]
(percentage.selected.d0 = sum(sel.d0) / length(d0))
(percentage.selected.d1 = sum(sel.d1) / length(d1))
emp.AUC(d0[sel.d0], d1[sel.d1])
# AUC for selected scores outside the uncertain interval
emp.AUC(d0[!sel.d0], d1[!sel.d1])
# AUC for deselected scores; these scores are almost indistinguishable
}
