\name{CancerEngine-class}
\alias{CancerEngine-class}
\alias{CancerEngine}
\alias{ClinicalEngine}
\alias{rand,CancerEngine-method}
\alias{nrow,CancerEngine-method}
\alias{nComponents,CancerEngine-method}
\alias{summary,CancerEngine-method}
\docType{class}
\title{The "CancerEngine" Class}
\description{
  A CancerEngine combines a CancerModel (which defines the combinatorics
  of hits that produce cancer subtypes) with a pair of gene expression
  Engines that can be used to simulate microarray data depending on the
  presence or absence of different hits.
}
\usage{
CancerEngine(cm, base, altered)
\S4method{summary}{CancerEngine}(object, \dots)
\S4method{nrow}{CancerEngine}(x)
\S4method{nComponents}{CancerEngine}(object, \dots)
\S4method{rand}{CancerEngine}(object, n, \dots)
ClinicalEngine(nFeatures, nClusters, isWeighted,
               bHyp = NULL, survivalModel = NULL,
               SURV = function(n) rnorm(n, 0, 0.3),
               OUT = function(n) rnorm(n, 0, 0.3))
}
\arguments{
  \item{cm}{object of class \code{CancerModel}}
  \item{base}{character string giving the name of an \code{\link{Engine}}
    or \code{\link{EngineWithActivity}}, or an object of class \code{Engine}.
    Represents the expected gene expression in the absence of any hits.}
  \item{altered}{character string giving the name of an \code{\link{Engine}}
    or \code{\link{EngineWithActivity}}, or an object of class \code{Engine}.
    Represents the expected gene expression in the presence of hits.}
  \item{object}{object of class \code{CancerEngine}}
  \item{x}{object of class \code{CancerEngine}}
  \item{n}{numeric scalar representing number of samples to be simulated}
  \item{\dots}{extra arguments for generic routines}
  \item{nFeatures}{an integer; the number of simulated clinical features}
  \item{nClusters}{an integer; the number of simulated clusters or subtypes}
  \item{isWeighted}{a logical value; used to determine whether the prevalence
    of subtypes is equal (unweighted) or unequal (weighted).}
  \item{bHyp}{an object of the class \code{BlockHyperParameters}. If NULL,
    then it is constructed using the default clinical parameters. See the
    Values section below.}
  \item{survivalModel}{an object of the \code{\link{SurvivalModel}} class.
    If NULL, then it is constructed using the default parameters.}
  \item{SURV}{a function; the same as used in a \code{\link{CancerModel}}.}
  \item{OUT}{a function; the same as used in a \code{\link{CancerModel}}.}
}
\section{Objects from the Class}{
  Although objects of the class can be created by a direct call to
  \link[methods]{new}, the preferred method is to use the
  \code{CancerEngine} or \code{ClinicalEngine} generator functions.
}
\section{Slots}{
  \describe{
    \item{\code{cm}:}{A \code{\link{CancerModel}} object.}
    \item{\code{base}:}{Object of class "character" giving the name of
      an \code{\link{Engine}} or \code{\link{EngineWithActivity}}.
      Represents the expected gene expression in the absence of any hits.}
    \item{\code{altered}:}{Object of class "character" giving the name of
      an \code{\link{Engine}} or \code{\link{EngineWithActivity}}.
      Represents the expected gene expression in the presence of hits.}
    \item{\code{localenv}:}{Object of class \code{"environment"}; used
      in the internal implementation. }
  }
}
\section{Methods}{
  \describe{
    \item{rand}{\code{signature(object = "CancerEngine")}: returns a list
      containing two named components, \code{clinical} and \code{data}, The
      clinical element is a data frame generated from the underlying
      \code{CancerModel}, and the data element is a matrix generated by the
      gene expression engines, altered by the appropriate "hits" present in
      each simulated individual.}
    \item{summary}{\code{signature(object = "CancerEngine")}: print
      summaries of the underlying cancer models and gene expression engines
      in the cancer engine. }  }
}
\section{Values}{Both the \code{CancerEngine} and \code{ClinicalEngine}
  constructors return objects of the \code{CancerEngine} class. The only
  practical differences are the default parameters set by the constructors.
  The primary change is in the \code{\link{CancerModel}} slot. The
  \code{CancerEngine} expects you to construct your own \code{CancerModel}
  explicitly before producing a \code{CancerEngine}.

  By contrast, the \code{ClinicalEngine} expects to use many fewer
  features, so gives a simpler interface and uses its parameters to construct the
  \code{CancerModel} for you. The "HIT" function will use 2 hits per subtype
  when there are fewer than 15 features, 3 hits between 15 and 45 features,
  and the older default of 5 hits when there are more than 45 features. The
  total number of possible hits is set equal to the number of features (N) when
  there are fewer than 12 features, approximately N/3 up to 50 features,
  approximately N/5 up to 100, and is then locked at 20. If \code{isWeighted}
  is \code{TRUE}, subtype prevalences are chosen from a Dirichlet
  distribution. Otherwise, each subtype is equally likely.

  The \code{nrow} and \code{nComponents} methods both return
  non-negative integer values describing the number of rows (features)
  and the number of components of the underlying gene expression or
  clinical data Engines. The \code{rand} method returns a matrix with
  \code{n} columns of smiulated data.
}
\references{
  Zhang J, Coombes KR.\cr
  \emph{Sources of variation in false discovery rate estimation include
  sample size, correlation, and inherent differences between groups.}\cr
  BMC Bioinformatics. 2012; 13 Suppl 13:S1. 
}
\author{
  Kevin R. Coombes \email{krc@silicovore.com},
  Caitlin E. Coombes \email{caitlin.coombes@osumc.edu}
}
\seealso{
  \code{\linkS4class{CancerModel}}
}
\examples{
showClass("CancerEngine")
set.seed(391629)
## Set up survival outcome; baseline is exponential
sm <- SurvivalModel(baseHazard=1/5, accrual=5, followUp=1)
## Build a CancerModel with 6 subtypes
nBlocks <- 20    # number of possible hits
cm <- CancerModel(name="cansim",
                  nPossible=nBlocks,
                  nPattern=6,
                  OUT = function(n) rnorm(n, 0, 1), 
                  SURV= function(n) rnorm(n, 0, 1),
                  survivalModel=sm)
## Include 100 blocks/pathways that are not hit by cancer
nTotalBlocks <- nBlocks + 100
## Assign values to hyperparameters
## block size
blockSize <- round(rnorm(nTotalBlocks, 100, 30))
## log normal mean hypers
mu0    <- 6
sigma0 <- 1.5
## log normal sigma hypers
rate   <- 28.11
shape  <- 44.25
## block corr
p <- 0.6
w <- 5
## Set up the baseline Engine
rho <- rbeta(nTotalBlocks, p*w, (1-p)*w)
base <- lapply(1:nTotalBlocks,
               function(i) {
                 bs <- blockSize[i]
                 co <- matrix(rho[i], nrow=bs, ncol=bs)
                 diag(co) <- 1
                 mu <- rnorm(bs, mu0, sigma0)
                 sigma <- matrix(1/rgamma(bs, rate=rate, shape=shape), nrow=1)
                 covo <- co *(t(sigma) \%*\% sigma)
                 MVN(mu, covo)
               })
eng <- Engine(base)
## Alter the means if there is a hit
altered <- alterMean(eng, normalOffset, delta=0, sigma=1)
## Build the CancerEngine using character strings
object <- CancerEngine(cm, "eng", "altered")
## Or build it using the actual Engine components
ob <- CancerEngine(cm, eng, altered)
summary(object)
summary(ob)
## Simulate the data
dset <- rand(object, 20)
summary(dset$clinical)
summary(dset$data[, 1:3])
}
\keyword{classes}
\keyword{datagen}

