\name{EngineWithActivity-class}
\alias{EngineWithActivity-class}
\alias{EngineWithActivity}
\alias{rand,EngineWithActivity-method}
\alias{summary,EngineWithActivity-method}
\docType{class}
\title{The "EngineWithActivity" Class}
\description{
  The \code{EngineWithActivity} is used to set some components in the object
  of class \code{Engine} to be transcriptionally inactive and transform the
  expression data to appropriate logarithmic scale.
}
\usage{
EngineWithActivity(active, components, base=2)
\S4method{rand}{EngineWithActivity}(object, n, \dots)
\S4method{summary}{EngineWithActivity}(object, \dots)
}
\arguments{
  \item{active}{logical vector with length equal to number of components
    specifying whether each component should be transcriptionally active, or
    a numeric scalar specifying the probability for a component to be active}
  \item{components}{list where each element contains the parameters for the
    underlying distribution that the gene expression follows}
  \item{base}{numeric scalar specifying the logarithmic scale to which the
    data should be transformed}
  \item{object}{object of class \code{EngineWithActivity}}
  \item{n}{number of samples to be simulated}
  \item{\dots}{extra arguments for generic routines}
}
\details{
  An ENGINE WITH ACTIVITY allows for the possibility that some
  components (or genes) in an expression engine (or tissue) might be
  transcriptionally inactive.  Thus, the true biological signal
  S_gi should really be viewed as a mixture:
  
  \eqn{S_gi = z_g * delta_0 + (1 - z_g) * T_gi}

  where delta_0 = a point mass at zero;   
  T_gi = a random variable supported on the positive real line;   
  z_g ~ Binom(pi) defines the activity state (1 = on, 0 = off)

  The \code{rand} method for an EngineWithActivity is a little bit
  tricky, since we do two things at once. First, we use the
  \code{base} slot to exponentiate the random variables generated by
  the underlying Engine on the log scale. We treat \code{base = 0} as
  a special case, which means that we should continue to work on
  the scale of the Engine. Second, we mask any inactive component
  by replacing the generated values with \code{0}.

  Note that this is terribly inefficient if we only have a single
  homogeneous population, since we generate a certain amount of
  data only to throw it away.  The power comes when we allow
  cancer disregulation to turn a block on or off, when the
  underlying data reappears.
}
\section{Objects from the Class}{
  Although objects of the class can be created by a direct call to
  \link[methods]{new}, the preferred method is to use the
  \code{EngineWithActivity} generator function.
}
\section{Slots}{
  \describe{
    \item{\code{active}:}{logical vector specifying whether each component
      should be transcriptionally active or not}
    \item{\code{base}:}{numeric scalar specifying the logarithmic scale}
    \item{\code{components}:}{list specifying the parameters of the
      underlying distribution}
  }
}
\section{Extends}{
  Class \code{\linkS4class{Engine}}, directly.
}
\section{Methods}{
  \describe{
    \item{rand(object, n, \dots)}{Generates nrow(EngineWithActivity)*n matrix
      representing gene expressions of \code{n} samples, and the
      transcriptionally inactive components are masked by \code{0}.}
    \item{summary(object, \dots)}{Prints out the total number of
      components and the number of active components in the object
      of \code{EngineWithActivity}.}
  }
}
\value{
  The \code{EngineWithActivity} generator returns an object of class
  \code{EngineWithActivity}.

  The \code{rand} method returns \eqn{nrow(EngineWithActivity)*n} gene
  expression matrix with the inactive components being masked by \code{0}.

  The \code{summary} method prints out the total number of components and
  the number of active components in the object of \code{EngineWithActivity}.
}
\author{
  Kevin R. Coombes \email{krc@silicovore.com},
  Jiexin Zhang \email{jiexinzhang@mdanderson.org},
}
\examples{
showClass("EngineWithActivity")
nComponents <- 10
nGenes <- 100
active <- 0.7
comp <- list()
for (i in 1:nComponents) {
  comp[[i]] <- IndependentNormal(rnorm(nGenes/nComponents, 6, 1.5),
                                 1/rgamma(nGenes/nComponents, 44, 28))
}
myEngine <- EngineWithActivity(active, comp, 2)
summary(myEngine)
myData <- rand(myEngine, 5)
dim(myData)
}
\keyword{datagen}
\keyword{classes}

