% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/strucDivNest.R
\name{strucDivNest}
\alias{strucDivNest}
\title{Quantify Spatial Structural Diversity Across Scales in an Arbitrary Raster Layer}
\usage{
strucDivNest(
  x,
  wslI = NULL,
  wslO = NULL,
  dimB = FALSE,
  oLap = NULL,
  priorB = FALSE,
  domain = FALSE,
  dist = 1,
  angle = "all",
  rank = FALSE,
  fun,
  delta = 0,
  na.handling = na.pass,
  padValue = NA,
  aroundTheGlobe = FALSE,
  ncores = 1,
  verbose = TRUE,
  filename = "",
  ...
)
}
\arguments{
\item{x}{raster layer. Input raster layer for which 
horizontal structural diversity should be calculated.}

\item{wslI}{uneven integer. The window side length of the inner scale, 
\code{wslI} x \code{wslI} defines the size of the inner moving window.
The window must be smaller than the dimensions of the input raster and smaller than the outer scale. 
Default is NULL, in which case no prior information is used.}

\item{wslO}{uneven integer.  The window side length of the outer scale, 
\code{wslO} x \code{wslO} defines the size of the outer moving window.
The window must be smaller than the dimensions of the input raster and larger than the inner scale (i.e. \code{wslI}). 
Defaults to \code{NULL}, in which case no prior information is used.}

\item{dimB}{a vector of length 2 or logical. This defines the block size (number of rows, number of columns). 
The domain (i.e. the input raster) is divided into equal size, overlapping blocks. 
Each block provides prior information for the inner window, which moves inside each block. 
Structural diversity is quantified in each block.
Blocks are merged together in a spatially weighted manner, using linear weights.
Defaults to \code{FALSE}, in which case no blocks are used.}

\item{oLap}{integer. This defines the size of overlap between the blocks. The overlap must
be at least \code{wslI-1} or bigger. Blocks can overlap by a maximum of half the 
rows of blocks in row-direction, and by half the columns of blocks in column-direction.
If oLap is not specified, the minimum overlap is used.
Defaults to \code{NULL} in which case no blocks are used.}

\item{priorB}{logical. Should blocks be used for prior information? 
If \code{priorB = TRUE}, then the spatial structure in a block serves 
as prior information for the inner scale. 
If \code{priorB = FALSE}, then the blocks are only used to increase speed through 
parallelization, not for prior information.
Defaults to \code{FALSE}.}

\item{domain}{logical. Should the domain (i.e. the input raster) be used for prior information?
If  \code{domain = TRUE}, then it is used as prior for all inner moving windows.
Defaults to \code{FALSE}.}

\item{dist}{integer. The distance between two pixels that should be considered as a pair, 
defaults to \code{dist = 1} (direct neighbors).}

\item{angle}{string. The angle on which pixels should be considered as pairs. 
Takes 5 options: \code{"horizontal"}, \code{"vertical"}, \code{"diagonal45"}, \code{"diagonal135"}, \code{"all"}. 
The direction-invariant version is \code{"all"}, which considers all of the 4 angles. Defaults to \code{"all"}.}

\item{rank}{logical. Should pixel values be replaced with ranks in each GLCM? Defaults to \code{FALSE}.}

\item{fun}{function, the structural diversity metric. Takes one of the following: \code{entropy},
\code{entropyNorm}, \code{contrast}, \code{dissimilarity}, or \code{homogeneity}. 
Structural diversity entropy is \code{entropy} with different \code{delta} parameters. Shannon entropy is employed when \code{delta = 0}. 
Shannon entropy has a scale-dependent maximum when \code{\link{strucDiv}} is used, but this maximum may be violated in \code{\link{strucDivNest}}, 
when information from different scales is combined, depending on the posterior probabilities of pixel value co-occurrences.
Additionally, the value gradient is considered with \code{delta = 1} and \code{delta = 2}. 
The values of structural diversity entropy with \code{delta = 1} or \code{delta = 2} are not restricted and depend on the values of the input raster.
the metric \code{entropyNorm} is Shannon entropy normalized over maximum entropy, which depends on the size of the moving window when no scales are nested. 
When information from different scales is combined in \code{\link{strucDivNest}}, the metric \code{entropyNorm} may be larger than 1, 
depending on the posterior probabilities of pixel value co-occurrences.
The metrics \code{contrast} and \code{dissimilarity} consider the value gradient, their values are not restricted and depend on the values of the input raster.
The metric \code{homogeneity} quantifies the closeness of empirical probabilities to the diagonal and ranges between 0 and 1 when scales are not nested. 
When information from different scales is combined in \code{\link{strucDivNest}}, the metric \code{homogeneity} may be larger than 1, 
depending on the posterior probabilities of pixel value co-occurrences.}

\item{delta}{numeric, takes three options: \code{0}, \code{1}, or \code{2}. 
The \code{delta} parameter defines how the differences between pixel values within a pixel pair should be weighted.  
If \code{rank = TRUE}, delta defines how the differences between ranks should be weighted.  
Defaults to \code{0} (no weight). Set \code{delta = 1} for absolute weights, 
or \code{delta = 2} for square weights. 
The \code{delta} parameter can only be set when the metric \code{entropy} is used. 
the metric \code{dissimilarity} automatically employs \code{delta = 1}, and \code{contrast} employs \code{delta = 2}.}

\item{na.handling}{\code{na.omit} or \code{na.pass}. 
If \code{na.handling = na.omit}, NAs are ignored and structural diversity metrics are calculated with less values. 
If \code{na.handling = na.pass} and if there is at least one missing value inside the moving window,
an NA is assigned to the center pixel. Therefore, the diversity map will contain more 
NAs than the input raster layer.
Defaults to \code{na.pass}.}

\item{padValue}{numeric or \code{NA}. The value of the padded cells at the edges of the input raster. 
Defaults to \code{NA}.}

\item{aroundTheGlobe}{logical. If the input raster goes around the whole globe, 
set \code{aroundTheGlobe = TRUE}, and the input raster will be 'glued together' from both sides
to calculate structural diversity without edge effects.
Defaults to \code{FALSE}.}

\item{ncores}{integer. The number of cores the computation will be parallelized on.
Parallelization is only available when blocks are used. i.e. dimB must be specified. 
Parallelization can be used independent of whether blocks are used as priors or not.}

\item{verbose}{logical. If \code{verbose = TRUE}, a progress bar will be visible.}

\item{filename}{character. If the output raster should be written to a file, define file name (optional).}

\item{...}{possible further arguments.}
}
\value{
The output is a (spatial) structural diversity map, returned as a raster layer.
If the outer scale is a moving window or the domain, then the output raster has the same dimensions as the input raster.
If the outer scale is a block, then the output raster may be smaller than the input raster 
because if there are edges that do not fit inside the blocks, they are cut off.
When \code{na.handling = na.pass}, then the output map will have an NA-edge of 0.5*(\code{wslO}-1), 
and it will contain more missing values than the input raster.
The output represents spatial structural diversity quantified across different spatial scale, which are defined by the 
size of the inner and the outer scale.
}
\description{
This is a wrapper function that returns a 'spatial structural diversity map' 
as a raster layer. Spatial refers to horizontal, i.e. spatially explicit, and 
'spatial structural diversity' will hereafter be used synonymous to 'structural diversity'. 
Pixels are considered as pairs in user-specified distances and angles. 
Angles include horizontal and vertical direction, and the diagonals at 45° and 135°. 
The direction-invariant version considers all angles. 
Spatial structural diversity is quantified based on the probabilities that pixel values 
are arranged in the specified way (distance and angle). The \code{\link{strucDiv}} function employs empirical probabilities of pixel value co-occurrence. 
The \code{\link{strucDivNest}} function combines information from two different scales with an empirical Bayesian approach and a Beta-Binomial model.
Two scales are nested inside each other - a larger, outer scale and a smaller, inner scale. 
Three different nesting schemes are available, whereby the inner scale is always a moving window.
The outer scale can either be another mowing window, a block, or the domain (i.e. the input raster).
The outer scale is used as prior information for data on the inner scale, and structural diversity is quantified based on
posterior probabilities of pixel value co-occurrences. 
In the Beta-Binomial model both the prior and the posterior follow a beta distribution, and the likelihood follows a conditional 
binomial distribution.
Posterior probabilities are estimated with mean estimates.
The final map is called a '(spatial) structural diversity map' and is returned as a raster layer. 
The output map represents structural diversity, quantified across different spatial scales, which are defined
by the outer scale and the inner scale.
}
\details{
The memory requirement of the function is determined 
by \code{raster::canProcessInMemory()}. 
If the raster file cannot be processed in memory, its size needs to be reduced before \code{\link{strucDivNest}} can be used.
}
\examples{
\dontrun{
# Construct a small raster file containing realizations of normal random variables:
a <- raster::raster(matrix(rnorm(400), 20, 20))
raster::plot(a)
# Calculate structural diversity entropy with delta = 2, double moving window scheme
sde_1 <- strucDivNest(a, wslI = 3, wslO = 5, angle = "horizontal", fun = entropy, delta = 2)
raster::plot(sde_1)

# Calculate structural diversity entropy with delta = 1, block nesting scheme 
b <- raster::raster(matrix(rnorm(2500), 50, 50))
raster::plot(b)
sde_b <- strucDivNest(b, wslI = 3, dimB = c(10, 10), oLap = 4, priorB = TRUE, fun = entropy, 
    delta = 1)
raster::plot(sde_b)

# Calculate entropy on simulated random patch, domain nesting scheme 
patch <- raster::raster(patch)
entropy_patch <- strucDivNest(patch, wslI = 5, domain = TRUE, angle = "vertical", fun = entropy)
raster::plot(entropy_patch)
}
}
