% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/SGraph.R
\name{SGraph}
\alias{SGraph}
\title{Graphic Modeling Using LASSO-Type Sparse Learning Algorithm.}
\description{
This function builds a gaussian or binary graph based on the bootstrap ranking LASSO regression method.
}
\usage{
SGraph(x, graphtype = c("gaussian"), B = 5, Boots = 100, edgerule = c("AND"), kfold = 10, 
  plot = TRUE, seed = 0123)
}
\arguments{
\item{x}{the matrix of gaussian or binary variables}

\item{graphtype}{the type of gaussian or binary graph, with the default value gaussian}

\item{B}{the external loop for intersection operation, with the default value 5}

\item{Boots}{the internal loop for bootstrap sampling, with the default value 100}

\item{edgerule}{the AND or OR rule for estimating the edge set, with the default value AND}

\item{kfold}{the K-fold cross validation, with the default value 10}

\item{plot}{the indication of whether to show the graph or not, with the default value TRUE}

\item{seed}{the random seed for bootstrap sampling, with the default value 0123}
}
\value{
\item{Adjmatrix}{ The adjacency matrix of variables.}
}
\references{
[1] Guo, P., Zeng, F., Hu, X., Zhang, D., Zhu, S., Deng, Y., & Hao, Y. (2015). Improved Variable 
Selection Algorithm Using a LASSO-Type Penalty, with an Application to Assessing Hepatitis B Infection 
Relevant Factors in Community Residents. PLoS One, 27;10(7):e0134151.

[2] Strobl, R., Grill, E., Mansmann, U. (2012). Graphical modeling of binary data using the LASSO: a 
simulation study. BMC Medical Research Methodology, 12:16.

[3] Meinshausen, N., Buehlmann, P. (2006). High-dimensional graphs and variable selection with the Lasso.
Ann Stat, 34:1436-1462.
}
\examples{
# Example 1: Gaussian graph with AND-rule.
library(datasets)
head(iris)
X <- as.matrix(subset(iris,iris$Species=="setosa")[, -5])
# Build a sparse gaussian graph using the bootstrap ranking LASSO model.
# The parameters of B and Boots in the following example are set as small values to 
# reduce the running time, however the default values are proposed. 
SGraph.and <- SGraph(x=X, graphtype=c("gaussian"), B=2, Boots=2, edgerule=c("AND"), 
plot=FALSE)
# Give out the adjacency matrix of variables.
SGraph.and$Adjmatrix
# Plot the graph based on the adjacency matrix of variables using the qgraph package.
library(qgraph)
qgraph(SGraph.and$Adjmatrix, directed=FALSE, color="blue", negCol="red",  
edge.labels=TRUE, layout="circle")
# Example 2: Gene network estimation using the bootstrap ranking LASSO method.
library(SIS)
data(leukemia.train)
# Genes screened by the LASSO algorithm as candidates for graphical modeling.
set.seed(0123)
x <- as.matrix(leukemia.train[, -7130])
y <- as.factor(leukemia.train[, 7130])
cvfit <- cv.glmnet(x=x, y=y, type.measure="deviance", nfolds=3, family="binomial")
model.final <- cvfit$glmnet.fit
nzero <- as.matrix(coef(model.final, s=cvfit$lambda.min))
var_nz <- names(nzero[nzero[,1]!=0, ])[-1]
sub_data <- leukemia.train[, c(var_nz, "V7130")]
# Gene expression data subset from patients with acute myeloid leukemia.
subset_1 <- subset(sub_data, sub_data$V7130==1)
subset_1 <- as.matrix(subset_1[, -dim(subset_1)[2]])
# The parameters of B and Boots in the following example are set as small values to
# reduce the running time, however the default values are proposed.
SGraph.fit <- SGraph(subset_1, graphtype=c("gaussian"), B=2, Boots=2, edgerule=c("OR"))
}
