% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tritraining.R
\name{triTrainingG}
\alias{triTrainingG}
\title{Tri-training generic method}
\usage{
triTrainingG(y, gen.learner, gen.pred)
}
\arguments{
\item{y}{A vector with the labels of training instances. In this vector the
unlabeled instances are specified with the value \code{NA}.}

\item{gen.learner}{A function for training three supervised base classifiers.
This function needs two parameters, indexes and cls, where indexes indicates
the instances to use and cls specifies the classes of those instances.}

\item{gen.pred}{A function for predicting the probabilities per classes.
This function must be two parameters, model and indexes, where the model
is a classifier trained with \code{gen.learner} function and
indexes indicates the instances to predict.}
}
\value{
A list object of class "triTrainingG" containing:
\describe{
\item{model}{The final three base classifiers trained using the enlarged labeled set.}
\item{model.index}{List of three vectors of indexes related to the training instances
used per each classifier. These indexes are relative to the \code{y} argument.}
\item{instances.index}{The indexes of all training instances used to
train the three models. These indexes include the initial labeled instances
and the newly labeled instances. These indexes are relative to the \code{y} argument.}
\item{model.index.map}{List of three vectors with the same information in \code{model.index}
but the indexes are relative to \code{instances.index} vector.}
}
}
\description{
Tri-training is a semi-supervised learning algorithm with a co-training
style. This algorithm trains three classifiers with the same learning scheme from a
reduced set of labeled examples. For each iteration, an unlabeled example is labeled
for a classifier if the other two classifiers agree on the labeling proposed.
}
\details{
TriTrainingG can be helpful in those cases where the method selected as
base classifier needs a \code{learner} and \code{pred} functions with other
specifications. For more information about the general triTraining method,
please see the \code{\link{triTraining}} function. Essentially, the \code{triTraining}
function is a wrapper of the \code{triTrainingG} function.
}
\examples{
library(SSLR)
library(caret)

## Load Wine data set
data(wine)

cls <- which(colnames(wine) == "Wine")
x <- wine[, - cls] # instances without classes
y <- wine[, cls] # the classes
x <- scale(x) # scale the attributes

## Prepare data
set.seed(20)
# Use 50\% of instances for training
tra.idx <- sample(x = length(y), size = ceiling(length(y) * 0.5))
xtrain <- x[tra.idx,] # training instances
ytrain <- y[tra.idx] # classes of training instances
# Use 70\% of train instances as unlabeled set
tra.na.idx <- sample(x = length(tra.idx), size = ceiling(length(tra.idx) * 0.7))
ytrain[tra.na.idx] <- NA # remove class information of unlabeled instances

# Use the other 50\% of instances for inductive testing
tst.idx <- setdiff(1:length(y), tra.idx)
xitest <- x[tst.idx,] # testing instances
yitest <- y[tst.idx] # classes of testing instances

## Example: Training from a set of instances with 1-NN (knn3) as base classifier.
gen.learner <- function(indexes, cls)
  caret::knn3(x = xtrain[indexes,], y = cls, k = 1)
gen.pred <- function(model, indexes)
  predict(model, xtrain[indexes,])

# Train
set.seed(1)

trControl_triTraining1 <- list(gen.learner = gen.learner,
                                  gen.pred = gen.pred)
md1 <- train_generic(ytrain, method = "triTrainingG", trControl = trControl_triTraining1)



# Predict testing instances using the three classifiers
pred <- lapply(
  X = md1$model,
  FUN = function(m) predict(m, xitest, type = "class")
)
# Combine the predictions
cls1 <- triTrainingCombine(pred)
table(cls1, yitest)

confusionMatrix(cls1, yitest)$overall[1]


## Example: Training from a distance matrix with 1-NN (oneNN) as base classifier.
dtrain <- as.matrix(proxy::dist(x = xtrain, method = "euclidean", by_rows = TRUE))
gen.learner <- function(indexes, cls) {
  m <- SSLR::oneNN(y = cls)
  attr(m, "tra.idxs") <- indexes
  m
}

gen.pred <- function(model, indexes) {
  tra.idxs <- attr(model, "tra.idxs")
  d <- dtrain[indexes, tra.idxs]
  prob <- predict(model, d, distance.weighting = "none")
  prob
}

# Train
set.seed(1)

trControl_triTraining2 <- list(gen.learner = gen.learner,
                               gen.pred = gen.pred)
md2 <- train_generic(ytrain, method = "triTrainingG", trControl = trControl_triTraining2)

# Predict
ditest <- proxy::dist(x = xitest, y = xtrain[md2$instances.index,],
                      method = "euclidean", by_rows = TRUE)

# Predict testing instances using the three classifiers
pred <- mapply(
  FUN = function(m, indexes) {
    D <- ditest[, indexes]
    predict(m, D, type = "class")
  },
  m = md2$model,
  indexes = md2$model.index.map,
  SIMPLIFY = FALSE
)
# Combine the predictions
cls2 <- triTrainingCombine(pred)
table(cls2, yitest)

confusionMatrix(cls2, yitest)$overall[1]
}
