\name{sqrlConfig}
\alias{sqrlConfig}
\title{
Configuration Files
}
\description{
This material does not describe a function, but (rather) the file format used to
configure \pkg{SQRL} interfaces and \pkg{RODBC} communications.

Configuration files can be used to define new data sources, set blanket
parameter values for existing sources, or set individually named parameter
values.
}
\section{Example Configuration File}{
\preformatted{
# Parameters for RODBC::odbcConnect/RODBC::odbcDriverConnect.
dsn                 = NULL
uid                 = "Blake"
pwd                 = 'C:/some/other/file.txt'
connection          = "driver=<driver>;dbalias=alpha;uid=<uid>;pwd=<pwd>;"
case                = "nochange"
believeNRows        = TRUE
colQuote            = c("`", "'")
tabQuote            = '"'
interpretDot        = TRUE
DBMSencoding        = ""
rows_at_time        = 100
readOnlyOptimize    = FALSE

# Additional parameters for RODBC::sqlQuery.
errors              = TRUE
as.is               = TRUE
max                 = 0
buffsize            = 1000
nullstring          = NA_character_
na.strings          = c("NA", "-", "")
dec                 = "."
stringsAsFactors    = FALSE

# Parameters for SQRL.
aCollapse           = ','
autoclose           = TRUE
driver              = "{IBM DB2 ODBC DRIVER}"
interface           = "Z"
lCollapse           = "\\n"
library             = "my/library/file.sqrl"
ping                = "select 1 from dual"
verbose             = FALSE
visible             = TRUE
prompt              = "Z"
retry               = TRUE
scdo                = TRUE
wintitle            = "(Zen)"
}
}
\section{Commentary on Example File}{
This is a sample configuration file, exhibiting almost all parameters.
In general, a file need not include all of these (default values are in place).

Parameters may be defined as the path to some other file.
The \var{driver} and \var{dsn} parameters will take that path as their value.
For all other parameters, a value will be read from within the file.
Such files may contain only a single line with nothing but the value on it, or
they may adhere to the full (multiple line) \sQuote{parameter = value} format
(as above).

Configuration scripts are parsed and evaluated as \R, but any expressions in
which the final assignment is made via \code{=} (as opposed to \code{<-}) are
interpreted as requests to set \pkg{SQRL}/\pkg{RODBC} parameter values, rather
than \R environment variables.
}
\seealso{
\code{\link{sqrlParams}},
\code{\link{sqrlSource}}
}
\examples{
# Define a new source (not from file).
sqrlSource("Orac",
           "DBQ=StarOne;UID=Avon;PWD=<pwd>",
           "Driver={Oracle in OraDB18Home1}")

# Review its configuration (parameter values).
Orac("config")

# Create a file, containing only 'TRUE'.
file1 <- normalizePath(tempfile(), "/", FALSE)
writeLines("TRUE", file1)

# Create a file, containing named parameter values.
file2 <- normalizePath(tempfile(), "/", FALSE)
writeLines(c("dsn = 'Aristo'",
             "uid = 'Ensor'",
             "autoclose = TRUE",
             "as.is = FALSE"),
           file2)

# Create a configuration file, referencing the two above.
# Observe the use of temporary (non-parameter) variable 'x'.
file3 <- tempfile()
writeLines(c("aCollapse = ', '",
             paste0("verbose = \"", file1, "\""),
             "x <- 4",
             "max = sqrt(100 * x)",
             "as.is = TRUE",
             paste0("autoclose = '", file2, "'")),
           file3)
readLines(file3)

# Configure from the main (third) file.
Orac(config = file3)

# Alternative file-import forms.
Orac("config", file3)
Orac(paste("config", file3))

# Confirm imported values.
# Only 'autoclose' has been read from file2.
Orac("config")

# Import one parameter value from a file
# containing only a single unnamed value.
Orac(readOnlyOptimize = file1)
Orac("readOnlyOptimize")

# Import only a single specific parameter value
# from a file containing several named values.
Orac(uid = file2)
Orac("uid")

# Define and configure a new source from file.
sqrlSource("Caro", file2)
Caro("config")

# Configuration can also be performed from a named list,
# which could be generated by any R function or script.
Caro(config = list(autoclose = FALSE, max = 100))
Caro("config")[c("autoclose", "max")]

# Delete files.
unlink(c(file1, file2, file3))
}
\keyword{file}
