% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SAMC.R
\name{SAMC}
\alias{SAMC}
\title{Stochastic Approximation Monte Carlo (SAMC) Sampler}
\usage{
SAMC(nv, energy, options = list())
}
\arguments{
\item{nv}{number of variables.}

\item{energy}{an \code{R} function for negative log density.}

\item{options}{a list specifying parameters/options for SAMC algorithm. Below, vector(\eqn{k}) means a vector of length \eqn{k}, and matrix likewise.
\tabular{lll}{
PARAMETER        \tab SPECIFICATION \tab DESCRIPTION \cr
\code{domain}    \tab vector(\eqn{2}) or matrix(\eqn{(nv\times 2)}) \tab domain of sample space \cr
\code{partition} \tab vector(\eqn{m}) \tab energy partition \cr
\code{vecpi}     \tab vector(\eqn{m-1}) \tab desired sampling distribution \cr
\code{tau}       \tab positive number \tab temperature \cr
\code{niter}     \tab positive integer \tab number of iterations to be run \cr
\code{t0}        \tab \eqn{(0.5,1]}  \tab gain factor sequence value \cr
\code{xi}        \tab positive number \tab gain factor sequence exponent \cr
\code{stepsize}  \tab positive number \tab stepsize for random-walk sampler \cr
\code{trange}    \tab vector(\eqn{2}) or matrix(\eqn{m\times 2}) \tab domain of estimates for \eqn{\log(g_i /\pi_i)}
}}
}
\value{
a named list containing \describe{
\item{samples}{an \eqn{(niter\times nv)} samples generated.}
\item{frequency}{length-\eqn{m} vector of visiting frequency for energy partition.}
\item{theta}{length-\eqn{m} vector of estimates of \eqn{\log(g_i / \pi_i)}}
}
}
\description{
The function \code{SAMC} is a generic SAMC sampler for distributions on continuous domain. An \eqn{R} function for 
negative log density of your choice is required, as well as some parameters in SAMC framework.
}
\examples{
\donttest{
##### Two-Dimensional Multimodal sampling
## Step 1 : Define negative log-density function as an R function
func_r = function(x){
x1 = x[1]; x2 = x[2];
val1 = (-(x1*sin(20*x2)+x2*sin(20*x1))^2)*cosh(sin(10*x1)*x1);
val2 = (-(x1*cos(10*x2)-x2*sin(10*x1))^2)*cosh(cos(20*x2)*x2);
return(val1+val2);
}

## Step 2 : Prepare a setting option
myoption = list()
myoption$partition = c(-Inf,seq(from=-8,to=0,length.out=41))
myoption$tau       = 1.0
myoption$domain    = c(-1.1,1.1)
myoption$vecpi     = as.vector(rep(1/41,41))
myoption$niter     = 20000
myoption$stepsize  = c(0.25, 10)

## Step 3 : Run The Code
res = SAMC(2,func_r,options=myoption)

## Step 4 : Visualize
select = seq(from=101,to=myoption$niter,by=100) # 100 burn-in, 1/100 thinning 
par(mfrow=c(1,2))
plot(res$samples[select,1], res$samples[select,2],xlab='x',ylab='y',main='samples')
barplot(as.vector(res$frequency/sum(res$frequency)),
        main="visiting frequency by energy partition",
        names.arg=myoption$partition[-1], xlab="energy")
}

}
\references{
\insertRef{SAMC}{SAMCpack}
}
\author{
Kisung You
}
