#' Compute the robust effect size index estimate from Z statistic (alternative)
#'
#' This function computes the robust effect size index from Vandekar, Rao, & Blume (2020).
#' Vector arguments are accepted. If different length arguments are passed they are dealt with in the usual way of R.
#' @param z The Z statistic for the parameter of interest.
#' @param n Number of independent samples.
#' @details This function computes S, the RESI, from a Z statistic using the alternative
#' formula. There is another function, \code{\link{z2S}}, that is the default for
#' \code{\link{resi}}. This function's formula is derived by squaring the Z statistic
#' and using the \code{\link{chisq2S}} formula. This function may be appealing for its
#' intuitive relationship to the Chi-square statistic; the absolute value of RESI estimates
#' using this formula will be equal to a RESI estimate using a Chi-square statistic for
#' the same model. However, this estimator does have finite sample bias, which is an
#' important consideration for the coverage of the bootstrapping that \code{resi} uses.
#'
#' The formula for this conversion is:
#'
#' \eqn{ \sqrt(max(0, (Z^2 - 1)/n)) * sign(Z)}
#' @return Returns a scalar or vector argument of the the robust effect size index estimate.
#' @examples
#' # to obtain example z values, first fit a glm
#' mod = glm(charges ~ region * age + bmi + sex, data = RESI::insurance)
#' # run coeftest to get z values using a robust variance-covariance function
#' zs = lmtest::coeftest(mod, vcov. = sandwich::vcovHC)[,'z value']
#'
#' # get RESI estimates
#' z2S(zs, n = nrow(RESI::insurance))
#' @export
z2S_alt <- function(z, n){
  ifelse((z^2-1)<0, 0, sqrt((z^2-1)/n)*(sqrt(z^2)/z))
}
