\name{bugs}
\alias{bugs}
\title{Run WinBUGS and OpenBUGS from R}
\description{The \code{bugs} function takes data and starting values as input.
It automatically writes a WinBUGS script, calls the model,
and saves the simulations for easy access in R.}
\usage{
bugs(data, inits, parameters.to.save, model.file = "model.bug",
    n.chains = 3, n.iter = 2000, n.burnin = floor(n.iter/2),
    n.thin = max(1, floor(n.chains * (n.iter - n.burnin)/1000)),
    bin = (n.iter - n.burnin) / n.thin,
    debug = FALSE, DIC = TRUE, digits = 5, codaPkg = FALSE,
    bugs.directory = "c:/Program Files/WinBUGS14/",
    program = c("winbugs", "openbugs", "WinBugs", "OpenBugs"),
    working.directory = NULL, clearWD = FALSE, 
    useWINE = .Platform$OS.type != "windows", WINE = Sys.getenv("WINE"),
    newWINE = FALSE, WINEPATH = NULL)
}
\arguments{
  \item{data}{either a named list (names corresponding to variable names in the \code{model.file}) 
    of the data for the WinBUGS model, \emph{or}
    a vector or list of the names of the data objects used by the model.
    If \code{data = "data.txt"}, it is assumed that data have already been written to the working directory
    in a file called \file{data.txt}, e.g. by the function \code{\link{bugs.data}}.}
  \item{inits}{a list with \code{n.chains} elements; each element of the list is
    itself a list of starting values for the WinBUGS model, \emph{or}
    a function creating (possibly random) initial values.
    Alternatively, if \code{inits = NULL}, initial values are generated by WinBUGS}
  \item{parameters.to.save}{character vector of the names of the parameters to save which should be monitored}
  \item{model.file}{file containing the model written in WinBUGS code.
    The extension can be either \file{.bug} or \file{.txt}.
    
    If the extension is \file{.bug} and \code{program=="winbugs"}, 
    a copy of the file with extension \file{.txt} will be created
    in the \code{bugs()} call and removed afterwards.
    Note that similarly named \file{.txt} files will be overwritten.}
  \item{n.chains}{number of Markov chains (default: 3)}
  \item{n.iter}{number of total iterations per chain (including burn in; default: 2000)}
  \item{n.burnin}{length of burn in, i.e. number of iterations to discard at the beginning.
      Default is \code{n.iter/2}, that is, discarding the first half of the simulations.}
  \item{n.thin}{thinning rate.  Must be a positive integer.
   Set \code{n.thin} > 1 to save memory and computation time if \code{n.iter} is large.
   Default is \code{max(1, floor(n.chains * (n.iter-n.burnin) / 1000))}
   which will only thin if there are at least 2000 simulations.}
  \item{bin}{number of iterations between saving of results 
    (i.e. the coda files are saved after each \code{bin} iterations); 
    default is to save only at the end.}
  \item{debug}{if \code{FALSE} (default), WinBUGS is closed automatically
  when the script has finished running, otherwise WinBUGS remains open for further investigation}
  \item{DIC}{logical; if \code{TRUE} (default), compute deviance, pD, and DIC.
    This is done in WinBUGS directly using the rule \code{pD = Dbar - Dhat}.
    If there are less iterations than required for the adaptive phase, 
    the rule \code{pD = var(deviance) / 2} is used.}
  \item{digits}{number of significant digits used for WinBUGS input, see \code{\link{formatC}}}
  \item{codaPkg}{logical; if \code{FALSE} (default) a \code{bugs} object is returned,
    if \code{TRUE} file names of WinBUGS output are returned for easy access by the \pkg{coda} package
    through function \code{\link{read.bugs}}. (not used if \code{program = "openbugs"})}
  \item{bugs.directory}{directory that contains the WinBUGS executable}
  \item{program}{the program to use, either \code{winbugs}/\code{WinBugs} or \code{openbugs}/\code{OpenBugs},
    the latter makes use of function \code{\link{openbugs}} and requires the CRAN package \pkg{BRugs}.}
  \item{working.directory}{sets working directory during execution of this function;
    WinBUGS' in- and output will be stored in this directory;
    if \code{NULL}, the current working directory is chosen.}
  \item{clearWD}{logical; indicating whether the files \file{data.txt}, 
    \file{inits[1:n.chains].txt}, \file{log.odc}, \file{codaIndex.txt}, and
    \file{coda[1:nchains].txt} should be removed after WinBUGS has finished.
    If set to \code{TRUE}, this argument is only respected if \code{codaPkg = FALSE}.
  }
  \item{useWINE}{logical; attempt to use the WINE emulator to run WinBUGS, 
    defaults to \code{TRUE} on Windows, and \code{FALSE} otherwise.
    If WINE is used, the arguments \code{bugs.directory} and \code{working.directory} must be given in form of Linux paths
    rather than Windows paths (if not \code{NULL}).}
  \item{WINE}{character; name of WINE binary file}
  \item{newWINE}{Set this one to \code{TRUE} for new versions of WINE.}
  \item{WINEPATH}{Path the WINE, it is tried hard to get the information automatically if not given.}
}
\details{
To run:
\enumerate{
\item Write a WinBUGS model in a ASCII file.
\item Go into R.
\item Prepare the inputs to the \code{bugs} function and run it (see Example).
\item A WinBUGS window will pop up amd R will freeze up. The model
    will now run in WinBUGS. It might take awhile. You will see
    things happening in the Log window within WinBUGS. When WinBugs
    is done, its window will close and R will work again.
\item If an error message appears, re-run with \code{debug = TRUE}.
}}
\value{
 If \code{codaPkg = TRUE} the returned values are the names
 of coda output files written by WinBUGS containing
 the Markov Chain Monte Carlo output in the CODA format.
 This is useful for direct access with \code{\link{read.bugs}}.

 If \code{codaPkg = FALSE}, the following values are returned:
  \item{n.chains}{see Section \sQuote{Arguments}}
  \item{n.iter}{see Section \sQuote{Arguments}}
  \item{n.burnin}{see Section \sQuote{Arguments}}
  \item{n.thin}{see Section \sQuote{Arguments}}
  \item{n.keep}{number of iterations kept per chain (equal to \code{(n.iter-n.burnin) / n.thin})}
  \item{n.sims}{number of posterior simulations (equal to \code{n.chains * n.keep})}
  \item{sims.array}{3-way array of simulation output, with dimensions
    n.keep, n.chains, and length of combined parameter vector}
  \item{sims.list}{list of simulated parameters:\cr
    for each scalar parameter, a vector of length n.sims\cr
    for each vector parameter, a 2-way array of simulations,\cr
    for each matrix parameter, a 3-way array of simulations, etc.
     (for convenience, the \code{n.keep * n.chains} simulations in
     sims.matrix and sims.list (but NOT sims.array have been randomly permuted)}
  \item{sims.matrix}{matrix of simulation output, with \code{n.chains * n.keep} rows and
    one column for each element of each saved parameter
    (for convenience, the \code{n.keep * n.chains} simulations in
     sims.matrix and sims.list (but NOT sims.array have been randomly permuted)}
  \item{summary}{summary statistics and convergence information for each
    element of each saved parameter.}
  \item{mean}{a list of the estimated parameter means}
  \item{sd}{a list of the estimated parameter standard deviations}
  \item{median}{a list of the estimated parameter medians}
  \item{root.short}{names of argument \code{parameters.to.save} and \dQuote{deviance}}
  \item{long.short}{indexes; programming stuff}
  \item{dimension.short}{dimension of \code{indexes.short}}
  \item{indexes.short}{indexes of \code{root.short}}
  \item{last.values}{list of simulations from the most recent iteration; they
    can be used as starting points if you wish to run WinBUGS for further iterations}
  \item{pD}{an estimate of the effective number of parameters, for calculations see the section \dQuote{Arguments}.}
  \item{DIC}{\code{mean(deviance) + pD}}
}
\references{
    Gelman, A., Carlin, J.B., Stern, H.S., Rubin, D.B. (2003):
    \emph{Bayesian Data Analysis}, 2nd edition, CRC Press.

     Sturtz, S., Ligges, U., Gelman, A. (2005): 
     R2WinBUGS: A Package for Running WinBUGS from R.  
     \emph{Journal of Statistical Software} 12(3), 1-16.
}
    
\author{Andrew Gelman, \email{gelman@stat.columbia.edu}, \url{http:/www.stat.columbia.edu/~gelman/bugsR/};
    modifications and packaged by Sibylle Sturtz, \email{sturtz@statistik.uni-dortmund.de}, and Uwe Ligges.}
\seealso{\code{\link{print.bugs}}, \code{\link{plot.bugs}}, and the \pkg{coda} package}
\examples{
# An example model file is given in:
model.file <- system.file(package = "R2WinBUGS", "model", "schools.txt")
# Let's take a look:
file.show(model.file)

# Some example data (see ?schools for details):
data(schools)
schools

J <- nrow(schools)
y <- schools$estimate
sigma.y <- schools$sd
data <- list ("J", "y", "sigma.y")
inits <- function(){
    list(theta = rnorm(J, 0, 100), mu.theta = rnorm(1, 0, 100),
         sigma.theta = runif(1, 0, 100))
}
## or alternatively something like:
# inits <- list(
#   list(theta = rnorm(J, 0, 90), mu.theta = rnorm(1, 0, 90),
#        sigma.theta = runif(1, 0, 90)),
#   list(theta = rnorm(J, 0, 100), mu.theta = rnorm(1, 0, 100),
#        sigma.theta = runif(1, 0, 100))
#   list(theta = rnorm(J, 0, 110), mu.theta = rnorm(1, 0, 110),
#        sigma.theta = runif(1, 0, 110)))

parameters <- c("theta", "mu.theta", "sigma.theta")

\dontrun{
## You may need to edit "bugs.directory",
## also you need write access in the working directory:
schools.sim <- bugs(data, inits, parameters, model.file,
    n.chains = 3, n.iter = 5000,
    bugs.directory = "c:/Program Files/WinBUGS14/",
    working.directory = NULL, clearWD = TRUE)
print(schools.sim)
plot(schools.sim)
}
}
\keyword{interface}
\keyword{models}
