#' @title Print Methods for Various Objects
#'
#' @description
#' Print concise, user-friendly summaries of objects generated by the Qval package.
#' Supports objects of classes \code{\link[Qval]{CDM}}, \code{\link[Qval]{validation}}, \code{\link[Qval]{sim.data}},
#' as well as their corresponding summary objects.
#'
#' @details
#' The \code{print} methods provide an at-a-glance view of key information:
#' \describe{
#'   \item{print.CDM}{displays sample size, item and attribute counts, and package information.}
#'   \item{print.validation}{shows suggested modifications to the Q-matrix, marking changed entries with an asterisk.}
#'   \item{print.sim.data}{reports dimensions of simulated data and offers guidance on extraction.}
#'   \item{print.summary.CDM}{prints fitted model details and alpha-pattern distribution from a \code{summary.CDM} object.}
#'   \item{print.summary.validation}{prints suggested Q-matrix changes or a message if none are recommended.}
#'   \item{print.summary.sim.data}{prints attribute-pattern frequencies and proportions from \code{summary.sim.data}.}
#' }
#'
#' @param x An object of the appropriate class (e.g., \code{\link[Qval]{CDM}}, 
#'          \code{\link[Qval]{validation}}, \code{\link[Qval]{sim.data}}, or their summaries).
#' @param ... Currently unused.  Additional arguments are ignored.
#' 
#' @return Invisibly returns \code{x}.
#'
#' @examples
#' set.seed(123)
#' library(Qval)
#' 
#' \donttest{
#' ################################################################
#' # Example 1: print a CDM object                                #
#' ################################################################
#' Q <- sim.Q(3, 20)
#' IQ <- list(P0 = runif(20, 0.0, 0.2), P1 = runif(20, 0.8, 1.0))
#' data.obj <- sim.data(Q = Q, N = 500, IQ = IQ, 
#'                      model = "GDINA", distribute = "horder")
#' CDM.obj <- CDM(data.obj$dat, Q, model = "GDINA", 
#'                method = "EM", maxitr = 2000, verbose = 1)
#' print(CDM.obj)
#' 
#'
#' ################################################################
#' # Example 2: print a validation object                         #
#' ################################################################
#' set.seed(123)
#' MQ <- sim.MQ(Q, 0.1)
#' CDM.obj <- CDM(data.obj$dat, MQ)
#' validation.obj <- validation(data.obj$dat, MQ, CDM.obj, 
#'                              method = "GDI")
#' print(validation.obj)
#' 
#'
#' ################################################################
#' # Example 3: print a sim.data object                           #
#' ################################################################
#' set.seed(123)
#' Q2 <- sim.Q(3, 10)
#' data.obj2 <- sim.data(Q = Q2, N = 1000)
#' print(data.obj2)
#' 
#'
#' ################################################################
#' # Example 4: print summary objects                             #
#' ################################################################
#' summary.CDM.obj <- summary(CDM.obj)
#' print(summary.CDM.obj)
#' 
#' summary.val.obj <- summary(validation.obj)
#' print(summary.val.obj)
#' 
#' summary.sim.obj <- summary(data.obj2)
#' print(summary.sim.obj)
#' }
#'
#' @name print
NULL

#' @describeIn print Print method for CDM objects
#' @export
print.CDM <- function(x, ...) {
  printPackageInfo()
  out <- summary(x)
  cat("==============================================\n")
  cat(" Number of items       =", out$base[2], "\n",
      "Number of attributes  =", out$base[3], "\n",
      "Number of individuals =", out$base[1], "\n",
      "To extract components, use the method extract.\n")
  invisible(x)
}

#' @describeIn print Print method for validation objects
#' @export
print.validation <- function(x, ...) {
  printPackageInfo()
  cat("==============================================\n")
  Q.sug <- data.frame(extract.validation(x, "Q.sug"))
  Q.orig <- data.frame(extract.validation(x, "Q.orig"))
  if (any(Q.sug != Q.orig)) {
    cat("\nSuggested Q-matrix: \n\n")
    Q.sug[Q.sug != Q.orig] <- paste0(Q.sug[Q.sug != Q.orig], "*")
    print(Q.sug, right = FALSE)
    cat("Note: * denotes a modified element.\n")
  } else {
    cat("\nNo Q-matrix modifications are suggested.\n")
  }
  invisible(x)
}

#' @describeIn print Print method for sim.data objects
#' @export
print.sim.data <- function(x, ...) {
  printPackageInfo()
  cat("==============================================\n")
  cat(" Number of items       =", nrow(x$Q), "\n",
      "Number of attributes  =", ncol(x$Q), "\n",
      "Number of individuals =", nrow(x$dat), "\n",
      "To extract components, use the method extract.\n")
  invisible(x)
}

#' @describeIn print Print method for summary.CDM objects
#' @export
print.summary.CDM <- function(x, ...) {
  cat("==============================================\n")
  cat(" Number of items       =", x$base[2], "\n",
      "Number of attributes  =", x$base[3], "\n",
      "Number of individuals =", x$base[1], "\n")
  cat("\nModel Fit:\n")
  print(x$model.fit)
  cat("\nDistribution of Alpha Patterns:\n")
  print(x$patterns)
  invisible(x)
}

#' @describeIn print Print method for summary.validation objects
#' @export
print.summary.validation <- function(x, ...) {
  cat("==============================================\n")
  Q.sug <- x$Q.sug
  Q.orig <- x$Q.orig
  if (any(Q.sug != Q.orig)) {
    cat("\nSuggested Q-matrix: \n\n")
    Q.sug[Q.sug != Q.orig] <- paste0(Q.sug[Q.sug != Q.orig], "*")
    print(Q.sug, right = FALSE)
    cat("Note: * denotes a modified element.\n")
  } else {
    cat("\nNo Q-matrix modifications are suggested.\n")
  }
  invisible(x)
}

#' @describeIn print Print method for summary.sim.data objects
#' @export
print.summary.sim.data <- function(x, ...) {
  cat("==============================================\n")
  cat(" Number of items       =", x$base[2], "\n",
      "Number of attributes  =", x$base[3], "\n",
      "Number of individuals =", x$base[1], "\n")
  cat("\nDistribution of Alpha Patterns:\n")
  print(x$patterns)
  invisible(x)
}
