#' 
#' @title Extract Elements from Various Objects
#' 
#' @description A unified extractor for retrieving components from objects generated by the Qval package.
#' @details
#' This generic function and its methods allow users to extract internal elements from
#' objects of class \code{\link[Qval]{CDM}}, \code{\link[Qval]{validation}}, or \code{\link[Qval]{sim.data}} without directly
#' accessing object slots. Supported components vary by class:
#' 
#' \describe{
#'   \item{\emph{\code{\link[Qval]{CDM}}}}{\describe{
#'     \item{analysis.obj}{The underlying GDINA or BM fit object.}
#'     \item{alpha}{EAP-based attribute estimates for each individual.}
#'     \item{P.alpha.Xi}{Posterior probabilities of attribute patterns.}
#'     \item{alpha.P}{Marginal mastery probability matrix.}
#'     \item{P.alpha}{Prior attribute weights from final iteration.}
#'     \item{Deviance}{Model deviance (-2 * marginal log-likelihood).}
#'     \item{npar}{Number of estimated parameters.}
#'     \item{AIC}{Akaike Information Criterion.}
#'     \item{BIC}{Bayesian Information Criterion.}
#'     \item{call}{Original function call.}
#'   }}
#'   \item{\emph{\code{\link[Qval]{validation}}}}{\describe{
#'     \item{Q.orig}{Original Q-matrix.}
#'     \item{Q.sug}{Suggested Q-matrix.}
#'     \item{time.cost}{time cost for validation.}
#'     \item{process}{Iteration-by-item modification log.}
#'     \item{iter}{Total iterations performed.}
#'     \item{priority}{Attribute priority matrix (PAA method only).}
#'     \item{Hull.fit}{Data for Hull plot (Hull method only).}
#'     \item{call}{Original function call.}
#'   }}
#'   \item{\emph{\code{\link[Qval]{sim.data}}}}{\describe{
#'     \item{dat}{Simulated response matrix (N × I).}
#'     \item{Q}{Simulated Q-matrix.}
#'     \item{attribute}{True attribute pattern matrix (N × K).}
#'     \item{catprob.parm}{Category success probabilities list.}
#'     \item{delta.parm}{Delta parameter list.}
#'     \item{higher.order.parm }{Higher-order parameter list.}
#'     \item{mvnorm.parm}{Multivariate normal distribution parameters.}
#'     \item{LCprob.parm}{Item/category success probabilities matrix.}
#'     \item{call}{Original function call.}
#'   }}
#' }
#' @param object An object of class \code{\link[Qval]{CDM}}, \code{\link[Qval]{validation}}, or \code{\link[Qval]{sim.data}}.
#' @param what A character string naming the component to extract.
#' @param ... Additional arguments (unused).
#' @return The requested component, whose type depends on \code{what} and the class of \code{object}.
#' 
#' @examples
#' library(Qval)
#' set.seed(123)
#' 
#' \dontrun{
#' ################################################################
#' # Example 1: sim.data extraction                               #
#' ################################################################
#' Q <- sim.Q(3, 10)
#' data.obj <- sim.data(Q, N = 200)
#' extract(data.obj, "dat")
#' 
#' 
#' ################################################################
#' # Example 2: CDM extraction                                    #
#' ################################################################
#' CDM.obj <- CDM(data.obj$dat, Q)
#' extract(CDM.obj, "alpha")
#' extract(CDM.obj, "AIC")
#' 
#' 
#' ################################################################
#' # Example 3: validation extraction                             #
#' ################################################################
#' validation.obj <- validation(data.obj$dat, Q, CDM.obj)
#' extract(validation.obj, "Q.sug")
#' }
#' 
#' @aliases extract extract.CDM extract.validation extract.sim.data
#' @export
extract <- function(object, what, ...) {
  UseMethod("extract")
}

#' @describeIn extract Extract fields from a CDM object
#' @export
extract.CDM <- function(object, what, ...) {
  choices <- c("analysis.obj", "alpha", "P.alpha.Xi", "alpha.P", "P.alpha",
               "Deviance", "npar", "AIC", "BIC", "call")
  if (!what %in% choices) {
    stop(sprintf("'%s' is not a valid field for CDM objects. Choose from: %s",
                 what, paste(choices, collapse = ", ")), call. = FALSE)
  }
  switch(what,
         analysis.obj = object$analysis.obj,
         alpha        = object$alpha,
         P.alpha.Xi   = object$P.alpha.Xi,
         alpha.P      = object$alpha.P,
         P.alpha      = object$P.alpha,
         Deviance     = object$model.fit["Deviance"],
         npar         = object$model.fit["npar"],
         AIC          = object$model.fit["AIC"],
         BIC          = object$model.fit["BIC"],
         call         = object$call)
}

#' @describeIn extract Extract fields from a validation object
#' @export
extract.validation <- function(object, what, ...) {
  choices <- c("Q.orig", "Q.sug", "time.cost", "process", "iter",
               "priority", "Hull.fit", "call")
  if (!what %in% choices) {
    stop(sprintf("'%s' is not a valid field for validation objects. Choose from: %s",
                 what, paste(choices, collapse = ", ")), call. = FALSE)
  }
  switch(what,
         Q.orig   = object$Q.orig,
         Q.sug    = object$Q.sug,
         time.cost = object$time.cost,
         process  = object$process,
         iter     = object$iter,
         priority = object$priority,
         Hull.fit = object$Hull.fit,
         call     = object$call)
}

#' @describeIn extract Extract fields from a sim.data object
#' @export
extract.sim.data <- function(object, what, ...) {
  choices <- c("dat", "Q", "attribute", "catprob.parm", "delta.parm",
               "higher.order.parm", "mvnorm.parm", "LCprob.parm", "call")
  if (!what %in% choices) {
    stop(sprintf("'%s' is not a valid field for sim.data objects. Choose from: %s",
                 what, paste(choices, collapse = ", ")), call. = FALSE)
  }
  switch(what,
         dat                = object$dat,
         Q                  = object$Q,
         attribute          = object$attribute,
         catprob.parm       = object$catprob.parm,
         delta.parm         = object$delta.parm,
         higher.order.parm  = object$higher.order.parm,
         mvnorm.parm        = object$mvnorm.parm,
         LCprob.parm        = object$LCprob.parm,
         call               = object$call)
}