% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FRindex.R
\name{FRindex}
\alias{FRindex}
\title{Functional Regression Index Model}
\usage{
FRindex(formula, data, xarg = as.numeric(colnames(X)), sign_prob = 0.5, ...)
}
\arguments{
\item{formula}{\link[stats]{formula}.
Left-hand-side is the name of the response variable.
Right-hand-side is the name of the tabulated \link[base]{matrix} functional predictor}

\item{data}{\link[base]{data.frame}, including
a response variable and a \link[base]{matrix} of tabulated functional predictor.
Each row of such \link[base]{matrix} represents functional predictor values for each subject tabulated on a fine grid
and each column corresponds to the same point on the same fine grid common for all subjects
If the functional predictor is the \link[stats]{quantile} function,
then \code{data} is preferably the returned object of \link{clusterQp}.}

\item{xarg}{\link[base]{numeric} \link[base]{vector},
\eqn{x}-domain to be passed to \link{FR_gam}
corresponding to the common grid on which the functional predictor is tabulated}

\item{sign_prob}{\link[base]{numeric} scalar,
the argument for selecting the nearest-even quantile in \code{xarg},
which is used to define the \link[base]{sign} of the weight function \code{@FRwt} for \code{@FRi}.
Default is \code{.5} corresponding to the nearest-even \link[stats]{median}}

\item{...}{additional parameters of \link{FR_gam}}
}
\value{
\link{FRindex} returns an \linkS4class{FRindex} object.
}
\description{
Fits functional regression model and computes functional indices based on functional predictors in the fitted model
}
\details{
Functional regression index for each subject is defined as a functional predictor in the functional regression model
(integral of subject-specific functional predictor multiplied by the weight function common for all subjects).
The weight function \code{@FRwt} is either equal or negated estimated functional coefficient in the functional regression model.
The \link[base]{sign} of the weight function \code{@FRwt} is selected so that the resulting \code{@FRi}
are positively associated with the values of functional predictor when \code{xarg} is equal to \code{sign_prob}.
}
\examples{

library(survival)

pt = unique(Ki67$PATIENT_ID)
length(pt) # 622
# set.seed if needed
train_pt = sample(pt, size = 500L)
Ki67q = clusterQp(data = Ki67, exclude = c('tissueID','inner_x','inner_y'), contX = 'Marker')
train_q = subset(Ki67q, PATIENT_ID \%in\% train_pt)
test_q = subset(Ki67q, !(PATIENT_ID \%in\% train_pt))
train_q$Marker = log1p(train_q$Marker)
test_q$Marker = log1p(test_q$Marker)

FRi = FRindex(Surv(RECFREESURV_MO, RECURRENCE) ~ Marker, data = train_q)
FRi@FRi # functional regression index
FRi@FRwt # functional regression weights

(FRi_test = predict.FRindex(FRi, newdata = test_q$Marker))

FRi_train = predict.FRindex(FRi)
stopifnot(identical(FRi@FRi, c(FRi_train)), 
  identical(FRi@FRwt, attr(FRi_train, 'FRwt')))

# set.seed if needed
Ki67bbc_v2 = BBC_dichotom(Surv(RECFREESURV_MO, RECURRENCE) ~ NodeSt + Tstage, 
  data = data.frame(train_q, FRi_std = std_IQR(FRi_train)), 
  contX = 'FRi_std', R = 100L)
summary(Ki67bbc_v2)

}
