#' @title Quantum Walk
#' @author Pan Binghuang
#' @description Generate the modes, the probabilities that the walker being found at vertices.
#' An adjacency matrix is need for the process.
#' @usage qwdap.qwalk(edges, startindex, lens, scals, getfloat)
#'
#' @param edges your N*N adjacency matrix saved as list.
#' @param startindex the initial position of the quantum walker.
#' @param lens the number of records required in a round of sampling by a scaling factor.
#' Set the length of the series according to requirements.
#' @param scals the scaling factors used.
#' @param getfloat Whether to return floating point data.
#'
#' @return a object of class 'CTQW', the quantum walk results and some parameters.
#' @details 'qwdap.qwalk()' is used to generated modes for time series analysis, the result is a object
#' of class 'CTQW', the modes are saved in the object as a 3-dim array, and the parameters are also
#' store in the object.
#' The continuous time quantum walk is a continuous process, the modes are generated with a series
#' of times, the parameter 'scals' can be understood as the tolerance of the arithmetic time series.
#' Multiply tolerances can be passed in to obtain modes on different time scales through parameter
#' 'scals'.
#' The probability of the series with the probabilities that the walker being found at the vertices, 
#' and the length depends on parameter 'lens'.
#' The data generated by this function is not recorded from the initial state. 
#' The shortest distance between all vertices and the initial position of the quantum walker is
#' obtained by the Dijkstra algorithm.The probabilities corresponding to each vertex are
#' recorded starting from the vertex furthest in the shortest distance is not 0.
#' The function is single thread.
#' 
#' @import progress
#' @export
#'
#' @examples
#' edges <- matrix(c(0,1,0,0,0,0,0,
#'                   1,0,1,0,0,0,0,
#'                   0,1,0,1,0,0,0,
#'                   0,0,1,0,1,0,0,
#'                   0,0,0,1,0,1,0,
#'                   0,0,0,0,1,0,1,
#'                   0,0,0,0,0,1,0),
#'                 nrow = 7)
#' res.qwalk <- qwdap.qwalk(edges,1,100,scals=seq(from=0.01, by=0.01, length.out=5))
#' 
qwdap.qwalk<-function(edges, startindex = 1, lens=100, scals=c(0.01), getfloat=FALSE){
  if(!is.matrix(edges)){
    print("The parameter 'edges' is not a matrix.")
    return()
  }
  if(!is.vector(scals) || !is.numeric(scals)){
    print("The parameter 'scals' is not a numeric vector.")
    return()
  }
  edges = as.data.frame(edges)
  if(length(edges) < 4){
    print("The minimum nodes is 4.")
    return()
  }
  qwalk.startIndex <- startindex

  qwalk.nodes <- NA
  for(i in c(1:length(edges))){
    qwalk.nodes[[i]]<-i
  }

  qtools.tolerance <- 0.5e-15
  # qwalk.eigenvalues = NULL
  # qwalk.eigenprojectors = NULL
  
  dijkstra <- function(mgraph, v0){
    if(v0 < 1 || v0 > length(mgraph)){
      print("v0 is an error value.")
      return(NA)
    }
    for(i in c(1:nrow(mgraph))){
      for(j in c(1:ncol(mgraph))){
        if(i==j){
          mgraph[i,j] = 0
          next
        }
        if(mgraph[i,j] == 0){
          mgraph[i,j] = Inf
        }
      }
    }
    N = length(mgraph)
    final = p = d = as.data.frame(matrix(rep(0,length(mgraph)),nrow = 1))
    colnames(p) = colnames(d) = colnames(mgraph)
    v = w = k = min1 = NA
    for(v in c(1:N)){
      final[[v]] = 0
      d[[v]] = mgraph[v0,v]
      if(mgraph[v0,v] < Inf){
        p[[v]] = v0
      }else{
        p[[v]] = -1
      }
    }
    p[[v0]] = -1
    final[[v0]] = 1
    for(v in c(2:N)){
      if(v < 2) break
      min1 = Inf
      for(w in c(1:N)){
        if(!final[[w]] && d[[w]] < min1){
          k = w
          min1 = d[[w]]
        }
      }
      final[[k]] = 1
      for(w in c(1:N)){
        if(!final[[w]] && ((min1 + mgraph[k,w]) < d[[w]])){
          d[[w]] = min1 + mgraph[k,w]
          p[[w]] = k
        }
      }
    }
    return(list("dis" = d,"path" = p))
  }
  
  dij_edges = dijkstra(edges,startindex)
  Tag_node = which(dij_edges$dis==max(dij_edges$dis),arr.ind=TRUE)[2]

  numeric.rep = function (s, v, k=1) {

    n <- s[k]

    ret <- list()
    i<-0
    if (k == length(s)) {
      for(i in c(1:n)){
        ret[[i]] <- v
      }
      return (ret);
    }
    for (i in c(n:1)) { ret[[i]] <- numeric.rep(s, v, k + 1); }
    return (ret);
  }
  
  qmanip.getEdges = function(id1,id2){
    return (edges[[id1]][[id2]])
  }
  
  norVecC = function(N,Z,wi){
    scale1 = ssq = absxi = dummy1 = norm1 = NULL
    
    for(j in c(1:N)){
      scale1 = 0.0
      ssq = 1.0
      for (i in c(1:N)){
        if(Z[[i]][[j]] != 0){
          absxi = abs(Z[[i]][[j]])
          dummy1 = scale1 / absxi
          if(scale1 < absxi){
            ssq = 1.0 + ssq * dummy1 * dummy1
            scale1 = absxi
          }else{
            ssq = ssq + 1.0 / dummy1 / dummy1
          }
        } # End if(Z[[i]][[j]]!=0)
      } # End for i
      
      if(wi[[j]] != 0){
        for(i in c(1:N)){
          if( Z[[i]][[j + 1]] != 0 ){
            absxi = abs(Z[[i]][[j + 1]])
            dummy1 = scale1 / absxi
            if(scale1 < absxi){
              ssq = 1.0 + ssq * dummy1 * dummy1
              scale1 = absxi
            }else{
              ssq = ssq + 1.0 / dummy1 / dummy1
            }
          } # End if(Z[i][j+1]!=0)
        } # End for i
      } # End if(wi[[j]] != 0)
      
      norm1 = scale1 * sqrt(ssq)
      
      for(i in c(1:N)){
        Z[[i]][[j]] = Z[[i]][[j]] / norm1
      }
      if(wi[[j]] != 0){
        j = j + 1
        for(i in c(1:N)){
          Z[[i]][[j]] = Z[[i]][[j]] / norm1
        }
      } # End if(wi[[j]]!=0)
    } # End for j
    return(Z)
  } # End norVecC
  
  cdivA = function(ar, ai, br, bi, A, in1, in2, in3){
    s = ars = ais = brs = bis = NULL
    s = abs(br) + abs(bi)
    ars = ar / s
    ais = ai / s
    brs = br / s
    bis = bi / s
    s = brs * brs + bis * bis
    A[[in1]][[in2]] = (ars * brs + ais * bis) / s
    A[[in1]][[in2]] = (-(ars * bis) + ais * brs) / s
    return(A)
  } # End cdivA
  
  hqr2 = function(N, A, B, low1, igh, wi, wr, oPar.outEr){
    norm1 = 0.0
    t = 0.0
    p = q = ra = s = sa = tst1 = tst2 = vi = vr = w = x = y = zz = NULL
    k = 1
    l = m = mp2 = NULL
    en = igh
    incrFlag = 1
    itn = 30 * N
    its = enm2 = na = notlas = NULL
    
    for(i in c(1:N)){
      for(j in c(k:N)){
        norm1 = norm1 + abs(A[[i]][[j]])
      }
      k = i
      if((i < low1)||(i > igh)){
        wi[[i]] = 0.0
        wr[[i]] = A[[i]][[i]]
      } # End if((i < low1)||(i > igh))
    } # End for i
    while (en >= low1) {
      if(incrFlag){
        its = 0
        na = en - 1
        enm2 = na -1
      }else{
        incrFlag = 1
      }
      
      for(i in c(low1:en)){
        l = en + low1 - i
        if(l == low1){
          break
        }
        s = abs(A[[l-1]][[l-1]]) + abs(A[[l]][[l]])
        if(s == 0.0){
          s = norm1
        }
        tst1 = s
        tst2 = tst1 + abs(A[[l]][[l - 1]])
        if(tst2 == tst1){
          break
        }
      } # End for i
      x = A[[en]][[en]]
      
      if(l == en){ # One root found
        wr[[en]] = A[[en]][[en]] = x + t
        wi[[en]] = 0.0
        en = en - 1
        next
      } # End if(l == en)
      y = A[[na]][[na]]
      w = A[[en]][[na]] * A[[na]][[en]]
      
      if(l == na){ # Tow root found
        p = (-x + y) / 2
        q = p * p + w
        zz = sqrt(abs(q))
        x = A[[en]][[en]] = x + t
        A[[na]][[na]] = y + t
        if(q >= 0.0){
          zz = (if(p < 0.0) (-zz+p) else (p + zz))
          wr[[en]] = wr[[na]] = x + zz
          if(zz != 0.0){
            wr[[en]] = -(w / zz) + x
          }
          wi[[en]] = wi[[na]] = 0.0
          x = A[[en]][[na]]
          s = abs(x) + abs(zz)
          p = x / s
          q = zz / s
          r = sqrt(p * p + q * q)
          p = p / r
          q = q / r
          for(j in c(na:N)){
            zz = A[[na]][[j]]
            A[[na]][[j]] = q * zz + p * A[[en]][[j]]
            A[[en]][[j]] = -(p * zz) + q * A[[en]][[j]]
          } # End for j
          for(j in c(1:en)){
            zz = A[[j]][[na]]
            A[[j]][[na]] = q*zz+p*A[[j]][[en]]
            A[[j]][[en]] = -(p*zz)+q*A[[j]][[en]]
          } # End for j
          for (j in c(low1:igh)){
            zz = B[[j]][[na]]
            B[[j]][[na]] = q * zz + p * B[[j]][[en]]
            B[[j]][[en]] = -(p * zz) + q * B[[j]][[en]]
          } # End for j
        }else{ # q < 0.0
          wr[[en]] = wr[[na]] = x + p
          wi[[na]] = zz
          wi[[en]] = -zz
        } # End else
        en = en - 2
        next
      } # End if(l == na)
      
      if(itn == 0){
        oPar.outEr = en + 1
        return(list(A,B,wi,wr,oPar.outEr))
      } # End if(ith == 0)
      if ((its == 10) || (its == 20)){
        t = t + x
        for (i in c(low1:en)){
          A[[i]][[i]] = A[[i]][[i]] - x
        }
        s = abs(A[[en]][[na]]) + abs(A[[na]][[enm2]])
        y = x = 0.75 * s
        w = -0.4375 * s * s
      } # End if((its == 10) || (its == 20))
      its = its + 1
      itn = itn - 1
      
      for(m in c(enm2:l)){
        zz = A[[m]][[m]]
        r = -zz + x
        s = -zz + y
        p = (-w + r * s) / A[[m + 1]][[m]] + A[[m]][[m + 1]]
        q = -(zz + r + s) + A[[m + 1]][[m + 1]]
        r = A[[m + 2]][[m + 1]]
        s = abs(p) + abs(q) + abs(r)
        p = p / s
        q = q / s
        r = r / s
        if (m == l){
          break
        }
        tst1 = abs(A[[m - 1]][[m - 1]]) + abs(zz) + abs(A[[m + 1]][[m + 1]])
        tst1 = tst1 * abs(p)
        tst2 = tst1 + abs(A[[m]][[m - 1]]) * (abs(q) + abs(r))
        if(tst1 == tst2){
          break
        }
      } # End for m
      mp2 = m + 2
      for(i in c(mp2:en)){
        A[[i]][[i - 2]] = 0.0
        if(i == mp2){
          next
        }
        A[[i]][[i - 3]] = 0.0
      } # End for i
      
      for(i in c(m : na)){
        notlas = (if(i != na) 1 else 0)
        if(i != m){
          p = A[[i]][[i - 1]]
          q = A[[i + 1]][[i - 1]]
          r = (if(notlas) A[[i+2]][[i - 1]] else 0.0)
          x = abs(p) + abs(q) + abs(r)
          if(x == 0.0){
            next
          }
          p = p / x
          q = q / x
          r = r / x
        } # End if(i != m)
        
        s = sqrt(p * p + q * q + r * r)
        if(p < 0.0){
          s = -s
        }
        if(i != m){
          A[[i]][[i - 1]] = -(s * x)
        } else{
          if(l != m){
            A[[i]][[i - 1]] = -A[[i]][[i - 1]]
          }
        }
        
        p = p + s
        x = p / s
        y = q / s
        zz = r / s
        q = q / p
        r = r / p
        k = if ((i + 3) < en) (i+3) else en
        if (notlas){
          for(j in c(i : N)){
            p = A[[i]][[j]] + q * A[[i + 1]][[j]] + r * A[[i + 2]][[j]]
            A[[i]][[j]] = A[[i]][[j]] - (p * x)
            A[[i + 1]][[j]] = A[[i + 1]][[j]] - (p * y)
            A[[i + 2]][[j]] = A[[i + 2]][[j]] - (p * zz)
          } # End for j
          
          for(j in c(1:k)){
            p = x * A[[j]][[i]] + y * A[[j]][[i + 1]] + zz * A[[j]][[i + 2]]
            A[[j]][[i]] = A[[j]][[i]] - p
            A[[j]][[i + 1]] = A[[j]][[i + 1]] - (p * q)
            A[[j]][[i + 2]] = A[[j]][[i + 2]] - (p * r)
          } # End for j
          
          for(j in c(low1:igh)){
            p = x * B[[j]][[i]] + y * B[[j]][[i + 1]] + zz * B[[j]][[i + 2]]
            B[[j]][[i]] = B[[j]][[i]] - p
            B[[j]][[i + 1]] = B[[j]][[i + 1]] - p * q
            B[[j]][[i + 2]] = B[[j]][[i + 2]] - p * r
          } # End for j
        }else{
          for(j in c(i:N)){
            p = A[[i]][[j]] + q * A[[i + 1 ]][[j]]
            A[[i]][[j]] = A[[i]][[j]] - p * x
            A[[i + 1]][[j]] = A[[i + 1]][[j]] - p * y
          } # End for j
          for(j in c(1:k)){
            p = x * A[[j]][[i]] + y * A[[j]][[i + 1]]
            A[[j]][[i]] = A[[j]][[i]] - p
            A[[j]][[i + 1]] = A[[j]][[i + 1]] - p * q
          } # End for j
          for(j in c(low1:igh)){
            p = x * B[[j]][[i]] + y * B[[j]][[i + 1]]
            B[[j]][[i]] = B[[j]][[i]] - p
            B[[j]][[i + 1]] = B[[j]][[i + 1]] - p * q
          } # End for j
        } # End else for notlas
      } # End for i
      incrFlag = 0
    } # End while (en >= low1)
    if(norm1 == 0.0){
      return(list(A,B,wi,wr,oPar.outEr))
    }
    
    for(en in c(N:1)){
      p = wr[[en]]
      q = wi[[en]]
      na = en - 1
      if(q > 0.0){
        next
      }
      
      if(q == 0.0){ # Real vector
        m = en
        A[[en]][[en]] = 1.0
        
        for(j in c(na:1)){
          if(na<1) break
          w = -p + A[[j]][[j]]
          r = 0.0
          for(ii in c(m : en)){
            r = r + A[[j]][[ii]] * A[[ii]][[en]]
          }
          if(wi[[j]] < 0.0){
            zz = w
            s = r
          } else{ # wi[[j]] >= 0.0
            m = j
            if(wi[[j]] == 0.0){
              t = w
              if(t == 0.0){
                t = tst1 = norm1
                repeat{
                  t = t * 0.01
                  tst2 = norm1 + t
                  if(tst2 <= tst1){
                    break
                  }
                }
              } # End if t == 0.0
              A[[j]][[en]] = -(r / t)
            }else{ # else wi[j] > 0.0
              x = A[[j]][[j + 1]]
              y = A[[j + 1]][[j]]
              q = (-p + wr[[j]])*(-p + wr[[j]]) + wi[[j]]*wi[[j]]
              A[[j]][[en]] = t = (-(zz * r) + x * s) / q
              A[[j + 1]][[en]] = if(abs(x) > abs(zz)) -(r + w * t) / x else -(s + y * t) / zz
            } # End else wi[j] > 0.0
            
            t = abs(A[[j]][[en]])
            if(t == 0.0){
              next # go up to top of for j loop
            }
            tst1 = t
            tst2 = tst1 + 1.0 / tst1
            if(tst2 > tst1){
              next # go up to top of for j loop
            }
            for(ii in c(j : en)){
              A[[ii]][[en]] = A[[ii]][[en]] / t
            }
          } # End else wi[j] >= 0.0
        } # End for j
      }else{ # else q < 0.0
        m = na
        if(abs(A[[en]][[na]]) <= abs(A[[na]][[en]])){
          A = cdivA(0.0, -A[[na]][[en]], -p + A[[na]][[na]], q, A, na, na, en)
        }else{
          A[[na]][[na]] = q / A[[na]][[na]]
          A[[na]][[en]] = -(-p + A[[en]][[en]]) / A[[en]][[na]]
        } # End else abs(A[[en]][[na]]) > abs(A[[na]][[en]])
        A[[en]][[na]] = 0.0
        A[[en]][[en]] = 1.0
        
        for(j in c((na -1):1)){
          w = -p + A[[j]][[j]]
          sa = ra = 0.0
          
          for(ii in c(m:en)){
            ra = ra + A[[j]][[ii]] * A[[ii]][[na]]
            sa = sa + A[[j]][[ii]] * A[[ii]][[en]]
          } # End for ii
          if(wi[[j]] < 0.0){
            zz = w
            r = ra
            s = sa
            next
          } # End if(wi[[j]] < 0.0)
          # else wi[[j]] >= 0.0
          m = j
          if(wi[[j]] == 0.0){
            A = cdivA(-ra, -sa, w, q, A, j, na, en)
          } else{ # else wi[[j]] > 0.0
            x = A[[j]][[j + 1]]
            y = A[[j + 1]][[j]]
            vr = -(q * q) + (-p + wr[[j]]) * (-p + wr[[j]]) + wi[[j]] * wi[[j]]
            vi = (-p + wr[[j]]) * 2.0 * q
            if((vr == 0.0) && (vi == 0.0)){
              tst1 = norm1 * (abs(w) + abs(q) + abs(x) + abs(y) + abs(zz))
              vr = tst1
              repeat{
                vr = vr * 0.01
                tst2 = tst1 + vr
                if(tst2 <= tst1){
                  break
                }
              }
            } # End if vr and vi == 0.0
            A = cdivA(-(zz * ra) + x * r + q * sa, -(zz * sa + q * ra) + x * s, vr, vi, A, j, na, en)
            if(abs(x) > (abs(zz)+abs(q))){
              A[[j + 1]][[na]] = (-(ra + w * A[[j]][[na]]) + q * A[[j]][[en]]) / x
              A[[j + 1]][[en]] = -(sa + w * A[[j]][[en]] + q * A[[j]][[na]]) / x
            }else{
              A = cdivA(-(r + y * A[[j]][[na]]), -(s + y * A[[j]][[en]]), zz, q, A, j + 1, na, en)
            }
          } # End else wi[j] > 0.0
          t = max(abs(A[[j]][[na]]), abs(A[[j]][[en]]))
          if(t == 0.0){
            next # go to top of for j loop
          }
          tst1 = t
          tst2 = tst1 + 1.0 / tst1
          if(tst2 > tst1){
            next # go to top of for j loop
          }
          for(ii in c(j:en)){
            A[[ii]][[na]] = A[[ii]][[na]] / t
            A[[ii]][[en]] = A[[ii]][[en]] / t
          } # End for ii loop
        } # End for j
      } # End else q < 0.0
    } # End for en
    
    for(i in c(1:N)){
      if((i < low1)||(i > igh)){
        for(j in c(i:N)){
          B[[i]][[j]] = A[[i]][[j]]
        }
      } # End if i
    } # End for i
    
    for(i in c(N:low1)){
      m = if(i < igh) i else igh
      for(ii in c(low1:igh)){
        zz = 0.0
        for(jj in c(low1:m)){
          zz = zz + B[[ii]][[jj]] * A[[jj]][[i]]
        }
        B[[ii]][[i]] = zz
      } # End for ii
    } # End for i
    
    return(list(A,B,wi,wr,oPar.outEr))
  } # End of function hqr2
  
  calcEigSysReal = function(N,A){
    B = numeric.rep(c(N,2*N),0)
    wr = numeric.rep(N,0)
    wi = numeric.rep(N,0)
    oPar.outEr = 0
    scale1 = list()
    trace1 = list()
    radix = 2
    c = f = g = r = s = NULL
    b2 = radix * radix
    ierr = -1
    igh = low1 = noconv = NULL
    k = 1
    l = N
    
    noconv = l
    while(noconv >= 1){
      r = 0
      for(j in c(1:l)){
        if(j == noconv) next
        if(A[[noconv]][[j]] != 0.0){
          r = 1
          break
        }
      }# End for j
      if(r == 0){
        scale1[l] = noconv
        if (noconv != l){
          for(i in c(1:l)){
            f = A[[i]][[noconv]]
            A[[i]][[noconv]] = A[[i]][[l]]
            A[[i]][[l]] = f
          } # End for i
          for(i in c(1:N)){
            f = A[[noconv]][[i]]
            A[[noconv]][[i]] = A[[l]][[i]]
            A[[l]][[i]] = f
          } # End for i
        } # End if (noconv != l)
        
        if(l == 1){
          break
        }
        l = l - 1
        noconv = l
      }else{ # r != 0
        noconv = noconv - 1
      }
    } # End while(noconv>=0)
    
    if(l > 1){
      noconv = 1
      while(noconv <= l){
        c = 0
        
        for(i in c(k:l)){
          if(i == noconv) next
          if(A[[i]][[noconv]] != 0.0){
            c = 1
            break
          }
        } # End for i
        if(c == 0){
          scale1[[k]] = noconv
          
          if(noconv != k){
            for(i in c(1:l)){
              f = A[[i]][[noconv]]
              A[[i]][[noconv]] = A[[i]][[k]]
              A[[i]][[k]] = f
            } # End for i
            for(i in c(k:N)){
              f = A[[noconv]][[i]]
              A[[noconv]][[i]] = A[[k]][[i]]
              A[[k]][[i]] = f
            } # End for i
          } # End if(noconv != k)
          k = k + 1
          noconv = k
        }else{ # c!=0
          noconv = noconv + 1
        }
      } # End while(noconv <= l)
      for(i in c(k:l)){
        scale1[[i]] = 1.0
      }
      repeat{
        noconv = 0
        for( i in c(k:l)){
          c = r = 0.0
          for(j in c(k:l)){
            if(j == i) next
            c = c + abs(A[[j]][[i]])
            r = r + abs(A[[i]][[j]])
          } # End for j
          if((c == 0.0) || (r == 0.0)) next
          g = r / radix
          f = 1.0
          s = c + r
          while (c < g) {
            f = f * radix
            c = c * b2
          } # End while(c<g)
          g = r * radix
          while(c >= g) {
            f = f / radix
            c = c / b2
          }
          if((c + r) / f < 0.95 * s){
            g = 1.0 / f
            scale1[[i]] = scale1[[i]] * f
            noconv = 1
            for(j in c(k:N)){
              A[[i]][[j]] = A[[i]][[j]] * g
            }
            for(j in c(0:l)){
              A[[j]][[i]] = A[[j]][[i]] * f
            }
          } # End if((c + r) / f < 0.95 * s)
        } # End for i
        if(!noconv) break
      } # End repeat
    } # End if(l>0)
    low1 = k
    igh = l
    
    for(i in c((low1 + 1):(igh - 1))){
      if(low1>=igh) break
      k = i
      c = 0.0
      for(j in c(i:igh)){
        if(abs(A[[j]][[i - 1]]) > abs(c)){
          c = A[[j]][[i - 1]]
          k = j
        } # End if
      } # End for j
      trace1[[i]] = k
      if(k != i){
        for(j in c((i - 1):N)){
          r = A[[k]][[j]]
          A[[k]][[j]] = A[[i]][[j]]
          A[[i]][[j]] = r
        } # End for j
        
        for(j in c(1:igh)){
          r = A[[j]][[k]]
          A[[j]][[k]] = A[[j]][[i]]
          A[[j]][[i]] = r
        } # End for j
      } # End if (k != i)
      if(c != 0.0){
        for(m in c((i + 1):igh)){
          r = A[[m]][[i - 1]]
          if( r != 0.0){
            r = A[[m]][[i - 1]] = r / c
            for(j in c(i:N)){
              A[[m]][[j]] = A[[m]][[j]] - (r * A[[i]][[j]])
            }
            for(j in c(1:igh)){
              A[[j]][[i]] = A[[j]][[i]] + r * A[[j]][[m]]
            }
          } # End if(r!=0.0)
        } # End for m
      } # End if(c!=0.0)
    } # End for i
    
    for(i in c(1:N)){
      for(j in c(1:N)){
        B[[i]][[j]] = 0.0
      }
      B[[i]][[i]] = 1.0
    } # End for i
    
    for(i in c((igh - 1):(low1 + 1))){
      if(low1>=igh) break
      k = trace1[[i]]
      for(j in c((i + 1):igh)){
        B[[j]][[i]] = A[[j]][[i - 1]]
      }
      if(i == k){
        next
      }
      for(j in c(i:igh)){
        B[[i]][[j]] = B[[k]][[j]]
        B[[k]][[j]] = 0.0
      } # End for j
      B[[k]][[i]] = 1.0
    }
    oPar.outEr = ierr
    
    temp = hqr2(N, A, B, low1, igh, wi, wr, oPar.outEr);
    A = temp[[1]]
    B = temp[[2]]
    wi = temp[[3]]
    wr = temp[[4]]
    oPar.outEr = temp[[5]]
    ierr = oPar.outEr
    if(ierr == -1){
      if(low1 != igh && low1<=igh){
        for(i in c(low1:igh)){
          s = scale1[[i]]
          for(j in c(1:N)){
            B[[i]][[j]] = B[[i]][[j]] * s
          }
        } # End for i
      } # End for if(low1!=igh)
      i = low1 - 1
      while(i>=1){
        k = scale1[[i]]
        if(k!=i){
          for(j in (1:N)){
            s = B[[i]][[j]]
            B[[i]][[j]] = B[[k]][[j]]
            B[[k]][[j]] = s
          } # End for j
        }# End if(k!=i)
        i = i - 1
      } # End while i
      
      i = igh + 1
      while(i<=N){
        k = scale1[[i]]
        if(k!=i){
          for(j in c(1:N)){
            s = B[[i]][[j]]
            B[[i]][[j]] = B[[k]][[j]]
            B[[k]][[j]] = s
          } # End for j
        } # End for if(k!=i)
        i = i + 1
      } # End while i
      B = norVecC(N,B,wi);
    } # End if ierr = -1
    return(list(A,B,wr,wi,oPar.outEr))
  } # End of calcEigSysReal
  numeric._dim = function(x){
    ret1 = list()
    index = 1
    while(is.list(x)){
      ret1[[index]] = length(x)
      index = index + 1
      x = x[[1]]
    }
    return(ret1)
  }
  numeric.dim = function(x){
    y = z = NULL
    if(is.list(x)){
      y = x[[1]]
      if(is.list(y)){
        z = y[[1]]
        if(is.list(z)){
          return(numeric._dim(x))
        }
        return(list(length(x), length(y)))
      }
      return(list(length(x)))
    }
    return(list())
  }

  numeric.getBlock = function(x, from, to){
    s = numeric.dim(x)
    foo = function(x, k){
      i = NULL
      a = from[k] - 1 # 0
      n = to[k] - a - 1 # n-1
      ret1 = list()
      if(k == length(s)){
        for(i in c(n:0)){
          if(i < 0) break
          ret1[[i + 1]] = x[[i + a + 1]]
        }
        return(ret1)
      }
      for(i in c(n:0)){
        ret1[[i + 1]] = foo(x[[i + a + 1]], k + 1)
      }
      return(ret1)
    }
    tmp = foo(x, 1)
    return(tmp)
  }
  numeric.setBlock = function(x, from, to, B){
    s = numeric.dim(x)
    foo1 = function(x, y, k){
      a = from[k] - 1
      n = to[k] - a - 1
      if(k == length(s)){
        for(i in c(n:0)){
          x[[i + a + 1]] = y[[i + 1]]
        }
        return(x)
      }
      for(i in c(n:0)){
        x[[i + a + 1]] = foo1(x[[i + a + 1]], y[[i + 1]], k + 1)
      }
      return(x)
    }
    x = foo1(x, B, 1)
    return(x)
  }
  numeric.transpose = function(x){
    data = do.call(rbind,x)
    data = t(data)
    da = list()
    for(i in 1:nrow(data)){
      da[[i]] = list()
      for(j in 1:ncol(data)){
        da[[i]][[j]] = data[i,j][[1]]
      }
    }
    return (da)
  }
  numeric.dotMMsmall = function(x,y){
    i = j = k = p = q = r = ret1 = foo = bar = woo = i0 = k0 = p0 = r0 = NULL
    p = length(x)
    q = length(y)
    r = length(y[[1]])
    ret1 = list()
    for(i in c(p:1)){
      foo = list()
      bar = x[[i]]
      for(k in c(r:1)){
        woo = bar[[q]]*y[[q]][[k]]
        if((q-1)>=2){
          for(j in seq(q - 1,2,-2)){
            i0 = j - 1
            woo = woo + bar[[j]]*y[[j]][[k]] + bar[[i0]]*y[[i0]][[k]]
          }
          # j = j - 2
        }
        if(!is.null(j)&&(j - 2) == 1){
          woo = woo + bar[[1]]*y[[1]][[k]]
        }
        foo[[k]] = woo
      }
      ret1[[i]] = foo
    }
    return(ret1)
  }
  numeric._getCol = function(A,j,x){
    n = length(A)
    i = n
    while(i > 1){
      x[[i]] = A[[i]][[j]]
      i = i - 1
      x[[i]] = A[[i]][[j]]
      i = i - 1
    }
    if(i == 1){
      x[[1]] = A[[1]][[j]]
    }
    return(x)
  }
  numeric.dotVV = function(x,y){
    i = NULL
    n = length(x)
    i1 = NULL
    ret1 = x[[n]]*y[[n]]
    for(i in seq(n-1,2,-2)){
      i1 = i - 1
      ret1 = ret1 + x[[i]]*y[[i]] + x[[i1]]*y[[i1]]
    }
    if(i==3){
      ret1 = ret1 + x[[1]]*y[[1]]
    }
    return(ret1)
  }
  numeric.dotMMbig = function(x,y){
    gc1 = numeric._getCol
    p = length(y)
    v = list()
    m = length(x)
    n = length(y[[1]])
    A = list()
    xj = NULL
    VV = numeric.dotVV
    i = j = k = z = NULL
    p = p - 1
    m = m - 1
    i = m + 1
    while(i!=0){
      A[[i]] = list()
      i = i - 1
    }
    n = n - 1
    i = n + 1
    while(i!=0){
      v = gc1(y,i,v)
      j = m + 1
      while(j!=0){
        z = 0
        xj = x[[j]]
        A[[j]][[i]] = VV(xj,v)
        j = j - 1
      }
      i = i - 1
    }
    return(A)
  }
  numeric.dotMV = function(x,y){
    p = length(x)
    q = length(y)
    i = NULL
    ret1 = list()
    dotVV = numeric.dotVV
    for(i in c(p:1)){
      ret1[[i]] = dotVV(x[[i]],y)
    }
    return(ret1)
  }
  numeric.dotVM = function(x,y){
    i = j = k = p = q = r = ret1 = foo = bar = woo = NULL
    i0 = k0 = p0 = r0 = s1 = s2 = s3 = baz = accum = NULL
    p = length(x)
    q = length(y[[1]])
    ret1 = list()
    for(k in c(q:1)){
      woo = x[[p]]*y[[p]][[k]]
      j = p - 1
      while(j>=2){
        i0 = j - 1
        woo = woo + x[[j]]*y[[j]][[k]] + x[[i0]]*y[[i0]][[k]]
        j = j - 2
      }
      if(j == 1){
        woo = woo + x[[1]]*y[[1]][[k]]
      }
    }
    return(ret1)
  }
  numeric.add = function(x,y){
    for(i in c(1:length(x))){
      for(j in c(1:length(x[[i]]))){
        x[[i]][[j]] = x[[i]][[j]] + y[[i]][[j]]
      }
    }
    return(x)
  }
  numeric.add3 = function(x,y){
    for(i in c(1:length(x))){
      for(j in c(1:length(x[[i]]))){
        for(k in c(1:length(x[[i]][[j]]))){
          x[[i]][[j]][[k]] = x[[i]][[j]][[k]] + y[[i]][[j]][[k]]
        }
        
      }
    }
    return(x)
  }
  numeric.addeq = function(A,B){ # +=
    for(i in c(1:length(A))){
      for(j in c(1:length(B[[i]]))){
        A[[i]] = A[[i]]+B[[i]][[j]]
      }
    }
    return(A)
  }
  numeric.addeq1 = function(A,B){ # +=
    for(i in c(1:length(A))){
      for(j in c(1:length(A[[i]]))){
        A[[i]][[j]] = A[[i]][[j]] + B[[i]][[j]]
      }
    }
    return(A)
  }
  numeric.mul = function(A,B){
    A = as.double(A)
    for(i in c(1:length(B))){
      B[[i]][[1]] = A * as.double(B[[i]][[1]])
    }
    return(B)
  }
  numeric.mul1 = function(A,B){
    A = as.double(A)
    for(i in c(1:length(B))){
      for(j in c(1:length(B[[1]]))){
        B[[i]][[j]] = A * as.double(B[[i]][[j]])
      }
    }
    return(B)
  }
  numeric.mul2 = function(A,B){
    for(i in 1:length(A)){
      for(j in 1:length(A[[i]])){
        A[[i]][[j]] = A[[i]][[j]]*B[[i]][[j]]
      }
    }
    return(A)
  }
  numeric.mul3 = function(A,B){
    for(i in 1:length(A)){
      for(j in 1:length(A[[i]])){
        for(k in 1:length(A[[i]][[j]])){
          A[[i]][[j]][[k]] = A[[i]][[j]][[k]] * B
        }
      }
    }
    return(A)
  }
  numeric.mul4 = function(A,B){
    for(i in 1:length(A)){
      for(k in 1:length(A[[i]][[1]])){
        A[[i]][[1]][[k]] = A[[i]][[1]][[k]] * B[[1]]
      }
      for(k in 1:length(A[[i]][[2]])){
        A[[i]][[2]][[k]] = A[[i]][[2]][[k]] * B[[2]]
      }
    }
    return(A)
  }
  numeric.mulVS = function(x, y){
    print("Error:mulVS")
    return()
  }
  numeric.mulSV = function(x,y){
    print("Error:mulSV")
    return()
  }
  numeric.div = function(A,B){ # /
    B = as.double(B)
    da = list()
    for(i in 1:length(A)){
      da[[i]] = as.double(A[[i]])/B
    }
    return(da)
  }
  numeric.sub = function(A,B){ # -
    for(i in c(1:length(A))){
      for(j in c(1:length(A[[i]]))){
        A[[i]][[j]] = A[[i]][[j]] - B[[i]][[j]]
      }
    }
    return(A)
  }
  numeric.norm2Squared = function(x){
    # ['accum += xi*xi;','var accum = 0;']
    accum = 0.0
    for(i in c(1:length(x))){
      if(i < 1) break
      for(j in 1:length(x[[1]])){
        if(!is.na(x[[i]][[j]])){
          accum = accum + as.double(x[[i]][[j]])*as.double(x[[i]][[j]])
        }
      }
    }
    return(accum)
  }
  numeric.norm2 = function(x){
    return(sqrt(numeric.norm2Squared(x)))
  }
  numeric.dot = function(x,y){
    # a=as.character(length(numeric.dim(x))*1000 + length(numeric.dim(y)))
    # print(a)
    tmp = switch(
      as.character(length(numeric.dim(x))*1000 + length(numeric.dim(y))),
      "2002" = if(length(y) < 10) numeric.dotMMsmall(x,y) else numeric.dotMMbig(x,y),
      "2001" = numeric.dotMV(x,y),
      "1002" = numeric.dotVM(x,y),
      "1000" = numeric.mulVS(x,y),
      "1" = numeric.mulSV(x,y),
      "0" = x * y
    )
    if(is.null(tmp)){
      print("Error : numeric.dot only works on vectors and matrices")
    }
    return(tmp)
  }
  
  qtools.MGS = function(V){
    N = length(V)
    getV = function(A,i){
      return(numeric.getBlock(A,c(1,i),c(N,i)))
    }
    setV = function(A, i, v){
      A = numeric.setBlock(A, c(1,i),c(N,i),v)
      return(A)
    }
    T = numeric.transpose
    dot = numeric.dot
    mul = numeric.mul
    U = numeric.rep(c(N,N),0)
    v0 = getV(V,1)
    U = setV(U,1,numeric.div(v0,numeric.norm2(T(v0)[[1]])))
    for(i in c(2:N)){
      U = setV(U,i,getV(V,i))
      for(j in c(1:(i-1))){
        ui = getV(U,i)
        uj = getV(U,j)
        U = setV(U,i,numeric.sub(ui,mul(dot(T(ui),uj)[[1]][[1]],uj)))
      }
      ui = getV(U,i)
      U = setV(U,i,numeric.div(ui,numeric.norm2(T(ui)[[1]])))
    }
    return(U)
  }
  qtools.correct = function(A){
    for(i in c(1:length(A))){
      for(j in c(1:length(A[[i]]))){
        if(length(A[[i]][[j]]) == 0 || abs(A[[i]][[j]]) < qtools.tolerance){
          A[[i]][[j]] = 0
        }
      }
    }
    return(A)
  }

  qtools.eig = function (A){
    N = length(A)
    B = numeric.rep(c(N,2*N),0)
    wr = numeric.rep(N,0)
    wi = numeric.rep(N,0)
    oPar.outEr = 0
    tmp = calcEigSysReal(N,A)
    B = tmp[[2]]
    wr = tmp[[3]]
    wi = tmp[[4]]
    oPar.outEr = tmp[[5]]
    if(oPar.outEr != -1){
      warning("Eigenvector computation failed.")
    }
    if(any(wi>0)){
      warning("Real symmetric matrix should always have real eigenvalues.")
    }
    tmp_func <- function(v){
      tmp_list = list()
      tmp_tag = 1
      for(t_i in v){
        if(abs(t_i)<qtools.tolerance){
          tmp_list[[tmp_tag]] = 0
        }else{
          tmp_list[[tmp_tag]] = t_i
        }
        tmp_tag = tmp_tag + 1
      }
      return(tmp_list)
    }
    eigenvalues = tmp_func(wr)
    eigenvectors = numeric.getBlock(B, c(1,1), c(N,N))
    return(list(eigenvalues, qtools.correct(qtools.MGS(eigenvectors))))
  }
  
  qtools.specdecomp = function(A){
    eig = qtools.eig(A)
    eigenvalues = list()
    eigenprojectors = list()
    i = NA
    for(i in c(1:length(A))){
      found = FALSE
      if(1 <= length(eigenvalues)){
        for(j in c(1:length(eigenvalues))){
          if(abs(eig[[1]][[2]] - 0) < 0.0001){
            v = numeric.getBlock(eig[[2]],c(1,i),c(length(eig[[2]]),i))
            eigenprojectors[[j]] = numeric.addeq(eigenprojectors[[j]],numeric.dot(v,numeric.transpose(v)))
            found = TRUE
          }
        }
      }
      if(found == FALSE){
        eigenvalues = c(eigenvalues,eig[[1]][[i]])
        v = numeric.getBlock(eig[[2]],c(1,i),c(length(eig[[2]]),i))
        eigenprojectors = c(eigenprojectors,list(numeric.dot(v,numeric.transpose(v))))
      }
    }
    tmp_func <- function(v){
      for(i in c(1:length(v))){
        for(j in c(1:length(v[[1]]))){
          for(k in c(1:length(v[[1]][[1]]))){
            if(abs(v[[i]][[j]][[k]])<qtools.tolerance){
              v[[i]][[j]][[k]] = 0
            }
          }
        }
      }
      return(v)
    }
    return(list(eigenvalues,tmp_func(eigenprojectors)))
  }
  numeric.T = function(x,y){
    return(list(x,y))
  }
  numeric.t = function(x,y){
    return(numeric.T(x,y))
  }
  setInterval = function(){
    print("Set time!")
  }
  qtools.norm = function(A){
    eig = qtools.eig(numeric.dot(A,numeric.transpose(A)))
    max1 = eig[[1]][[1]]
    for(i in c(2:length(eig[[1]]))){
      if(eig[[1]][[i]] > max1){
        max1 = eig[[1]][[i]]
      }
    }
    return(sqrt(max1))
  }
  numeric.diag = function(d){
    i = i1 = j = Ai = NULL
    n = length(d)
    A = list()
    for(i in c(n:1)){
      Ai = list()
      i1 = i + 2
      j = n
      while(j>=i1){
        Ai[[j]] = 0
        Ai[[j-1]] = 0
        j = j - 2
      }
      if(j > i)
        Ai[[j]] = 0
      Ai[[i]] = d[[i]]
      j = i - 1
      while(j>=2){
        Ai[[j]] = 0
        Ai[[j-1]] = 0
        j = j - 2
      }
      if(j == 1)
        Ai[[1]] = 0
      A[[i]] = Ai
    }
    return(A)
  }
  numeric.identity = function(n){
    return(numeric.diag(numeric.rep(c(n),1)))
  }
  qtools.testBasis = function(N,eigenprojectors){
    print("qtools.testBasis():")
    Z = eigenprojectors[[1]]
    for(i in c(2:length(eigenprojectors))){
      Z = numeric.add(Z,eigenprojectors[[i]])
    }
    error = qtools.norm(qtools.correct(numeric.sub(numeric.identity(N),Z)))
    pass = (error < qtools.tolerance)
    if(pass){
      print("Passed")
    }else{
      print("Failed")
    }
    return(pass)
  }
  qtools.testDecomp = function(A,eigenvalues,eigenprojectors){
    print("qtools.testDecomp():")
    Z = numeric.rep(c(length(A),length(A)),0)
    for(i in c(1:length(eigenvalues))){
      Z = numeric.addeq1(Z,numeric.mul1(eigenvalues[[i]],eigenprojectors[[i]]))
    }
    pass = (qtools.norm(qtools.correct(numeric.sub(A,Z))) < qtools.tolerance)
    if(pass){
      print("Passed")
    }else{
      print("Failed")
    }
    return(pass)
  }
  
  qtools.testSpectralDecomposition = function(A,B,verbose){
    print("***************************")
    print("NOW TEST!")
    
    return(qtools.testBasis(length(A), B[[2]]) && qtools.testDecomp(A, B[[1]], B[[2]]))
  }
  
  qtools.qwalk = function(eigenvalues, eigenprojectors, N, t){
    U = numeric.t(numeric.rep(c(N,N),0),numeric.rep(c(N,N),0))
    i = 1
    while(i<=length(eigenvalues)){
      foo = function(A,B){
        real = A[[1]]*B[[1]] - A[[2]]*B[[2]]
        imag = A[[1]]*B[[2]] + A[[2]]*B[[1]]
        A[[1]] = real
        A[[2]] = imag
        B[[1]] = cos(A[[2]])*exp(A[[1]])
        B[[2]] = sin(A[[2]])*exp(A[[1]])
        return(B)
      }
      c = foo(numeric.t(0,-t),eigenvalues[[i]])
      foo2 = function(A,B){
        C = A
        C[[1]] = numeric.add(numeric.mul1(B[[1]],A[[1]]),numeric.mul1(B[[1]],A[[2]]))
        C[[2]] = numeric.add(numeric.mul1(B[[2]],A[[1]]),numeric.mul1(B[[2]],A[[2]]))
        return(C)
      }
      tmp = foo2(eigenprojectors[[i]],c)
      U = numeric.add3(U,tmp)
      i = i + 1
    }
    return(U)
  }

  # pertime, time_scals
  qwalk.startFromGraph<-function(pertime,time_scals){
    if(pertime<=0||!is.numeric(time_scals)||!is.vector(time_scals)){
      return()
    }
    if(is.na(qwalk.startIndex)){
      print("Error.")
      return()
    }
    N<-length(edges)
    qwalk.mat = numeric.rep(c(N,N),0)
    for(i in c(1:N)){
      for(j in c(1:N)){
        if(i == j){
          next
        }
        edge <- qmanip.getEdges(i,j)
        qwalk.mat[[i]][[j]] = if (edge > 0)  1 else 0
      }
    }
    qwalk.curTime <- 0
    B <- qtools.specdecomp(qwalk.mat)
    
    temp_B = B[[1]]
    tag_B = 1
    for(i in c(1:length(temp_B))){
      temp_B[[tag_B]] = numeric.t(temp_B[[tag_B]],0)
      tag_B = tag_B + 1
    }
    qwalk.eigenvalues <- temp_B
    temp_B = B[[2]]
    tag_B = 1
    for(i in c(1:length(temp_B))){
      temp_B[[tag_B]] = numeric.t(temp_B[[tag_B]],numeric.rep(c(N,N),0))
      tag_B = tag_B + 1
    }
    qwalk.eigenprojectors <- temp_B

    istrue = qtools.testSpectralDecomposition(qwalk.mat,B,FALSE)
    if(istrue){
      print("SUCCESS!")
    }else{
      print("FAILED!")
    }
    # set NULL
    alldata = NULL
    
    times<-length(time_scals)
    pb<-progress_bar$new(
      format = "Completed: [:bar] :percent Elapsed time: :elapsed",
      total = times, clear = FALSE, width = 60)
    for(i in 1:times){
      recodeCounter = 0
      flag = 0
      # define a data, list
      t_data = list()
      while(1){
        qwalk.nodes = qwalk.loop(qwalk.nodes,time_scals[i], qwalk.curTime, qwalk.eigenvalues, qwalk.eigenprojectors)
        qwalk.curTime = qwalk.nodes[[2]]
        qwalk.nodes = qwalk.nodes[[1]]
        if(flag == 1 || qwalk.nodes[[Tag_node]] > 0){
          flag = 1
          
          t_data = c(t_data,list(qwalk.nodes))
          recodeCounter = recodeCounter + 1
        }
        if(recodeCounter==pertime){
          break
        }
      }
      t_data <- do.call(rbind.data.frame,t_data)
      t_data <- as.matrix(t_data)
      if(is.null(alldata)){
        alldata <- array(t_data, dim = c(dim(t_data),i),
                         dimnames = list(c(1:nrow(t_data)),colnames(edges),paste("S",c(1:i),sep="")))
      }else{
        alldata <- array(c(alldata, t_data), dim = c(dim(t_data),i),
                         dimnames = list(c(1:nrow(t_data)),colnames(edges),paste("S",c(1:i),sep="")))
      }
      qwalk.curTime <- 0
      pb$tick()
      Sys.sleep(1/100)
    }
    # alldata <- do.call(rbind.data.frame,alldata)
    # colnames(alldata)<-c(1:length(alldata))
    return(alldata)
  }
  
  qwalk.loop<-function(qwalk.nodes,deltatime, curtime, eigenvalues, eigenprojectors){
    N = length(qwalk.nodes)
    t = curtime
    U = qtools.qwalk(eigenvalues,eigenprojectors,N,t)

    for(i in c(1:length(qwalk.nodes))){
      ampl = list()
      ampl[[1]] = numeric.getBlock(U[[1]],c(i,qwalk.startIndex),c(i,qwalk.startIndex))[[1]]
      ampl[[2]] = numeric.getBlock(U[[2]],c(i,qwalk.startIndex),c(i,qwalk.startIndex))[[1]]
      prob = ampl[[1]][[1]]*ampl[[1]][[1]]+ampl[[2]][[1]]*ampl[[2]][[1]]
      if(!getfloat){
        qwalk.nodes[[i]] <- round(prob*100)
      }else{
        qwalk.nodes[[i]] <- if ((prob) >= 1e-6) prob else 0
      }
    }
    # qwalk.curTime <- qwalk.curTime + deltatime
    return(list(QN = qwalk.nodes, cur = curtime+deltatime))
  }
  # edges, startindex, scals, lens, getfloat
  res<-qwalk.startFromGraph(lens, scals)
  # res data.frame
  res<-list(ctqw = res,edges=as.matrix(edges),startindex=startindex,
            scals = scals, lens=lens, getfloat=getfloat, tag_node=Tag_node)
  res<-structure(res,class="CTQW")
  return(res)
}
