% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GAD.R
\name{GAD}
\alias{GAD}
\alias{dGAD}
\alias{pGAD}
\alias{qGAD}
\alias{rGAD}
\title{Generalized quantile-based asymmetric family}
\usage{
dGAD(y, eta, phi, alpha, f, g)

pGAD(q, eta, phi, alpha, F, g)

qGAD(beta, eta, phi, alpha, F, g, QF = NULL, lower = -Inf,
  upper = Inf)

rGAD(n, eta, phi, alpha, F, g, lower = -Inf, upper = Inf, QF = NULL)
}
\arguments{
\item{y, q}{These are each a vector of quantiles.}

\item{eta}{This is the location parameter \eqn{\eta}.}

\item{phi}{This is the scale parameter  \eqn{\phi}.}

\item{alpha}{This is the index parameter  \eqn{\alpha}.}

\item{f}{This is the reference density function \eqn{f} which is a standard version of a unimodal and symmetric around 0 density.}

\item{g}{This is the "link" function. The function \eqn{g} is to be differentiated. Therefore, \eqn{g} must be written as a function. For example, {g<-function(y)\{log(y)\}} for log link function.}

\item{F}{This is the cumulative distribution function \eqn{F} of the unimodal and symmetric around 0 reference density function \eqn{f}.}

\item{beta}{This is a vector of probabilities.}

\item{QF}{This is the quantile function of the reference density \eqn{f}.}

\item{lower}{This is the lower limit of the domain (support of the random variable) \eqn{f_{\alpha}^g(y;\eta,\phi)}, default {-Inf}.}

\item{upper}{This is the upper limit of the domain (support of the random variable) \eqn{f_{\alpha}^g(y;\eta,\phi)}, default {Inf}.}

\item{n}{This is the number of observations, which must be a positive integer that has length 1.}
}
\description{
Density, cumulative distribution function, quantile function and random sample generation
from the generalized quantile-based asymmetric family of densities defined in Gijbels et al. (2019b).
}
\examples{
# Example 1: Let F be a standard normal cumulative distribution function then
f_N<-function(s){dnorm(s, mean = 0,sd = 1)} # density function of N(0,1)
F_N<-function(s){pnorm(s, mean = 0,sd = 1)} # distribution function of N(0,1)
QF_N<-function(beta){qnorm(beta, mean = 0, sd = 1, lower.tail = TRUE, log.p = FALSE)}

# For identiy link function
g_id<-function(y){y}
# For log-link function
g_log<-function(y){log(y)}

rnum<-rnorm(100)
beta=c(0.25,0.50,0.75)

# Density
dGAD(y=rnorm(100),eta=10,phi=1,alpha=0.5,f=f_N,g=g_id) # For identity link
dGAD(y=rexp(100,0.1),eta=10,phi=1,alpha=0.5,f=f_N,g=g_log) # For log-link

# Distribution function
pGAD(q=rnorm(100),eta=0,phi=1,alpha=.5,F=F_N,g=g_id) # For identity link
pGAD(q=rexp(100,0.1),eta=10,phi=1,alpha=.5,F=F_N,g=g_log) # For log-link

# Quantile function
qGAD(beta=beta,eta=0,phi=1,alpha=0.5,F=F_N,g=g_id) # For identity link
qGAD(beta=beta,eta=10,phi=1,alpha=0.5,F=F_N,g=g_log,lower = 0, upper = Inf)  # For log-link

# random sample generation
rGAD(n=100,eta=0,phi=1,alpha=.5,F=F_N,g=g_id ,lower = -Inf, upper = Inf,QF=NULL) # For identity link
rGAD(n=100,eta=10,phi=1,alpha=.5,F=F_N,g=g_log ,lower =0, upper = Inf,QF=NULL)   # For log-link


# Example 2: Let F be a standard Laplace cumulative distribution function then
f_La<-function(s){0.5*exp(-abs(s))} # density function of Laplace(0,1)
F_La<-function(s){0.5+0.5*sign(s)*(1-exp(-abs(s)))} # distribution function of Laplace(0,1)
QF_La<-function(beta){-sign(beta-0.5)*log(1-2*abs(beta-0.5))}

# For identiy link function
g_log<-function(y){log(y)}
beta=c(0.25,0.50,0.75)

# Density
dGAD(y=rnorm(100),eta=10,phi=1,alpha=0.5,f=f_La,g=g_id) # For identity-link
dGAD(y=rexp(100,0.1),eta=10,phi=1,alpha=0.5,f=f_La,g=g_log) # For log-link

# Distribution function
pGAD(q=rnum,eta=0,phi=1,alpha=.5,F=F_La,g=g_id) # For identity-link
pGAD(q=rexp(100,0.1),eta=10,phi=1,alpha=.5,F=F_La,g=g_log) # For log-link

# Quantile function
qGAD(beta=beta,eta=0,phi=1,alpha=0.5,F=F_La,g=g_id,lower = -Inf, upper = Inf) # For identity link
qGAD(beta=beta,eta=10,phi=1,alpha=0.5,F=F_La,g=g_log,lower = 0, upper = Inf) # For log-link

# random sample generation
rGAD(n=100,eta=0,phi=1,alpha=.5,F=F_La,g=g_id) # For identity link
rGAD(n=100,eta=10,phi=1,alpha=.5,F=F_La,g=g_log ,lower =0, upper = Inf,QF=NULL)   # For log-link




}
\references{
{
 Gijbels, I., Karim, R. and Verhasselt, A. (2019b). Quantile estimation in a generalized  asymmetric distributional setting. To appear in \emph{Springer Proceedings in Mathematics & Statistics, Proceedings of `SMSA 2019', the 14th Workshop on Stochastic Models, Statistics and their Application}, Dresden, Germany, in March 6--8, 2019. Editors: Ansgar Steland, Ewaryst Rafajlowicz, Ostap Okhrin.
}
}
