#' @title draw an GenePathwayOncoplots
#' @description Load the data in MAF format and draws an GenePathwayOncoplots.
#' @param maffile A data of MAF format.
#' @param path_gene User input pathways geneset list.
#' @param isTCGA Is input MAF file from TCGA source. If TRUE uses only first 12 characters from Tumor_Sample_Barcode.
#' @param top How many top genes to be drawn,genes are arranged from high to low depending on the frequency of mutations. defaults to 20.
#' @param mut_status The mutations matrix,generated by `get_mut_matrix`.
#' @param risk_score Samples' PTMB-related risk score,which could be a biomarker for survival analysis and immunotherapy prediction.
#' @param cut_off A threshold value(the median risk score as the default value).Using this value to divide the sample into high and low risk groups with different overall survival.
#' @param pathway_name The name of the pathway that you want to visualize.For example "Gap junction"
#' @param clinicalFeatures Columns names from 'clinical.data' slot of MAF to be drawn in the plot. Dafault "sample_group".
#' @param annotationColor Custom colors to use for sample annotation-"sample_group". Must be a named list containing a named vector of colors. Default "red" and "green".
#' @param final_signature The pathway signature,use to map gene in the GenePathwayOncoplots.
#' @param sortByAnnotation Logical sort oncomatrix (samples) by provided 'clinicalFeatures'. Sorts based on first 'clinicalFeatures'. Defaults to TRUE. column-sort.
#' @param removeNonMutated Logical. If TRUE removes samples with no mutations in the GenePathwayOncoplots for better visualization. Default FALSE.
#' @param drawRowBar Logical. Plots righ barplot for each gene. Default TRUE.
#' @param drawColBar Logical plots top barplot for each sample. Default TRUE.
#' @param leftBarData Data for leftside barplot. Must be a data.frame with two columns containing gene names and values. Default 'NULL'.
#' @param leftBarLims Limits for 'leftBarData'. Default 'NULL'.
#' @param rightBarData Data for rightside barplot. Must be a data.frame with two columns containing to gene names and values. Default 'NULL' which draws distibution by variant classification. This option is applicable when only 'drawRowBar' is TRUE.
#' @param rightBarLims Limits for 'rightBarData'. Default 'NULL'.
#' @param topBarData Default 'NULL' which draws absolute number of mutation load for each sample. Can be overridden by choosing one clinical indicator(Numeric) or by providing a two column data.frame contaning sample names and values for each sample. This option is applicable when only 'drawColBar' is TRUE.
#' @param logColBar Plot top bar plot on log10 scale. Default FALSE.
#' @param draw_titv Logical Includes TiTv plot. Default FALSE
#' @param showTumorSampleBarcodes Logical to include sample names.
#' @param fill Logical. If TRUE draws genes and samples as blank grids even when they are not altered.
#' @param showTitle Default TRUE.
#' @param titleText Custom title. Default 'NULL'.
#' @importFrom maftools subsetMaf
#' @importFrom maftools oncoplot
#' @return No return value
#' @export
#' @examples
#' #obtain the risksciore
#' data(km_data)
#' risk_score<-km_data$multiple_score
#' names(risk_score)<-rownames(km_data)
#' cut_off<-median(risk_score)
#' #load the dtata
#' data(final_signature,path_gene,mut_status,maffile)
#' ##draw an GenePathwayOncoplots
#' get_Oncoplots(maffile,path_gene,mut_status,risk_score,cut_off,final_signature,"Gap junction")
#'
get_Oncoplots<-function(maffile,path_gene,mut_status,risk_score,cut_off,final_signature,pathway_name,isTCGA=FALSE,top=20,
                        clinicalFeatures = "sample_group",annotationColor=c("red","green"),sortByAnnotation = TRUE,
                        removeNonMutated= FALSE,drawRowBar= TRUE,drawColBar= TRUE,leftBarData= NULL,leftBarLims= NULL,
                        rightBarData= NULL,rightBarLims= NULL,topBarData= NULL,logColBar= FALSE,draw_titv= FALSE,
                        showTumorSampleBarcodes= FALSE,fill= TRUE,showTitle = TRUE,titleText = NULL){
  all_sample<-as.data.frame(maffile@clinical.data)
  rownames(all_sample)<-all_sample[,1]
  if(isTCGA){
    samples<-substr(gsub(pattern = "\\.",replacement = "\\-",names(risk_score)),1,12)
  }else{
    samples<-names(risk_score)
  }
  b<-c()
  for(i in 1:length(samples)){
    a<-which(grepl(pattern = samples[i],x =all_sample[,1],ignore.case = TRUE))[1]
    b<-c(b,a)
  }
  clinical<-as.data.frame(cbind(all_sample[b,],sample_group=risk_score))
  clinical[,2]<-as.numeric(clinical[,2])
  a<-which(clinical$sample_group<=cut_off)
  clinical[a,2]<-"low_risk"
  clinical[-a,2]<-"high_risk"
  colnames(clinical)<-c("Tumor_Sample_Barcode","sample_group")
  newdata<-matrix(ncol = 1,nrow=dim(maffile@clinical.data)[1])
  maffile@clinical.data<-cbind(maffile@clinical.data,newdata)
  colnames(maffile@clinical.data)<-c("Tumor_Sample_Barcode","sample_group")
  loc<-match(maffile@clinical.data$Tumor_Sample_Barcode,clinical$Tumor_Sample_Barcode)
  maffile@clinical.data$sample_group<-clinical$sample_group[loc]
  maffile<-subsetMaf(maffile,tsb = clinical[,1])
  col<-annotationColor
  names(col)<-c("high_risk","low_risk")
  fabcolors<-list(sample_group=col)
  #gene
  b<-path_gene[final_signature]
  gene_fre<-as.matrix(apply(mut_status,1,function(x){length(which(x!=0))/length(x)}))
  Genes<-c()
  Pathway<-c()
  for(i in 1:length(path_gene)){
    aa<-as.matrix(gene_fre[match(intersect(rownames(gene_fre),path_gene[[i]]),rownames(gene_fre)),])
    b<-as.numeric(aa[,1])
    names(b)<-rownames(aa)
    b<-sort(b,decreasing = T)[1:top]
    c<-unlist(rep(names(path_gene)[[i]],length(b)))
    Genes<-c(Genes,names(b))
    Pathway<-c(Pathway,c)
  }
  pathway<-data.frame(Genes=Genes,Pathway=Pathway)
  pathway_spec<-pathway[which(pathway$Pathway%in%pathway_name),]
  oncoplot(maffile,clinicalFeatures = "sample_group",sortByAnnotation = sortByAnnotation,annotationColor = fabcolors,pathways = pathway_spec,removeNonMutated = removeNonMutated,
           drawRowBar= drawRowBar,drawColBar= drawColBar,leftBarData= leftBarData,leftBarLims= leftBarLims,rightBarData= rightBarData,rightBarLims= rightBarLims,topBarData= topBarData,logColBar= logColBar,draw_titv= draw_titv,
           showTumorSampleBarcodes= showTumorSampleBarcodes,fill= fill,showTitle = showTitle,titleText = titleText)
}
