#' @title density_plot

#' @description Function plots a density plot for gene expression data prepared by the score_genes function. The plots can be used for a quick assessment of the overall gene expression distribution.
#'
#' @param data Requires a  data frame generated by score_genes; class - data frame
#' @return  multiple plots; class - plots
#' @import lattice
#' @importFrom  dendextend find_k
#' @importFrom  dendextend color_labels
#' @importFrom  dendextend  color_branches
#' @importFrom  dendextend   colored_bars
#' @import cluster
#' @importFrom ggplot2 aes
#' @importFrom ggplot2 theme
#' @ImpportFrom ggplot2 element_text
#' @ImportFrom ggplot2 geom_point
#' @ImportFrom ggplot2 ggplot
#' @import RColorBrewer
#' @importFrom graphics legend
#' @import utils
#' @importFrom stats  hclust
#' @import methods
#' @importFrom stats as.dendrogram
#' @importFrom  RCurl getURL
#' @examples
#' \dontrun{
#' path_to_test_data<- system.file("extdata", "test_data.tabular", package="OmicInt")
#' # basic usage of density_plot
#' df<-utils::read.table(path_to_test_data)
#' density_plot(df)}
#' @export
density_plot<-function(data){

  #plot structures
  #access data
  classes_url <- RCurl::getURL("https://gitlab.com/Algorithm379/databases/-/raw/main/HS_protein_classes_curated.csv")
  classes <- utils::read.csv(text = classes_url)

  #download the data from curated databases
  location_url <- RCurl::getURL("https://gitlab.com/Algorithm379/databases/-/raw/main/Subcellular.locationmerged_protein_data.csv")
  location_df <- utils::read.csv(text = location_url)

  #prepare data frame
  data$"Class"<-ifelse(data$"Symbol"%in%classes$"Gene",classes$"Class","NA")

  #add associations

  data$"Location"<-ifelse(data$"Symbol"%in%location_df$"Symbol",location_df$"Subcellular.location","NA")
  #rename NAs
  data$"Location"<-ifelse( is.na(data$"Location"),"NA", data$"Location")

  #extract class features
  df<-table(data$"Class")
  df<-as.data.frame(sort(df,decreasing = TRUE))
  colnames(df)<-c("Class","Freq")
  if(nrow(df)>10){
    class_vector<-df$"Class"[1:10]
    n=10
  }else if (nrow(df)<10){
    class_vector<-df$"Class"
    n=nrow(df)
  }

  #select only class vectors
  data<-data[which(data$"Class"%in%class_vector),]

  #prepare color pallete
  qual_col_pals <- brewer.pal.info[which(brewer.pal.info$"category"%in%c('qual')),] #max number of colours 335, setting for qual gives 74
  col_vector <- unlist(mapply(brewer.pal, qual_col_pals$"maxcolors", rownames(qual_col_pals)))



  p1<-lattice::densityplot(~ Interactors, groups = data$"Class", data = data, plot.points = FALSE, auto.key = TRUE, par.settings = list(superpose.line = list(col = col_vector[1:length(class_vector)])),main=paste(n," highest frequency classes  distribution for interactors"))
  methods::show(p1)

  p2<-lattice::densityplot(~ LFCscore, groups = data$"Class", data = data, plot.points = FALSE, auto.key = TRUE, par.settings = list(superpose.line = list(col = col_vector[1:length(class_vector)])),main=paste(n," highest frequency classes  distribution for LFCscore"))
  methods::show(p2)

  p3<-lattice::densityplot(~ Specificity_score, groups = data$"Class", data = data, plot.points = FALSE, auto.key = TRUE, par.settings = list(superpose.line = list(col = col_vector[1:length(class_vector)])),main=paste(n," highest frequency classes  distribution for specificity score"))
  methods::show(p3)

  p4<-lattice::densityplot(~ Association_score, groups = data$"Class", data = data, plot.points = FALSE, auto.key = TRUE, par.settings = list(superpose.line = list(col = col_vector[1:length(class_vector)])),main=paste(n," highest frequency classes distribution for  association score"))
  methods::show(p4)

  #extract location features
  df<-table(data$"Location")
  df<-as.data.frame(sort(df,decreasing = TRUE))
  colnames(df)<-c("Location","Freq")
  if(nrow(df)>10){
    location_vector<-df$"Location"[1:10]
  }else if (nrow(df)<10){
    location_vector<-df$"Location"
  }

  #select only location vectors
  data<-data[which(data$"Location"%in%location_vector),]


  p1<-lattice::densityplot(~ Interactors, groups = data$"Location", data = data, plot.points = FALSE, auto.key = TRUE, par.settings = list(superpose.line = list(col = col_vector[1:length(location_vector)])),main=paste(n," highest frequency locations  distribution for interactors"))
  methods::show(p1)

  p2<-lattice::densityplot(~ LFCscore, groups = data$"Location", data = data, plot.points = FALSE, auto.key = TRUE, par.settings = list(superpose.line = list(col = col_vector[1:length(location_vector)])),main=paste(n," highest frequency locations  distribution for LFCscore"))
  methods::show(p2)

  p3<-lattice::densityplot(~ Specificity_score, groups = data$"Location", data = data, plot.points = FALSE, auto.key = TRUE, par.settings = list(superpose.line = list(col = col_vector[1:length(location_vector)])),main=paste(n," highest frequency locations  distribution for specificity score"))
  methods::show(p3)

  p4<-lattice::densityplot(~ Association_score, groups = data$"Location", data = data, plot.points = FALSE, auto.key = TRUE, par.settings = list(superpose.line = list(col = col_vector[1:length(location_vector)])),main=paste(n," highest frequency locations distribution for  association score"))
  methods::show(p4)

}
