% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MR.quantreg.r
\name{MR.quantreg}
\alias{MR.quantreg}
\title{Multiply Robust Estimation for Quantile Regression}
\usage{
MR.quantreg(model, imp.model = NULL, mis.model = NULL, moment = NULL,
  order = 1, L = 30, data, bootstrap = FALSE, bootstrap.size = 500,
  alpha = 0.05)
}
\arguments{
\item{model}{The quantile regression model of interest, defined by \code{\link{def.quantreg}}.}

\item{imp.model}{A list of possibly multiple lists of the form \code{list(list.1, list.2, ..., list.K)}, where \code{K} is the total number of different imputation models. For the \emph{k}-th imputation model, \code{list.k} is a list of possibly multiple models, each of which is defined by \code{\link{def.glm}} and imputes one single missing variable marginally. See details.}

\item{mis.model}{A list of missingness probability models defined by \code{\link{def.glm}}. The dependent variable is always specified as \code{R}.}

\item{moment}{A vector of auxiliary variables whose moments are to be calibrated.}

\item{order}{A numeric value. The order of moments to be calibrated.}

\item{L}{Number of imputations.}

\item{data}{A data frame with missing data encoded as \code{NA}.}

\item{bootstrap}{Logical. Should a bootstrap method be applied to calculate the standard error of the estimator and construct a Wald confidence interval for the quantile regression coefficients.}

\item{bootstrap.size}{A numeric value. Number of bootstrap resamples generated if \code{bootstrap = TRUE}.}

\item{alpha}{Significance level used to construct the 100(1 - \code{alpha})\% Wald confidence interval.}
}
\value{
\item{\code{coefficients}}{The estimated quantile regression coefficients.}
\item{\code{SE}}{The bootstrap standard error of \code{coefficients} when \code{bootstrap = TRUE}.}
\item{\code{CI}}{A Wald-type confidence interval based on \code{coefficients} and \code{SE} when \code{bootstrap = TRUE}.}
\item{\code{fit}}{A fitted object inheriting from class "\code{\link[quantreg]{rq}}" on \code{model}.}
}
\description{
\code{MR.quantreg()} is used for quantile regression with missing responses and/or missing covariates. Multiple missingness probability models and imputation models are allowed.
}
\details{
The function \code{MR.quantreg()} currently deals with data with one missingness pattern. When multiple variables are subject to missingness, their values are missing simultaneously. The method in Han et al. (2019) specifies an imputation model by modeling the joint distribution of the missing variables conditional on the fully observed variables. In contrast, the function \code{MR.quantreg()} specifies an imputation model by separately modeling the marginal distribution of each missing variable conditional on the fully observed variables. These marginal distribution models for different missing variables constitute one joint imputation model. Different imputation models do not need to model the marginal distribution of each missing variable differently.
}
\examples{
# Simulated data set
set.seed(123)
n <- 400
gamma0 <- c(1, 2, 3)
alpha0 <- c(-0.8, -0.5, 0.3)
S <- runif(n, min = -2.5, max = 2.5) # auxiliary variables
X1 <- rbinom(n, size = 1, prob = 0.5) # covariate X1
X2 <- rexp(n) # covariate X2
p.obs <- 1 / (1 + exp(alpha0[1] + alpha0[2] * S + alpha0[3] * S ^ 2)) # non-missingness probability
R <- rbinom(n, size = 1, prob = p.obs)
a.x <- gamma0[1] + gamma0[2] * X1 + gamma0[3] * X2
Y <- rnorm(n, a.x)
dat <- data.frame(S, X1, X2, Y)
dat[R == 0, c(2, 4)] <- NA # X1 and Y may be missing

# quantile regression model of interest
reg <- def.quantreg(formula = Y ~ X1 + X2, tau = 0.75)
# marginal imputation models for X1
impX1.1 <- def.glm(formula = X1 ~ S, family = binomial(link = logit))
impX1.2 <- def.glm(formula = X1 ~ S + X2, family = binomial(link = cloglog))
# marginal imputation models for Y
impY.1 <- def.glm(formula = Y ~ S, family = gaussian)
impY.2 <- def.glm(formula = Y ~ S + X2, family = gaussian)
# missingness probability models
mis1 <- def.glm(formula = R ~ S + I(S ^ 2), family = binomial(link = logit))
mis2 <- def.glm(formula = R ~ I(S ^ 2), family = binomial(link = cloglog))
# this example considers the following K = 3 imputation models for imputing the missing (X1, Y)
imp1 <- list(impX1.1, impY.1)
imp2 <- list(impX1.1, impY.2)
imp3 <- list(impX1.2, impY.1)

results <- MR.quantreg(model = reg, imp.model = list(imp1, imp2, imp3),
                       mis.model = list(mis1, mis2), L = 10, data = dat)
results$coefficients
summary(results$fit)
MR.quantreg(model = reg, moment = c(S, X2), order = 2, data = dat)$coefficients
}
\references{
Han, P., Kong, L., Zhao, J. and Zhou, X. (2019). A general framework for quantile estimation with incomplete data. \emph{Journal of the Royal Statistical Society: Series B (Statistical Methodology)}. \strong{81}(2), 305--333.
}
