\encoding{utf8}
\name{placePatch}
\alias{placePatch}
\title{Project semi-landmarks from a predefined atlas onto all specimen in a sample}
\usage{
placePatch(atlas, dat.array, path, prefix = NULL, fileext = ".ply",
  ray = TRUE, inflate = NULL, tol = inflate, relax.patch = TRUE,
  keep.fix = NULL, rhotol = NULL, silent = FALSE, mc.cores = 1)
}
\arguments{
  \item{atlas}{object of class "atlas" created by
  \code{\link{createAtlas}}}

  \item{dat.array}{k x 3 x n array containing reference
  landmarks of the sample or a matrix in case of only one
  target specimen.}

  \item{path}{character: specify the directory where the
  surface meshes of the sample are stored.}

  \item{prefix}{character: prefix to the specimens names
  (stored in \code{dimnames(dat.array)[[3]]}) to match the
  corresponding file names. If \code{dat.array} has no
  dimnames (e.g. because it is a matrix - see example
  below), this can also be a character vector containing
  the filenames to which \code{fileext} will be appended.}

  \item{fileext}{character: file extension of the surface
  meshes.}

  \item{ray}{logical: projection will be along surface
  normals instead of simple closest point search.}

  \item{inflate}{inflate (or deflate - if negative sign)
  the semilandmarks along the normals of the deformed atlas
  to make sure that they stay on the outside (inside) of
  the target mesh.}

  \item{tol}{numeric: threshold to follow the ray back
  after inflation. See details below. If no surface is hit
  after \code{tol} mm, the simple closest point will be
  used.}

  \item{relax.patch}{logical: request relaxation minimising
  bending energy toward the atlas.}

  \item{keep.fix}{integer: rowindices of those landmarks
  that are not allowed to be relaxed in case
  \code{relax.patch=TRUE}. If not specified, all landmarks
  will be kept fix.}

  \item{rhotol}{numeric: maximum amount of deviation a hit
  point's normal is allowed to deviate from the normal
  defined on the atlas. If \code{relax.patch=TRUE}, those
  points exceeding this value will be relaxed freely (i.e.
  not restricted to tangent plane).}

  \item{silent}{logical: suppress messages.}

  \item{mc.cores}{run in parallel (experimental stuff now
  even available on Windows). On windows this will only
  lead to a significant speed boost for many
  configurations, as all required packages (Morpho and
  Rvcg) need to be loaded by each newly spawned process.}
}
\value{
array containing the projected coordinates appended to the
data.array specified in the input. In case dat.array is a
matrix only a matrix is returned.
}
\description{
Project semi-landmarks from a predefined atlas onto all
specimen in a sample. Various mechanisms are implemented to
avoid errorneous placement on the wrong surface layer (e.g.
inside the bone).
}
\details{
This function allows the (relatively) easy projection of
surface points defined on an atlas onto all surface of a
given sample by Thin-Plate Spline deformation and
additional mechanisms to avoid distortions. The algorithm
can be outlined as followed.  \enumerate{ \item deform
atlas onto targets by TPS based on predefined landmarks
(and curves). \item project coordinates on deformed atlas
onto target mesh \item 'inflate' or 'deflate' configuration
along their normals to make sure all coordinates are on the
outside/inside \item Project inflated points back onto
surface along these normals. \item Check if normals are
roughly pointing into the same direction as those on the
(deformed) atlas. \item Relax all points against atlas.

}
}
\examples{
\dontrun{
data(nose)
require(rgl)
###create mesh for longnose
longnose.mesh <- warp.mesh(shortnose.mesh,shortnose.lm,longnose.lm)
## create atlas
fix <- c(1:5,20:21)
atlas <- createAtlas(shortnose.mesh, landmarks =
           shortnose.lm[fix,], patch=shortnose.lm[-c(1:5,20:21),])
## view atlas

plotAtlas(atlas)

## create landmark array with only fix landmarks
data <- bindArr(shortnose.lm[fix,], longnose.lm[fix,], along=3)
dimnames(data)[[3]] <- c("shortnose", "longnose")

### write meshes to disk
mesh2ply(shortnose.mesh, filename="shortnose")
mesh2ply(longnose.mesh, filename="longnose")

patched <- placePatch(atlas, data, path="./", inflate=5)
## now browse through placed patches
checkLM(patched, path="./", atlas=atlas)

## same example with only one target specimen
data <- longnose.lm[fix, ]

patched <- placePatch(atlas, data, prefix="longnose", path="./", inflate=5)
wire3d(longnose.mesh,col=3)
spheres3d(patched)
}
}
\author{
Stefan Schlager
}
\references{
Schlager S. 2013. Soft-tissue reconstruction of the human
nose: population differences and sexual dimorphism. PhD
thesis,
\enc{Universitätsbibliothek}{Universitaetsbibliothek}
Freiburg.  URL:
\url{http://www.freidok.uni-freiburg.de/volltexte/9181/}.
}
\seealso{
\code{\link{createAtlas}, \link{relaxLM}, \link{checkLM},
\link{slider3d}, \link{warp.mesh}}
}
\keyword{~kwd1}
\keyword{~kwd2}

