#' Gradient Boosting with Regression Trees
#' 
#' Gradient boosting for optimizing arbitrary loss functions where regression
#' trees are utilized as base-learners.
#' 
#' @param family optional \code{\link[mboost]{Family}} object.  Set
#'   automatically according to the class type of the response variable.
#' @param mstop number of initial boosting iterations.
#' @param nu step size or shrinkage parameter between 0 and 1.
#' @param risk method to use in computing the empirical risk for each boosting
#'   iteration.
#' @param stopintern logical inidicating whether the boosting algorithm stops
#'   internally when the out-of-bag risk increases at a subsequent iteration.
#' @param trace logical indicating whether status information is printed during
#'   the fitting process.
#' @param teststat type of the test statistic to be applied for variable
#'   selection.
#' @param testtype how to compute the distribution of the test statistic.
#' @param mincriterion value of the test statistic or 1 - p-value that must be
#'   exceeded in order to implement a split.
#' @param minsplit minimum sum of weights in a node in order to be considered
#'   for splitting.
#' @param minbucket minimum sum of weights in a terminal node.
#' @param maxdepth maximum depth of the tree.
#' @param saveinfo logical indicating whether to store information about
#'   variable selection in \code{info} slot of each \code{partynode}.
#' @param ... additional arguments to \code{\link[partykit]{ctree_control}}.
#' 
#' @details
#' \describe{
#'   \item{Response Types:}{\code{binary}, \code{numeric}, \code{Surv}}
#'   \item{\link[=TunedModel]{Automatic Tuning} of Grid Parameters:}{
#'     \code{mstop}, \code{maxdepth}
#'   }
#' }
#' 
#' Default values for the \code{NULL} arguments and further model details can be
#' found in the source links below.
#' 
#' @return \code{MLModel} class object.
#' 
#' @seealso \code{\link[mboost]{blackboost}}, \code{\link[mboost]{Family}},
#' \code{\link[partykit]{ctree_control}}, \code{\link{fit}},
#' \code{\link{resample}}, \code{\link{tune}}
#' 
#' @examples
#' library(MASS)
#' 
#' fit(type ~ ., data = Pima.tr, model = BlackBoostModel)
#'
BlackBoostModel <- function(family = NULL, mstop = 100, nu = 0.1,
                            risk = c("inbag", "oobag", "none"),
                            stopintern = FALSE, trace = FALSE,
                            teststat = c("quadratic", "maximum"),
                            testtype = c("Teststatistic", "Univariate",
                                         "Bonferroni", "MonteCarlo"),
                            mincriterion = 0, minsplit = 10, minbucket = 4,
                            maxdepth = 2, saveinfo = FALSE, ...) {
  
  teststat <- match.arg(teststat)
  testtype <- match.arg(testtype)
  
  args <- params(environment())
  is_main <- names(args) %in% "family"
  is_control <- names(args) %in% c("mstop", "nu", "risk", "stopintern", "trace")

  params <- args[is_main]
  params$control <- as.call(c(.(mboost::boost_control), args[is_control]))
  params$tree_controls <- as.call(c(.(partykit::ctree_control),
                                    args[!(is_main | is_control)]))

  MLModel(
    name = "BlackBoostModel",
    label = "Gradient Boosting with Regression Trees",
    packages = c("mboost", "partykit"),
    response_types = c("binary", "numeric", "Surv"),
    predictor_encoding = "terms",
    params = params,
    grid = function(x, length, ...) {
      list(
        mstop = round(seq_range(0, 50, c(1, 1000), length + 1)),
        maxdepth = 1:min(length, 10)
      )
    },
    fit = function(formula, data, weights, family = NULL, ...) {
      if (is.null(family)) {
        family <- switch_class(response(data),
                               "factor" = mboost::Binomial(),
                               "numeric" = mboost::Gaussian(),
                               "Surv" = mboost::CoxPH())
      }
      mboost::blackboost(formula, data = as.data.frame(data),
                         na.action = na.pass, weights = weights,
                         family = family, ...)
    },
    predict = function(object, newdata, times, ...) {
      newdata <- as.data.frame(newdata)
      if (object$family@name == "Cox Partial Likelihood") {
        y <- object$response
        lp <- drop(predict(object, type = "link"))
        new_lp <- drop(predict(object, newdata = newdata, type = "link"))
        predict(y, lp, times, new_lp, ...)
      } else {
        predict(object, newdata = newdata, type = "response")
      }
    },
    varimp = function(object, ...) {
      structure(mboost::varimp(object), class = "numeric")
    }
  )
  
}

MLModelFunction(BlackBoostModel) <- NULL
