#' Predictive function.
#' @description Method returning the predictive probability density in-sample or of a vector of points at \code{t = T + 1}.
#' @param object Model specification of class \code{MSGARCH_SPEC} created with \code{\link{create.spec}}
#' or fit object of type \code{MSGARCH_MLE_FIT} created with \code{\link{fit.mle}} or \code{MSGARCH_BAY_FIT}
#' created with \code{\link{fit.bayes}}.
#' @param x Vector (of size N) of point at \code{t = T + 1} to be evaluated (used when \code{do.its = FALSE}).
#' @param theta Vector (of size d) or matrix (of size M x d) of parameter estimates (not require when using a fit object).
#' @param y  Vector (of size T) of observations (not require when using a fit object).
#' @param log  Boolean indicating if the log-density is returned. (Default: \code{log = TRUE})
#' @param do.its  Boolean indicating if the in-sample predictive is returned. (Default: \code{do.its = FALSE})
#' @details If a matrix of MCMC posterior draws estimates is given, the Bayesian Probability integral transform is calculated.
#' If \code{do.its = FALSE}, the points \code{x} are evaluated as \code{t = T + 1} realization and the method uses the variance estimate at \code{t = T + 1}.
#' If \code{do.its = TRUE}, \code{y} is evaluated using their respective variance estimate at each time \code{t}.
#' @examples 
#'\dontrun{
#'# load data
#'data("sp500")
#'
#'# create model specification
#'spec = MSGARCH::create.spec() 
#'
#'# fit the model on the data with ML estimation using DEoptim intialization
#' set.seed(123)
#'fit = MSGARCH::fit.mle(spec = spec, y = sp500)
#'                            
#'# run pred method in-sample     
#'pred.its = MSGARCH::pred(object = fit, log = TRUE, do.its = TRUE)  
#'
#'sum(pred.its$pred, na.rm = TRUE)
#'                                              
#'# create mesh
#'x = seq(-3,3,0.01)
#'
#'# run pred method on mesh at T + 1
#'pred = MSGARCH::pred(object = fit, x = x, log = TRUE, do.its = FALSE)
#'
#'plot(pred)
#'}
#' @return A list of class \code{MSGARCH_PRED} containing two components:
#' \itemize{
#' \item \code{pred}:\cr If \code{do.its = FALSE}: (Log-)Predictive of of the points \code{x} at \code{t = T + 1} (vector of size N). \cr
#'                   If \code{do.its = TRUE}: In-sample Predictive of \code{y} (vector of size T or matrix of size M x T). 
#' \item \code{x}:\cr If \code{do.its = FALSE}: Vector (of size N) of point at \code{t = T + 1} evaluated.\cr
#'                 If \code{do.its = TRUE}: Vector (of size T) of observations.
#' }
#'The class \code{MSGARCH_PRED} contains the \code{plot} method only if \code{do.its = FALSE}.
#' @export
pred <- function(object, x, theta, y, log = TRUE, do.its = FALSE) {
  UseMethod("pred", object)
}

#' @export
pred.MSGARCH_SPEC <- function(object, x = NULL, theta, y, log = TRUE, do.its = FALSE) {
  y <- f.check.y(y)
  N <- nrow(theta)
  if (isTRUE(do.its)) {
    nx <- length(y)
    x <- y
  } else {
    nx <- length(x)
  }
  tmp <- matrix(data = NA, nrow = N, ncol = nx)
  for (i in 1:N) {
    tmp[i, ] <- MSGARCH::pdf(object, x, theta = theta[i, ], y = y, log = FALSE,
                            do.its = do.its)$pdf
  }
  tmp <- colMeans(tmp)
  if (log) {
    tmp <- log(tmp)
  }
  out <- list()
  out$pred <- tmp
  out$x <- x
  out$do.its <- do.its
  class(out) <- "MSGARCH_PRED"
  return(out)
}

#' @export
pred.MSGARCH_MLE_FIT <- function(object, x = NULL, theta = NULL, y = NULL, log = TRUE,
                                do.its = FALSE) {
  return(MSGARCH::pred(object = object$spec, x = x, theta = object$theta, y = object$y,
                       log = log, do.its = do.its))
}

#' @export
pred.MSGARCH_BAY_FIT <- function(object, x = NULL, theta = NULL, y = NULL, log = TRUE,
                                do.its = FALSE) {
  return(MSGARCH::pred(object = object$spec, x = x, theta = object$theta, y = object$y,
                       log = log, do.its = do.its))
}