% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Mainfun.plot.R
\name{MFggplot}
\alias{MFggplot}
\title{ggplot2 extension for a MF1_single or MF2_multiple object}
\usage{
MFggplot(output, model = "LMM.both", caption = "slope", by_group = NULL)
}
\arguments{
\item{output}{the output obtained from \code{MF1_single} or \code{MF2_multiple}. \cr
For output obtained from \code{MF1_single}, if BEF relationships are desired within each category specified \code{by_group}, the \code{by_group} column must be included in the input.}

\item{model}{specifying the fitting model, \code{model = "lm"} for linear model; \code{model = "LMM.intercept"},
\code{"LMM.slope"} and \code{"LMM.both"} for linear mixed models with random effects for intercepts, slopes, and both, respectively.
Default is \code{model = "LMM.both"}. \cr
If \code{output} is obtained from \code{MF2_multiple} with \code{by_pair = FALSE}, \code{model} can only set as \code{"lm"}.}

\item{caption}{caption that will be shown in the BEF plots; \code{caption = "slope"} to show the estimated slopes in each plot,
or \code{caption = "R.squared"} to show the ordinary R-squared for linear models or estimated marginal and conditional R-squared for linear mixed models in each plot.
Default is \code{caption = "slope"}.}

\item{by_group}{the column name of the stratifying variable that is used to group data for model fitting. For example, if \code{by_group = "country"}, then model will be fitted within each country. Default is \code{NULL}. \cr
It is required if a linear mixed model is selected in the \code{model}. \cr
If \code{output} is obtained from \code{MF2_multiple} with \code{by_pair = TRUE}, the \code{by_group} setting must be the same as that set in \code{MF2_multiple}. \cr
If \code{output} is obtained from \code{MF2_multiple} with \code{by_pair = FALSE}, the \code{by_group} must be \code{NULL}, because alpha/beta/gamma data points are not sufficient to fit model in each group.}
}
\value{
For an \code{MF1_single} object, this function returns a figure that plots the BEF relationship between multifunctionality of
order q (= 0, 1 and 2) and species diversity of the same order q for two cases (i) correlations between functions are not corrected for, and (ii) correlations between
functions are corrected. The fitted lines for the chosen model are also shown in the figure. \cr

For an \code{MF2_multiple} object of given individual function weights, this function returns a list of two objects (\code{$corr_uncorrected} and \code{$corr_corrected})
that respectively for two cases: (i) correlations between functions are not corrected for, and (ii) correlations between functions are corrected for. \cr

Each object consists of four figures: "\code{$ALL}" returns a figure that depicts the BEF relationship between alpha/beta/gamma multifunctionality of
order q (= 0, 1 and 2) and the corresponding species diversity of the same order q. The fitted lines for the chosen model are also shown in the figure.
"\code{$Gamma}" returns only the gamma part of "\code{$ALL}", "\code{$Alpha}" returns only the alpha part of "\code{$ALL}", and "\code{$Beta}" returns only the beta part of "\code{$ALL}".
}
\description{
\code{MFggplot} provides graphical BEF relationships based on the output from the function \code{MF1_single} or \code{MF2_multiple}.
}
\examples{

library(dplyr)

### Use the entire of 209 plots data from six countries

\donttest{
## within-ecosystem MF computation
data("forest_function_data_normalized")
data("forest_biodiversity_data")
output1 <- MF1_single(func_data = forest_function_data_normalized[,6:31], weight = 1,
                      species_data = forest_biodiversity_data)
  
## BEF relationships: display fitted line of linear mixed model with random slopes and 
## random intercepts  
output1 <- data.frame(output1, country=rep(forest_function_data_normalized$country, each = 6))
MFggplot(output1, model = "LMM.both", by_group="country", caption = "slope")


### Use partial data to quickly obtain output  
### (Take the first 18 plots in Germany and the last 18 plots in Italy)

## within-ecosystem MF computation for partial data
data("forest_function_data_raw")
data("forest_biodiversity_data")
GER_ITA_forest_function_raw <- filter(forest_function_data_raw, 
                                      country=="GER"|country=="ITA")[c(1:18,57:74),]
GER_ITA_forest_function_normalized <- function_normalization(data = GER_ITA_forest_function_raw,
                                                             fun_cols = 6:31, 
                                                             negative = c("soil_cn_ff_10","wue"),
                                                             by_group = "country")
GER_ITA_forest_biodiversity <- forest_biodiversity_data[c(49:82,181:229),]
output2 <- MF1_single(func_data = GER_ITA_forest_function_normalized[,6:31], weight = 1,
                      species_data = GER_ITA_forest_biodiversity)


## BEF relationships: display fitted line of linear mixed model with random slopes and 
## random intercepts  
output2 <- data.frame(output2, country=rep(GER_ITA_forest_function_normalized$country, each = 6))
MFggplot(output2, model = "LMM.both", by_group="country", caption = "slope")


### Use data from plots in five countries (data in Finland are excluded)
 
## MF decomposition for all pairs of ecosystems
data("forest_function_data_normalized")
data("forest_biodiversity_data")
forest_function_data_normalized <- filter(forest_function_data_normalized, country != "FIN")
forest_biodiversity_data <- forest_biodiversity_data[-(1:48),]
output3 <- MF2_multiple(func_data = forest_function_data_normalized[,6:32],
                        species_data = forest_biodiversity_data,
                        weight = 1,
                        by_group = "country")

## BEF relationships: display fitted line of linear mixed model with random slopes and 
## random intercepts  
figure_LMM <- MFggplot(output3, model = "LMM.both", by_group = "country", 
                       caption = "slope")
figure_LMM$corr_uncorrected$ALL
figure_LMM$corr_corrected$ALL
}

\donttest{
### Use partial data to quickly obtain output  
### (Take the first 18 plots in Germany and the last 18 plots in Italy)

## MF decomposition for all pairs of ecosystems for partial data
data("forest_function_data_raw")
data("forest_biodiversity_data")
GER_ITA_forest_function_raw <- filter(forest_function_data_raw, 
                                      country=="GER"|country=="ITA")[c(1:18,57:74),]
GER_ITA_forest_function_normalized <- function_normalization(data = GER_ITA_forest_function_raw,
                                                             fun_cols = 6:31, 
                                                             negative = c("soil_cn_ff_10","wue"),
                                                             by_group = "country")
GER_ITA_forest_biodiversity <- forest_biodiversity_data[c(49:82,181:229),]
output4 <- MF2_multiple(func_data = GER_ITA_forest_function_normalized[,6:32],
                        species_data = GER_ITA_forest_biodiversity,
                        weight = 1,
                        by_group = "country")


## BEF relationships: display fitted line of linear mixed model with random slopes and 
## random intercepts for partial data 
figure_LMM_GER_ITA <- MFggplot(output4, model = "LMM.both", by_group = "country", 
                               caption = "slope")
figure_LMM_GER_ITA$corr_uncorrected$ALL
figure_LMM_GER_ITA$corr_corrected$ALL
}


### Use partial data to calculate multifunctionality based on 3 plots in each country, not by pairs 
### (Take the first 3 plots in each country)

data("forest_function_data_raw")
data("forest_biodiversity_data")

forest_function_data_raw_3plots <- forest_function_data_raw[c(1:3,29:31,67:69,103:105,
                                                                      146:148,174:176),]
forest_function_data_normalized_3plots <- 
                    function_normalization(data = forest_function_data_raw_3plots,
                    fun_cols = 6:31, 
                    negative = c("soil_cn_ff_10","wue"),
                    by_group = "country")
forest_biodiversity_data_3plots <- 
                   forest_biodiversity_data[c(1:6,49:52,141:148,230:232,351:355,411:417),]

output5 <- MF2_multiple(func_data = forest_function_data_normalized_3plots[,6:32],
                        species_data = forest_biodiversity_data_3plots,
                        weight = 1,
                        by_group = "country", by_pair = FALSE)
                        
## BEF relationships: display fitted line of linear model  
figure_all_plots <- MFggplot(output5, model = "lm", caption = "slope")                                
figure_all_plots$corr_uncorrected$ALL
figure_all_plots$corr_corrected$ALL

}
