## Copyright 2011-2012 Sebastian Gibb
## <mail@sebastiangibb.de>
##
## This file is part of MALDIquant for R and related languages.
##
## MALDIquant is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## MALDIquant is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with MALDIquant. If not, see <http://www.gnu.org/licenses/>

setMethod(f="labelPeaks",
    signature=signature(object="MassPeaks"),
    definition=function(object, 
        index,
        mass, tolerance=0.002,
        digits=3,
        underline=TRUE, 
        ## verticalOffset ca. 0.0125 of plot height
        verticalOffset=abs(diff(par("usr")[3:4]))*0.0125,
        absoluteVerticalPos,
        adj=c(0.5, 0), cex=0.7, family="sans",
        ...) {

    if (!missing(mass) && is.numeric(mass)) {
        massIndex <- unlist(sapply(mass, function(x) {
            return(.which.nearby(object@mass, x, tolerance=tolerance));
        }));

        if (!all(!is.na(massIndex))) {
            warning("No peak(s) found at mass: ",
                    paste(mass[is.na(massIndex)], collapse=", "));
            massIndex <- massIndex[!is.na(massIndex)];
        }

        if (!missing(index)) {
            if (is.logical(index)) {
                warning("Could not handle a logical ", sQuote("index"), 
                        " and a numeric ", sQuote("mass"), " vector. ",
                        "Replacing ", sQuote("index"), " by ", sQuote("mass"),
                        ".");
                index <- massIndex;
            } else {
                index <- c(index, massIndex);
            }
        } else {
            index <- massIndex;
        }
        
        ## remove duplicated indices
        index <- unique(index);
    } else if (missing(mass) && missing(index)) {
        index <- 1:length(object@mass);
    }

    isValidIndex <- length(index) >= 1 && 
                    length(index) <= length(object@mass) && 
                    ((min(index) >= 1 && max(index) <= length(object@mass)) ||
                      is.logical(index));
    if (!isValidIndex) {
        stop("No valid ", sQuote("index"), " nor ", sQuote("mass"), " given.");
    }

    x <- object@mass[index];

    if (missing(absoluteVerticalPos)) {
        y <- object@intensity[index]+verticalOffset;
    } else {
        y <- absoluteVerticalPos;
    }

    peakLabels <- round(x=x, digits=digits);

    if (underline) {
        peakLabels <- as.expression(sapply(peakLabels,
                function(x)substitute(underline(a), list(a=x))));
    }

    text(x=x, y=y, labels=peakLabels, cex=cex, adj=adj, ...);
});

