% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/anova.R
\name{anova}
\alias{anova}
\alias{anova.lmm}
\title{Multivariate Wald Tests For Linear Mixed Model}
\usage{
\method{anova}{lmm}(
  object,
  effects = NULL,
  robust = FALSE,
  rhs = NULL,
  df = !is.null(object$df),
  ci = TRUE,
  transform.sigma = NULL,
  transform.k = NULL,
  transform.rho = NULL,
  transform.names = TRUE,
  ...
)
}
\arguments{
\item{object}{a \code{lmm} object. Only relevant for the anova function.}

\item{effects}{[character] Should the Wald test be computed for all variables (\code{"all"}),
or only variables relative to the mean (\code{"mean"} or \code{"fixed"}),
or only variables relative to the variance structure (\code{"variance"}),
or only variables relative to the correlation structure (\code{"correlation"}).
Can also be use to specify linear combinations of coefficients, similarly to the \code{linfct} argument of the \code{multcomp::glht} function.}

\item{robust}{[logical] Should robust standard errors (aka sandwich estimator) be output instead of the model-based standard errors.}

\item{rhs}{[numeric vector] the right hand side of the hypothesis. Only used when the argument effects is a matrix.}

\item{df}{[logical] Should a F-distribution be used to model the distribution of the Wald statistic. Otherwise a chi-squared distribution is used.}

\item{ci}{[logical] Should a confidence interval be output for each hypothesis?}

\item{transform.sigma, transform.k, transform.rho, transform.names}{are passed to the \code{vcov} method. See details section in \code{\link{coef.lmm}}.}

\item{...}{Not used. For compatibility with the generic method.}
}
\value{
A list of matrices containing the following columns:\itemize{
\item \code{null}: null hypothesis
\item \code{statistic}: value of the test statistic
\item \code{df.num}: degrees of freedom for the numerator (i.e. number of hypotheses)
\item \code{df.denom}: degrees of freedom for the denominator (i.e. Satterthwaite approximation)
\item \code{p.value}: p-value.
}
as well as an attribute contrast containing the contrast matrix encoding the linear combinations of coefficients (in columns) for each hypothesis (in rows).
}
\description{
Simultaneous tests of linear combinations of the model paramaters using Wald tests.
}
\details{
By default confidence intervals and p-values are adjusted based on the distribution of the maximum-statistic.
This is refered to as a single-step Dunnett multiple testing procedures in table II of Dmitrienko et al. (2013) and is performed using the multcomp package with the option \code{test = adjusted("single-step")}.
}
\examples{
## simulate data in the long format
set.seed(10)
dL <- sampleRem(100, n.times = 3, format = "long")

## fit Linear Mixed Model
eUN.lmm <- lmm(Y ~ X1 + X2 + X5, repetition = ~visit|id, structure = "UN", data = dL)

## chi-2 test
summary(anova(eUN.lmm, df = FALSE))

## F-test
anova(eUN.lmm)
summary(anova(eUN.lmm, effects = "all"))
anova(eUN.lmm, effects = c("X1=0","X2+X5=10"))

## another example
if(require(multcomp)){
amod <- lmm(breaks ~ tension, data = warpbreaks)
e.glht <- glht(amod, linfct = mcp(tension = "Tukey"))
summary(e.glht, test = Chisqtest()) ## 0.000742

e.amod <- anova(amod, effect = mcp(tension = "Tukey"))
summary(e.amod)


anova(amod, effect = mcp(tension = "Tukey"), ci = TRUE)
}
}
\references{
Dmitrienko, A. and D'Agostino, R., Sr (2013), Traditional multiplicity adjustment methods in clinical trials. Statist. Med., 32: 5172-5218. https://doi.org/10.1002/sim.5990.
}
\seealso{
\code{\link{summary.anova_lmm}} for a summary of the results. \cr
}
