% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/isosim.R
\name{isosim}
\alias{isosim}
\title{Simulate isotopic values}
\usage{
isosim(data, mean_model_fix_coef = c(intercept = 64, elev = -0.01,
  lat_abs = -2.3, lat_2 = 0, long = 0, long_2 = 0),
  disp_model_fix_coef = c(intercept = 5.8, elev = 0, lat_abs = 0, lat_2 =
  0, long = 0, long_2 = 0), mean_model_matern_coef = c(nu = 0.35, rho =
  5e-05, lambda = 899), disp_model_matern_coef = c(nu = 0.32, rho =
  1.5e-05, lambda = 5), mean_model_uncorr_coef = c(nugget = 0, lambda =
  0), disp_model_uncorr_coef = c(nugget = 0, lambda = 0),
  dist_method = "Earth", seed = NULL, save_dataframe = FALSE,
  verbose = interactive())
}
\arguments{
\item{data}{A \var{data.frame} containing the covariates needed for the
simulation, or alternatively a structural raster of class \var{RasterLayer}}

\item{mean_model_fix_coef}{A \var{vector} of coefficients for fixed-effects}

\item{disp_model_fix_coef}{A \var{vector} of coefficients for fixed-effects}

\item{mean_model_matern_coef}{A \var{vector} of coefficients for the spatial
random effect}

\item{disp_model_matern_coef}{A \var{vector} of coefficients for the spatial
random effect}

\item{mean_model_uncorr_coef}{A \var{vector} of coefficients for the
uncorrelated random effect}

\item{disp_model_uncorr_coef}{A \var{vector} of coefficients for the
uncorrelated random effect}

\item{dist_method}{A \var{string} indicating the distance method}

\item{seed}{An \var{integer} used to set the seed of the random generator}

\item{save_dataframe}{A \var{logical} indicating whether the detailed
\var{data.frame} containing the simulated data should be saved}

\item{verbose}{A \var{logical} indicating whether information about the
progress of the procedure should be displayed or not while the function is
running. By default verbose is \var{TRUE} if users use an interactive R
session and \var{FALSE} otherwise.}
}
\value{
This function returns a \var{list} of class \var{ISOSCAPE}
containing a set of raster and an optional \var{data.frame}. The set
contains the raster \code{mean_raster} storing the mean isotopic value, and
the raster \cr \code{disp_raster} storing the residual dispersion variance.
The optional \var{data.frame} contains the simulated data and details of the
computation in an object called \code{data}.
}
\description{
This function allows for the simulation of isoscapes. Both partial or
complete (i.e. maps) isoscapes can be simulated.
}
\details{
This function takes as inputs the values for all covariates matching a
series of locations (which can be provided as an structural raster or as a
\var{data.frame}), as well as the parameters of the isoscape model. The
function is not required to fit an isoscape, nor to perform assignments. It
is an additional function that can be useful to test the method, and to
study the effect of different parameters on isoscapes. We chose default
values for parameters inspired to model fits in order to simulate a
relatively realistic isoscape. We say 'relatively' because, in the
simulations, the Matérn realizations for the mean and the dispersion are
drawn independently, which is not the case in nature. Note that extra
parameters present in the input lists will not make the function crash but
won't be considered during computations either.
}
\note{
The spatial autocorrelation and the Nugget are computed by the
functions \code{\link[RandomFields:RMmatern]{RMwhittle}} and
\code{\link[RandomFields]{RMnugget}}, respectively. These two functions are
part of the powerful package \pkg{\link[RandomFields]{RandomFields}}.
}
\examples{


## The examples below will only be run if sufficient time is allowed
## You can change that by typing e.g. options_IsoriX(example_maxtime = XX)
## if you want to allow for examples taking up to ca. XX seconds to run
## (so don't write XX but put a number instead!)

if(getOption_IsoriX("example_maxtime") > 60) {

## We simulate data under default settings
Simu <- isosim(data = ElevRasterDE,
               save_dataframe = TRUE,
               seed = 1)

Simu

## We build the plots of the outcome using IsoriX
PlotMeanSimu <- plot(x = Simu, which = "mean")

PlotDispSimu <- plot(x = Simu, which = "disp")


## We fit the simulated data by sampling 50 locations

set.seed(123)
Newdat <- Simu$data[sample(1:nrow(Simu$data), 50), ]

NewdatFit <- isofit(data = Newdat,
                    mean_model_fix = list(elev = TRUE, lat_abs = TRUE))

Isoscape <- isoscape(ElevRasterDE, NewdatFit)

PlotMeanFitted <- plot(x = Isoscape, which = "mean", sources = list(draw = FALSE))

PlotDispFitted <- plot(x = Isoscape, which = "disp", sources = list(draw = FALSE))

## We compare simulated and fitted data visually
print(PlotMeanSimu, split = c(1, 1, 2, 2), more = TRUE)
print(PlotDispSimu, split = c(2, 1, 2, 2), more = TRUE)
print(PlotMeanFitted, split = c(1, 2, 2, 2), more = TRUE)
print(PlotDispFitted, split = c(2, 2, 2, 2), more = FALSE)

## We compare simulated and fitted data numerically
## Note that differences are expected, as the geographic
##   area is small compared to the scale at which
##   spatial auto-correlation occurs
round(cbind(
    simulated = c(
        intercept = 64,
        lat_abs = -2.3, 
        elev = -0.01,
        nu = 0.35,
        rho = 5e-5,
        rho_div_nu  =  5e-5/0.35,
        lambda_ID = 0,
        lambda_matern = 899,
        intercept_disp = 5.8,
        nu_disp = 3.2e-01,
        rho_disp = 1.5e-05,
        lambda_matern_source_ID = 0,
        lambda_matern_disp = 5),
    fitted = c(
        intercept = NewdatFit$mean_fit$fixef[1],
        lat.abs = NewdatFit$mean_fit$fixef[2], 
        elev = NewdatFit$mean_fit$fixef[3],
        nu = get_ranPars(NewdatFit$mean_fit, which = "corrPars")[[1]]$nu,
        rho = get_ranPars(NewdatFit$mean_fit, which = "corrPars")[[1]]$rho,
        rho_div_nu = with(get_ranPars(NewdatFit$mean_fit, which = "corrPars")[[1]],rho/nu),
        lambda.matern = NewdatFit$mean_fit$lambda,
        intercept.disp = NewdatFit$disp_fit$fixef[1],
        nu.disp = get_ranPars(NewdatFit$disp_fit, which = "corrPars")[[1]]$nu,
        rho.disp = get_ranPars(NewdatFit$disp_fit, which = "corrPars")[[1]]$rho,
        lambda.matern.disp = NewdatFit$disp_fit$lambda)), 4)

}

}
\seealso{
\code{\link{isofit}} for the function fitting the isoscape model

\code{\link{IsoriX}} for the complete work-flow
}
\keyword{simulate}
\keyword{simulation}
