% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/3-UserMinimaxFunctions.R
\name{minimax}
\alias{minimax}
\title{Minimax and Standardized Maximin D-Optimal Designs}
\usage{
minimax(formula, predvars, parvars, family = gaussian(), lx, ux, lp, up,
  iter, k, n.grid = 0, fimfunc = NULL, ICA.control = list(),
  sens.minimax.control = list(), crt.minimax.control = list(),
  standardized = FALSE, initial = NULL, localdes = NULL,
  npar = length(lp), plot_3d = c("lattice", "rgl"))
}
\arguments{
\item{formula}{A nonlinear model \code{\link[stats]{formula}}.
A symbolic description of the model consists of predictors and the unknown model parameters.
Will be coerced to a \code{\link[stats]{formula}} if necessary.}

\item{predvars}{A vector of characters. Denotes the predictors in the \code{\link[stats]{formula}}.}

\item{parvars}{A vector of characters. Denotes the unknown parameters in the \code{\link[stats]{formula}}.}

\item{family}{A description of the response distribution and link function to be used in the model.
This can be a family function, a call to a family function or a character string naming the family.
 Every family function has a link argument allowing to specify the link function to be applied on the response variable.
  If not specified, default links are used. For details see \code{\link[stats]{family}}.
   By default, a linear gaussian model \code{gaussian()} is applied.}

\item{lx}{Vector of lower bounds for the predictors. Should be in the same order as \code{predvars}.}

\item{ux}{Vector of upper bounds for the predictors. Should be in the same order as \code{predvars}.}

\item{lp}{Vector of lower bounds for the model parameters. Should be in the same order as \code{parvars} or \code{param} in the argument \code{fimfunc}.}

\item{up}{Vector of upper bounds for the model parameters. Should be in the same order as \code{parvars} or \code{param} in the argument \code{fimfunc}.
When a parameter is known (has a fixed value), set its associated lower and upper bounds  in \code{lp} and \code{up}  equal to the known value.}

\item{iter}{Maximum number of iterations.}

\item{k}{Number of design points. Must be at least equal to the number of model parameters to avoid singularity of the FIM.}

\item{n.grid}{The total number of grid points from the parameter space is \code{n.grid^p}.
Used to map the parameter space to a \strong{discrete} grid. Disabled when \code{n.grid = 0} (default).
When set, optimal design protects the experimenter against the worst scenario over the grid points.
    See 'Details' of \code{\link{minimax}}.}

\item{fimfunc}{A function. Returns the FIM as a \code{matrix}. Required when \code{formula} is missing. See 'Details' of \code{\link{minimax}}.}

\item{ICA.control}{ICA control parameters. For details, see \code{\link{ICA.control}}.}

\item{sens.minimax.control}{Control parameters to verify the general equivalence theorem. For details, see the function \code{\link{sens.minimax.control}}.}

\item{crt.minimax.control}{Control parameters to optimize the minimax or standardized maximin criterion at a given design over a \strong{continuous} parameter space (when \code{n.grid = 0}).
For details, see the function \code{\link{crt.minimax.control}}.}

\item{standardized}{Maximin standardized design? When \code{standardized = TRUE}, the argument \code{localdes} must be given.
Defaults to \code{FALSE}. See 'Details' of \code{\link{minimax}}.}

\item{initial}{A matrix of the  initial designs that will be used as initial solutions (countries).
Every row is a design, i.e.  a concatenation of \code{x} and \code{w}. Will be coerced to a \code{matrix} if necessary.  See 'Details' of \code{\link{minimax}}.}

\item{localdes}{A function that takes the parameter values  as inputs and returns the design points and weights of the locally optimal design.
Required when \code{standardized = "TRUE"}. See 'Details' of \code{\link{minimax}}.}

\item{npar}{Number of model parameters.  Used when \code{fimfunc} is given instead of \code{formula} to specify the number of model parameters.
If not specified truly, the sensitivity (derivative) plot may be shifted below the y-axis. When \code{NULL}, it will be set here to \code{length(lp)}.}

\item{plot_3d}{Which package should be used to plot the sensitivity (derivative) function for two-dimensional design space. Defaults to \code{"lattice"}.}
}
\value{
an object of class \code{minimax} that is a list including three sub-lists:
\describe{
  \item{\code{arg}}{A list of design and algorithm parameters.}
  \item{\code{evol}}{A list of length equal to the number of iterations that stores
   the information about the best design (design with least criterion value)
    of each iteration. \code{evol[[iter]]} contains:
    \tabular{lll}{
      \code{iter}                   \tab      \tab Iteration number.\cr
      \code{x}                      \tab      \tab Design points. \cr
      \code{w}                      \tab      \tab Design weights. \cr
      \code{min_cost}               \tab      \tab Cost (criterion value) of the best imperialist in iteration \code{iter}. \cr
      \code{mean_cost}              \tab      \tab Mean of costs of all imperialists in iteration \code{iter}. \cr
      \code{sens}                   \tab      \tab An object of class 'sensminimax'. See below. \cr
      \code{param}                  \tab      \tab Vector of parameters.\cr
    }
  }

  \item{\code{empires}}{A list of all empires of the last iteration.}
  \item{\code{alg}}{A list with following information:
    \tabular{lll}{
      \code{nfeval}           \tab      \tab Number of function evaluations. See below. \cr
      \code{nlocal}           \tab      \tab Number of successful local search. \cr
      \code{nrevol}           \tab      \tab Number of successful revolutions. \cr
      \code{nimprove}         \tab      \tab Number of successful movements toward the imperialists in assimilation step. \cr
      \code{convergence}      \tab      \tab Stopped by \code{'maxiter'} or \code{'equivalence'}?\cr
    }
  }
}

\code{sens} contains information about the design verification by the general equivalence theorem. See \code{sensminimax} for more details.
It is only available every \code{ICA.control$checkfreq} iterations
and the last iteration if   \code{ICA.control$checkfreq >= 0}. Otherwise, \code{NULL}.

 \code{param} is a vector of parameters that is the global minimum of
  the minimax criterion or the global maximum of the standardized maximin criterion over the parameter space, given \code{x}, \code{w}.

\code{nfeval} does not count the function evaluations from checking the general equivalence theorem.
}
\description{
Finds minimax and standardized maximin D-optimal designs for nonlinear models.
 It should be used when the user assumes each of the unknown parameters belongs to an uncertainty interval (or parameter interval)
  and the purpose is to \strong{protect the experiment from the worst scenario}
  over the space of the Cartesian product of the given uncertainty intervals, denoted by \eqn{\Theta}.
  Sometimes, \eqn{\Theta} is  called 'region of uncertainty', that is the parameter space.

If you are looking for a design
   that is \strong{in average} robust with respect to the parameter space, then you should apply
   Bayesian optimal designs (continuous parameter space) or robust designs (discrete parameter space). See,  functions \code{\link{bayes}} and \code{\link{robust}}.\cr

  In general, although standardized criteria have some good theoretical features,
   we recommend applying them \strong{only}
  when the user can provide a closed-form for
  locally D-optimal designs for the model of interest by the argument \code{localdes}.
}
\details{
Let \eqn{\Xi} be the space of all  approximate designs with
 \eqn{k} design points (support points) at \eqn{x_1, x_2, ...,  x_k} from  design space \eqn{\chi} with
 corresponding weights  \eqn{w_1, . . . ,w_k}.
 Let \eqn{M(\xi, \theta)} be the Fisher information
  matrix (FIM) of a \eqn{k-}point design \eqn{\xi} and \eqn{\theta} be the vector of unknown parameters.\cr
 A  minimax D-optimal design \eqn{\xi^*}{\xi*} minimizes over \eqn{\Xi}
  \deqn{\max_{\theta \in \Theta} -\log|M(\xi, \theta)|.}{
   max over \Theta -log|M(\xi, \theta)|.}

 A standardized maximin D-optimal design \eqn{\xi^*}{\xi*} maximizes over \eqn{\Xi}
 \deqn{\inf_{\theta \in \Theta} \left[\left(\frac{|M(\xi, \theta)|}{|M(\xi_{\theta}, \theta)|}\right)^\frac{1}{p}\right],}{
  inf over \Theta {|M(\xi, \theta)| / |M(\xi_\theta, \theta)|}^p,}
  where \eqn{p} is the number of model parameters and \eqn{\xi_\theta} is the locally D-optimal design with respect to \eqn{\theta}.\cr

A minimax criterion (cost function or objective function) is evaluated at each design (decision variables) by maximizing the criterion over the parameter space.
We call the optimization problem over the parameter space as \emph{inner optimization problem}.
Here, two separate strategies may be
 applied to solve the inner problem at a given design (design points and weights):
\enumerate{
\item \strong{Continuous inner problem}: optimizes the criterion over a continuous parameter space. We use function \code{\link[nloptr]{nloptr}} for this purpose.
The tuning parameters can be regulated via the argument \code{\link{crt.minimax.control}}. The most important control parameter is \strong{\code{maxeval}} that requires especial attention.
The output design here protects the experiment from the worst scenario  over
the continuous parameter space.
\item  \strong{Discrete inner problem}: maps the parameter space to some grid points and optimizes the criterion over a discrete parameter space
(the inner optimization problem becomes a discrete space).
In this case, the number of grid points can be regulated via \code{n.grid}.
This strategy is quite efficient (ans fast) when  the maxima most likely attain the vertices of the continuous parameter space at any given design.
The output design here protects the experiment from the worst scenario
over the grid points.
}

The \code{formula} is used to create the Fisher information matrix (FIM) for a nonlinear model when the distribution of the response variable belongs to the natural exponential family.
User can also apply the argument  \code{fimfunc} to provide the FIM her/himself, directly.
 In this case,
argument \code{fimfunc} takes a \code{function} that has three arguments as follows:
 \enumerate{
  \item \code{x} a vector of design points. For design points with more than one dimension,
   it is a concatenation of the design points, but \strong{dimension-wise}.
   For example, let the model has three predictors   \eqn{(I, S, Z)}.
    Then,  (three-dimensional) design points of a two-point optimal design are
   \eqn{\{\mbox{point1} = (I_1, S_1, Z_1), \mbox{point2} = (I_2, S_2, Z_2)\}}{{point1 = (I1, S1, Z1), point2 = (I2, S2, Z2)}}.
    Then, the argument \code{x} is equivalent to
    \code{x = c(I1, I2, S1, S2, Z1, Z2)}.
  \item \code{w} a vector that includes the design weights associated with \code{x}.
  \item \code{param} a vector of parameter values associated with \code{lp} and \code{up}.
}
 The output must be the Fisher information matrix with number of rows equal to \code{length(param)}. See 'Examples'.



 Minimax optimal designs can have very different criterion values depending on the nominal set of parameter values.
 Accordingly, it is desirable to standardize the criterion and control for the potentially widely varying magnitude of the criterion (Dette, 1997).
 Evaluating a standardized maximin criterion requires knowing locally optimal designs.
We strongly advise setting \code{standardized = 'TRUE'}, only when analytical solutions for the locally D-optimal designs is available.
When \code{standardized = 'TRUE'}, the user must  provide the closed-form of the locally optimal design (support points \code{x} and weights \code{w}) as a function of the model parameters.\cr
\code{localdes} takes a function with the following arguments.
\itemize{
\item If \code{formula} is given (\code{!missing(formula)}):
    \itemize{
    \item The parameter names given by \code{parvars} in the same order.

    }
\item FIM is given by the argument \code{fimfunc} (\code{missing(formula)}):
   \itemize{
   \item \code{param}: A vector of the parameters same as the argument \code{param} in \code{fimfunc}.
    }
}
The function must return a list with components \code{x} and \code{w} (they match the same arguments in the function \code{fimfunc}).
  See 'Examples'.\cr
The standardized D-criterion is equal to the  D-efficiency and it must be between 0 and 1, theoretically.
 However, in practice, it may take values larger than one because the function given via \code{localdes} does not
  return the true (accurate) locally optimal designs for some requested parameter values from \eqn{\Theta}.
 In this case, the function \code{minimax}
stops and returns a set of values for the model parameters.
 The user here must investigate the accuracy of the locally D-optimal designs by
  checking the output of the function set via the argument \code{localdes} given this set of parameter values.




Each row of \code{initial} is one design, i.e. a concatenation of values for design (support) points  and the associated design weights.
Let \code{x0} and \code{w0} be the vector of initial values with exactly the same length and order as \code{x} and \code{w} (the arguments of \code{fimfunc}).
 As an example, the first row  of the matrix \code{initial} is equal to \code{initial[1, ] = c(x0, w0)}.
  For models with more than one predictors, \code{x0} is a concatenation of the initial values for design points, but \strong{dimension-wise}.
  See the details of the argument \code{fimfunc}, above.

 To verify the optimality of the output design by the general equivalence theorem,
 the user can either \code{plot} the results or set  \code{checkfreq} in \code{\link{ICA.control}}
 to \code{Inf}. In either case, the function \code{\link{sensminimax}} is called for verification.
  Note that  the function \code{\link{sensminimax}} always verifies the optimality of a design assuming a continues parameter space.
See 'Examples'.
}
\note{
For larger parameter space or model with more number of unknown parameters,
 it is always important to increase the value of  \code{ncount} in \code{ICA.control}
and \code{optslist$maxeval} in \code{crt.minimax.control} to produce very accurate designs.
}
\examples{
########################################
# Two-parameter exponential growth model
########################################
res1 <- minimax (formula = ~a + exp(-b*x), predvars = "x", parvars = c("a", "b"),
                 lx = 0, ux = 1, lp = c(1, 1), up = c(1, 10),
                 iter = 1, k = 4, ICA.control= ICA.control(rseed = 100))
\dontrun{
  res1 <- iterate(res1, 150)
  # iterating the algorithm up to 150 more iterations
}

res1 # print method
plot(res1) # veryfying the general equivalence theorem

########################################
# Two-parameter logistic model.
########################################
# A little bit tickling with the tuning parameters
# reducing the value of maxeval to 200 to increase the speed
cont1 <- crt.minimax.control(optslist = list(maxeval = 500))
cont2 <- ICA.control(rseed = 100, checkfreq = Inf, ncount = 60)

\dontrun{
  res2 <- minimax (formula = ~1/(1 + exp(-b *(x - a))), predvars = "x",
                   parvars = c("a", "b"),
                   family = binomial(), lx = -3, ux = 3,
                   lp = c(0, 1), up = c(1, 2.5), iter = 200, k = 3,
                   ICA.control= cont2, crt.minimax.control = cont1)
  print(res2)
  plot(res2)
}

############################################
# An example of a model with two predictors
############################################
# Mixed inhibition model
lower <- c(1, 4, 2, 4)
upper <- c(1, 5, 3, 5)
cont <- crt.minimax.control(optslist = list(maxeval = 100)) # to be faster
\dontrun{
res3 <- minimax(formula =  ~ V*S/(Km * (1 + I/Kic)+ S * (1 + I/Kiu)),
                predvars = c("S", "I"),
                parvars = c("V", "Km", "Kic", "Kiu"),
                lx = c(0, 0), ux = c(30, 60), k = 4,
                iter = 1, lp = lower, up = upper,
                ICA.control= list(rseed = 100),
                crt.minimax.control = cont)

  res3 <- iterate(res3, 100)
  print(res3)
  plot(res3) # sensitivity plot
}


# Now consider grid points instead of assuming continuous parameter space
# set n.grid to 5
\dontrun{
  res4 <- minimax(formula =  ~ V*S/(Km * (1 + I/Kic)+ S * (1 + I/Kiu)),
                  predvars = c("S", "I"),
                  parvars = c("V", "Km", "Kic", "Kiu"),
                  lx = c(0, 0), ux = c(30, 60),
                  k = 4, iter = 130, n.grid = 5, lp = lower, up = upper,
                  ICA.control= list(rseed = 100, checkfreq = Inf),
                  crt.minimax.control = cont)
  print(res4)
  plot(res4) # sensitivity plot
}

############################################
# Standardized maximin D-optimal designs
############################################
# Now assume the purpose is finding STANDARDIZED designs
# We know from the literature that the locally D-optimal design (LDOD)
# for this model has analytical solution.
# The follwoing function takes the parameter as input and returns
# the design points and weights of LDOD.
# x and w are exactly similar to the arguments of 'fimfunc'.
# x is a vector and returns the design points 'dimension-wise'.
# see explanation of the arguments of 'fimfunc' in 'Details'.

LDOD <- function(V, Km, Kic, Kiu){
  #first dimention is for S and the second one is for I.
  S_min <- 0
  S_max <- 30
  I_min <- 0
  I_max <- 60
  s2 <- max(S_min, S_max*Km*Kiu*(Kic+I_min)/
              (S_max*Kic*I_min+S_max*Kic*Kiu+2*Km*Kiu*I_min+2*Km*Kiu*Kic))
  i3 <- min((2*S_max*Kic*I_min + S_max*Kic*Kiu+2*Km*Kiu*I_min+Km*Kiu*Kic)/
              (Km*Kiu+S_max*Kic), I_max)
  i4 <- min(I_min + (sqrt((Kic+I_min)*(Km*Kic*Kiu+Km*Kiu*I_min+
                                         S_max*Kic*Kiu+S_max*Kic*I_min)/
                            (Km*Kiu+S_max*Kic))), I_max )
  s4 <- max(-Km*Kiu*(Kic+2*I_min-i4)/(Kic*(Kiu+2*I_min-i4)), S_min)
  x <- c(S_max, s2, S_max, s4, I_min, I_min, i3, i4)
  return(list(x = x, w =rep(1/4, 4)))

}
formalArgs(LDOD)
\dontrun{
  minimax(formula =  ~ V*S/(Km * (1 + I/Kic)+ S * (1 + I/Kiu)),
          predvars = c("S", "I"),
          parvars = c("V", "Km", "Kic", "Kiu"),
          lx = c(0, 0), ux = c(30, 60),
          k = 4, iter = 300,
          lp = lower, up = upper,
          ICA.control= list(rseed = 100, checkfreq = Inf),
          crt.minimax.control = cont,
          standardized = TRUE,
          localdes = LDOD)
}


################################################################
# Not necessary!
# The rest of the examples here are only for professional uses.
################################################################
# Imagine you have written your own FIM, say in Rcpp that is faster than
# the FIM created by the formula interface above.

###########################################
# An example of a model with two predictors
###########################################
# For example, th cpp FIM function for the mixed inhibition model is named:
formalArgs(FIM_mixed_inhibition)

# We should reparamterize the arguments to match the standard of the
# argument 'fimfunc' (see 'Details').
myfim <- function(x, w, param){
  npoint <- length(x)/2
  S <- x[1:npoint]
  I <- x[(npoint+1):(npoint*2)]
  out <- FIM_mixed_inhibition(S = S, I = I, w = w, param = param)
  return(out)
}
formalArgs(myfim)

# Finds minimax optimal design, exactly as before, but NOT using the
# formula interface.
\dontrun{
  res5 <- minimax(fimfunc = myfim,
                  lx = c(0, 0), ux = c(30, 60), k = 4,
                  iter = 100, lp = lower, up = upper,
                  ICA.control= list(rseed = 100),
                  crt.minimax.control = cont)
  print(res5)
  plot(res5) # sensitivity plot
}
#########################################
# Standardized maximin D-optimal designs
#########################################
# To match the argument 'localdes' when no formula inteface is used,
# we should reparameterize LDOD.
# The input must be 'param' same as the argument of 'fimfunc'
LDOD2 <- function(param)
  LDOD(V = param[1], Km = param[2], Kic = param[3], Kiu = param[4])

# compare these two:
formalArgs(LDOD)
formalArgs(LDOD2)
\dontrun{
  res6 <- minimax(fimfunc = myfim,
                  lx = c(0, 0), ux = c(30, 60), k = 4,
                  iter = 300, lp = lower, up = upper,
                  ICA.control= list(rseed = 100, checkfreq = Inf),
                  crt.minimax.control = cont,
                  standardized = TRUE,
                  localdes = LDOD2)
  res6
  plot(res6)
}

}
\references{
Masoudi E, Holling H, Wong W.K. (2017). Application of Imperialist Competitive Algorithm to Find Minimax and Standardized Maximin Optimal Designs. Computational Statistics and Data Analysis, 113, 330-345. \cr
Dette, H. (1997). Designing experiments with respect to 'standardized' optimality criteria. Journal of the Royal Statistical Society: Series B (Statistical Methodology), 59(1), 97-110. \cr
}
\seealso{
\code{\link{sensminimax}}
}
