% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Pilliat.R
\name{Pilliat}
\alias{Pilliat}
\title{Pilliat multiple change-point detection algorithm}
\usage{
Pilliat(
  X,
  threshold_d_const = 4,
  threshold_bj_const = 6,
  threshold_partial_const = 4,
  K = 2,
  alpha = 1.5,
  empirical = FALSE,
  threshold_dense = NULL,
  thresholds_partial = NULL,
  thresholds_bj = NULL,
  N = 100,
  tol = 0.01,
  rescale_variance = TRUE,
  test_all = FALSE,
  debug = FALSE
)
}
\arguments{
\item{X}{Matrix of observations, where each row contains a time series}

\item{threshold_d_const}{Leading constant for the analytical detection threshold for the dense statistic}

\item{threshold_bj_const}{Leading constant for \eqn{p_0} when computing the detection threshold for the Berk-Jones statistic}

\item{threshold_partial_const}{Leading constant for the analytical detection threshold for the partial sum statistic}

\item{K}{Parameter for generating seeded intervals}

\item{alpha}{Parameter for generating seeded intervals}

\item{empirical}{If \code{TRUE}, detection thresholds are based on Monte Carlo simulation using \code{\link{Pilliat_calibrate}}}

\item{threshold_dense}{Manually specified value of detection threshold for the dense statistic}

\item{thresholds_partial}{Vector of manually specified detection thresholds for the partial sum statistic, for sparsities/partial sums \eqn{t=1,2,4,\ldots,2^{\lfloor \log_2(p)\rfloor} }}

\item{thresholds_bj}{Vector of manually specified detection thresholds for the Berk-Jones statistic, order corresponding to \eqn{x=1,2,\ldots,x_0}}

\item{N}{If \code{empirical=TRUE}, \code{N} is the number of Monte Carlo samples used}

\item{tol}{If \code{empirical=TRUE}, \code{tol} is the false error probability tolerance}

\item{rescale_variance}{If \code{TRUE}, each row of the data is re-scaled by a MAD estimate (see \code{\link{rescale_variance}})}

\item{test_all}{If \code{TRUE}, the algorithm tests for a change-point in all candidate positions of each considered interval}

\item{debug}{If \code{TRUE}, diagnostic prints are provided during execution}
}
\value{
A list containing 
  \item{changepoints}{vector of estimated change-points}
  \item{number_of_changepoints}{number of changepoints}
  \item{scales}{vector of estimated noise level for each series}
  \item{startpoints}{start point of the seeded interval detecting the corresponding change-point in \code{changepoints}}
  \item{endpoints}{end point of the seeded interval detecting the corresponding change-point in \code{changepoints}}
}
\description{
R wrapper function for C implementation of the multiple change-point detection algorithm by \insertCite{pilliat_optimal_2022;textual}{HDCD}, using seeded intervals generated by Algorithm 4 in \insertCite{moen2023efficient;textual}{HDCD}. For the sake of simplicity, detection thresholds are chosen independently of the width of the interval in which a change-point is tested for (so \eqn{r=1} is set for all intervals).
}
\examples{
library(HDCD)
n = 50
p = 50
set.seed(100)
# Generating data
X = matrix(rnorm(n*p), ncol = n, nrow=p)
# Adding a single sparse change-point:
X[1:5, 26:n] = X[1:5, 26:n] +2

# Vanilla Pilliat:
res = Pilliat(X)
res$changepoints

# Manually setting leading constants for detection thresholds
res = Pilliat(X, threshold_d_const = 4, threshold_bj_const = 6, threshold_partial_const=4)
res$changepoints #estimated change-point locations

# Empirical choice of thresholds:
res = Pilliat(X, empirical = TRUE, N = 100, tol = 1/100)
res$changepoints

# Manual empirical choice of thresholds (equivalent to the above)
thresholds_emp = Pilliat_calibrate(n,p, N=100, tol=1/100)
thresholds_emp$thresholds_partial # thresholds for partial sum statistic
thresholds_emp$thresholds_bj # thresholds for Berk-Jones statistic
thresholds_emp$threshold_dense # thresholds for Berk-Jones statistic
res = Pilliat(X, threshold_dense =thresholds_emp$threshold_dense, 
              thresholds_bj = thresholds_emp$thresholds_bj,
              thresholds_partial =thresholds_emp$thresholds_partial )
res$changepoints
}
\references{
\insertAllCited{}
}
