#' Generate text using the Generative AI model based on a prompt and an image
#'
#' This function establishes a connection to a Generative AI model by providing essential
#' parameters and generates text based on the prompt and the image.
#'
#' @param model.parameter A character vector containing the Generative AI service provider,
#' corresponding model, version, API key, and proxy status.
#' @param temperature A numeric value. A higher value results in more creative responses,
#' while a lower value produces more straightforward text.
#' @param prompt A character string representing the query for text generation.
#' @param image.path A character string representing the path to the image. It should be a link
#' starting with \code{https}.
#'
#' @return If successful, a character string generated by the Generative AI model based on
#' the provided query and parameters. If the API response indicates an error, the function
#' halts execution and provides an error message.
#'
#' @details Providing accurate and valid information for each parameter is crucial
#' to ensure successful text generation by the Generative AI model. If any of the
#' provided parameters is incorrect, the function will respond with an error message based
#' on the information received from the API. Use the function \code{available.models} to
#' see all supported Generative AI models.
#'
#' @examples
#' \dontrun{
#'  # Get available models
#'  models = available.models()
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  google.model = connect.genai("google",
#'                               models$google$model[1],
#'                               models$google$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'
#'  # Connect to the model, replace API_KEY with your api key
#'  openai.model = connect.genai("openai",
#'                               models$openai$model[1],
#'                               models$openai$version[1],
#'                               "API_KEY",
#'                               FALSE)
#'  # Generate text
#'  temperature = 0.9
#'  prompt = "Please describe the following image in 50 words."
#'  image.path = "https://h5n1.gd.edu.kg/images/abstract1.jpg"
#'  image.description = text.image(google.model,
#'                                 temperature,
#'                                 prompt,
#'                                 image.path)
#'  cat(image.description)
#'  image.description = text.image(openai.model,
#'                                 temperature,
#'                                 prompt,
#'                                 image.path)
#'  cat(image.description)
#' }
#'
#' @export
#'
#' @importFrom GenAI moderation.openai
#' @importFrom GenAI image.to.data.uri
text.image = function(model.parameter,
                      temperature,
                      prompt,
                      image.path) {
  if (prompt == "" || is.na(prompt) || !inherits(prompt, "character")) {
    stop("Prompt is not in correct format.")
  }
  if (image.path == "" || is.na(image.path) || !inherits(image.path, "character")) {
    stop("image.path is not in correct format.")
  }
  switch (model.parameter["provider"],
          google = {
            api.URL = ifelse(
              model.parameter["proxy"],
              paste0(
                "https://api.genai.gd.edu.kg/google/",
                model.parameter["version"],
                "/models/",
                model.parameter["model"],
                ":generateContent?key=",
                model.parameter["api"]
              ),
              paste0(
                "https://generativelanguage.googleapis.com/",
                model.parameter["version"],
                "/models/",
                model.parameter["model"],
                ":generateContent?key=",
                model.parameter["api"]
              )
            )
            img.info = image.to.data.uri(image.path)
            if (img.info[1] == "jpg") {
              img.info[1] = "jpeg"
            }
            requestBody = list(
              contents = list(
                parts = list(
                  list(
                    text = prompt
                  ),
                  list(
                    inline_data = list(
                      mime_type = paste0("image/", img.info[1]),
                      data = img.info[2]
                    )
                  )
                )
              ),
              generationConfig = list(temperature = temperature)
            )
            requestBodyJSON = jsonlite::toJSON(requestBody, auto_unbox = TRUE)
            response = httr::POST(
              url = api.URL,
              body = requestBodyJSON,
              httr::add_headers("Content-Type" = "application/json")
            )
            responseJSON = httr::content(response, "parsed")
            if (!is.null(responseJSON$error)) {
              stop(responseJSON$error$message)
            }
            if (!is.null(responseJSON$blockReason)) {
              stop("The prompt may contain harmful content.")
            }
            return (as.character(responseJSON$candidates[[1]]$content$parts[[1]]$text))
          },
          openai = {
            moderation.openai(model.parameter, prompt)
            api.URL = ifelse(
              model.parameter["proxy"],
              paste0(
                "https://api.genai.gd.edu.kg/openai/",
                model.parameter["version"],
                "/chat/completions"
              ),
              paste0(
                "https://api.openai.com/",
                model.parameter["version"],
                "/chat/completions"
              )
            )
            img.info = image.to.data.uri(image.path)
            requestBody = list(
              model = model.parameter["model"],
              messages = list(
                list(
                  role = "user",
                  content = list(
                    list(
                      type = "text",
                      text = prompt
                    ),
                    list(
                      type = "image_url",
                      image_url = list(
                        url = paste0("data:image/", img.info[1], ";base64,", img.info[2])
                      )
                    )
                  )
                )
              ),
              temperature = temperature,
              max_tokens = 4096
            )
            requestBodyJSON = jsonlite::toJSON(requestBody, auto_unbox = TRUE)
            response = httr::POST(
              url = api.URL,
              body = requestBodyJSON,
              httr::add_headers(
                "Content-Type" = "application/json",
                "Authorization" = paste("Bearer", model.parameter["api"])
              )
            )
            responseJSON = httr::content(response, "parsed")
            if (!is.null(responseJSON$error)) {
              stop(responseJSON$error$message)
            }
            return (as.character(responseJSON$choices[[1]]$message$content))
          })
}
