% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IntegrateDerive.R
\name{Derive}
\alias{Derive}
\title{Derivative of GeDS objects}
\usage{
Derive(object, order = 1L, x, n = 3L)
}
\arguments{
\item{object}{the \code{\link{GeDS-Class}} object containing the GeDS fit which should be differentiated.
It should be the result of fitting a univariate GeDS regression via \code{\link{NGeDS}} or
\code{\link{GGeDS}}.}

\item{order}{integer value indicating the order of differentiation required (e.g. first, second or higher derivatives).
Note that \code{order} should be lower than \code{n} and that non-integer values will be passed
to the function \code{\link{as.integer}}.}

\item{x}{numeric vector containing values of the independent
variable at which the derivatives of order \code{order} should be computed.}

\item{n}{integer value (2, 3 or 4) specifying the order (\eqn{=} degree \eqn{+ 1}) of the GeDS fit
to be differentiated.
By default equal to \code{3L}.}
}
\description{
This function computes derivatives of a fitted GeDS regression model.
}
\details{
The function is based on \code{\link[splines]{splineDesign}} and
it computes the exact derivative of the fitted GeDS regression.

The function uses the property that the \eqn{m}-th derivative of a spline, \eqn{m= 1,2,...},
 expressed in terms of B-splines can be computed by differencing the corresponding B-spline coefficients
 (see e.g. De Boor, 2001, Chapter X, formula (15)).
Since the GeDS fit is a B-spline representation of the predictor,
it cannot work on the response scale in the GNM (GLM) framework.
}
\examples{
# Generate a data sample for the response variable
# Y and the covariate X
set.seed(123)
N <- 500
f_1 <- function(x) (10*x/(1+100*x^2))*4+4
X <- sort(runif(N, min = -2, max = 2))
# Specify a model for the mean of Y to include only
# a component non-linear in X, defined by the function f_1
means <- f_1(X)
# Add (Normal) noise to the mean of Y
Y <- rnorm(N, means, sd = 0.1)

# Fit GeDS regression with only a spline component in the predictor model
Gmod <- NGeDS(Y ~ f(X), beta = 0.6, phi = 0.995, Xextr = c(-2,2))

# Compute the second derivative of the cubic GeDS fit
# at the points 0, -1 and 1
Derive(Gmod, x = c(0, -1, 1), order = 2, n = 4)
}
\references{
De Boor, C. (2001). \emph{A Practical Guide to Splines (Revised Edition)}.
Springer, New York.
}

