% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MainFunctions.R
\name{FixedPoint}
\alias{FixedPoint}
\title{A function for finding the fixed point of a contraction mapping}
\usage{
FixedPoint(Function, Inputs, Outputs = c(), Method = c("Anderson", "Simple",
  "Aitken", "Newton", "MPE", "RRE", "VEA", "SEA"),
  ConvergenceMetric = function(Resids) {     max(abs(Resids)) },
  ConvergenceMetricThreshold = 1e-10, MaxIter = 1000, MaxM = 10,
  ExtrapolationPeriod = 7, Dampening = 1, PrintReports = FALSE,
  ReportingSigFig = 5, ConditionNumberThreshold = 1000, Plot = c("NoPlot",
  "ConvergenceFig", "ChangePerIterate"), ConvergenceFigLags = 5,
  ChangePerIteratexaxis = c())
}
\arguments{
\item{Function}{This is the function for which a fixed point is sought. This function must take and return a vector of the same dimension.}

\item{Inputs}{This can be either a vector of values that is an initial guess for a fixed point or it can be an N x A matrix of previous inputs for which corresponding
outputs are available. In this case N is the dimensionality of the fixed point vector you are seeking (Hence each column is a matrix that is input to the "Function")
and A is the number of previous Inputs/Outputs that are being provided to the fixed point. Where a matrix is input, a corresponding outputs must be provided or the last
column of the outputs matrix is taken as a startpoint guess and the rest of the inputs and output matrices are discarded.}

\item{Outputs}{(Optional) This is a matrix of the Function values for each column of the input. It must be provided so that column k of the outputs matrix is equal to
Function(Column k of inputs matrix).}

\item{Method}{This is the fixed point method to be used. It can be "Anderson", "Simple", "Aitken", "Newton", "MPE", "RRE", "VEA" or "SEA". See vignette and references
to see explanations of these methods.}

\item{ConvergenceMetric}{This is a function that takes in a vector of residuals from one iterate of the function (defined as f(x) - x for vector x and function f) and
returns a scalar. This scalar should be low when convergence is close to being achieved. By default this is the maximum residual by absolute value (the sup norm in the
space of residuals).}

\item{ConvergenceMetricThreshold}{This is the threshold for terminating the algorithm. The algorithm will terminate when the scalar that ConvergenceMetric returns is less
than ConvergenceMetricThreshold. This can be set to a negative number in which case the algorithm will run until MaxIter is hit or an error occurs (Note that an error is
likely in trying to use any method other than "Simple" when a fixed point is already found).}

\item{MaxIter}{This is the maximum number of iterates that will be undertaken.}

\item{MaxM}{This is the maximum number of saved iterates that are used in the Anderson algorithm. It has no effect if another method is chosen. Note that the number of
previous iterates that will actually be used is the minimum of MaxIter, the dimensionality of the function's vector and the number of inputs that have been tried to
previously (the width of the Outputs matrix at each given stage of the algorithm). If PrintReports = TRUE, the number of previous iterates actually used is reported as
the algorithm is running.}

\item{ExtrapolationPeriod}{This is the number of simple iterates to perform before extrapolating. This is used for the MPE, RRE, VEA and SEA methods and has no effect
if another method is chosen Where an epsilon algorithm is used this should be a multiple of 2, ie (4,6,8,etc).}

\item{Dampening}{This is the dampening parameter. By default it is 1 which means no dampening takes place. It can also be less than 1 (indicating dampening) or more
than 1 (indicating extrapolation).}

\item{PrintReports}{This is a boolean describing whether to print ongoing ConvergenceMetric values for each iterate.}

\item{ReportingSigFig}{This is the number of significant figures that will be used in printing the convergence values to the console (only if PrintReports is TRUE).}

\item{ConditionNumberThreshold}{This is a threshold for what condition number is acceptable for solving the least squares problem for the Anderson Method. If the
condition number is larger than this threshold then fewer previous iterates will be used in solving the problem. This has no effect unless the "Anderson" method is
used.}

\item{Plot}{This determines whether a plot should be drawn for every iterate. It can be "NoPlot", "ConvergenceFig" or "ChangePerIterate". By default it is
"NoPlot" and no plot is drawn. If it is "ConvergenceFig" then a plot is shown with iterates on the x axis and convergence (as defined by the ConvergenceMetric) is on
the y axis. If it is "ChangePerIterate" then there is the index of the array value on the x axis and the value of the array value on the y axis. The previous iterate
is also shown so the change per iterate can be visualised.}

\item{ConvergenceFigLags}{This only affects anything if Plot == "ConvergenceFig". This gives how many previous iterates should be shown on the x axis. By default it
is 5. To see them all set it to a high number.}

\item{ChangePerIteratexaxis}{This only affects anything if Plot == "ChangePerIterate". Sometimes there is a more appropriate xaxis value to use than (the default)
value index for this figure. For instance in the consumption smoothing problem in the vignette every value is a value function value at a given budget level.
In this case the budget levels could be used for this xaxis.}
}
\value{
A list containing the FixedPoint, the Inputs and corresponding Outputs, and convergence values (which are computed under the "ConvergenceMetric").
 The list will also include a "Finish" statement describing why it has finished. This is often going to be due to either MaxIter or ConvergenceMetricThreshold being
 reached. It may also terminate due to an error in generating a new input guess or using the function with that guess. If this occurs the function will terminate early
 and the "Finish" statement will describe the issue. In this event there will also be additional objects returned in the list "NewInputVector" and possibly
 "NewOutputVector" that are useful in debugging the issue.
}
\description{
This function takes in a function and an initial guess for the fixed point of that function. It then uses a fixed point method to find the fixed point of that function.
}
\examples{
# For the simplest possible example we can seek the fixed point of the cos function with a scalar.
Inputs = 0.3
Function = function(x){ cos(x) }
A = FixedPoint(Function, Inputs, Method = "Aitken", Dampening = 0.5)
B = FixedPoint(Function, Inputs, Method = "Anderson", Dampening = 1.0)

# For this next one the ConvergenceMetricThreshold is negative so the algorithm
# will keep running until MaxIter is met.
C = FixedPoint(Function, Inputs, Method = "Simple", MaxIter = 4, ConvergenceMetricThreshold = -1)
# But we can continue solving for this fixed point but now switching to the Newton method.
D = FixedPoint(Function, C$Inputs, C$Outputs, Method = "Newton")

# We can also find a 4 dimensional fixed point vector of this function.
Inputs = c(0.3, 98, 0, pi)
E = FixedPoint(Function, Inputs, Method = "Anderson")
F = FixedPoint(Function, Inputs, Method = "Anderson", MaxM = 4, ReportingSigFig = 13)
}
