% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/itemStability.R
\name{itemStability}
\alias{itemStability}
\title{Item Stability Statistics from \code{\link[EGAnet]{bootEGA}}}
\usage{
itemStability(bootega.obj, orig.wc, item.freq = 0.1, plot.item.rep = TRUE)
}
\arguments{
\item{bootega.obj}{A \code{\link[EGAnet]{bootEGA}} object}

\item{orig.wc}{Numeric or character.
A vector with community numbers or labels for each item.
Typically uses community results (\code{wc}) from \code{\link[EGAnet]{EGA}}}

\item{item.freq}{A value for lowest frequency allowed in \code{item.dim.rep} output.
Removes noise from table to allow for easier interpretation.
Defaults to \code{.10}}

\item{plot.item.rep}{Should the plot be produced for \code{item.replication}?
If \code{TRUE}, then a plot for the \code{item.replication} output will be produced.
Defaults to \code{TRUE}}
}
\value{
Returns a list containing:

\item{item.replication}{The proportion of times each item replicated
within the defined dimension}

\item{mean.dim.rep}{The average replication proportion of items replicating
in each dimension. More simply, the average of the \code{item.replication}
output for each dimension}

\item{item.dim.rep}{The proportion of times each item replicated
within each possible dimension. Dimensions greater than the maximum
number used in the \code{orig.wc} argument are labeled based on the
largest remaining components after the dimensions used to \code{orig.wc}}

\item{item.loadings}{Matrix of the average standardized network loading
(computed using \code{\link[EGAnet]{net.loads}}) for each item in each dimension}

\item{wc}{A matrix containing the community membership values for
each bootstrapped sample. The values correspond to the values input
for the \code{orig.wc} argument}

\item{plot.itemStability}{A plot of the number of times each item
replicates in its original community membership (\code{orig.wc})}
}
\description{
Based on the \code{\link[EGAnet]{bootEGA}} results, this function
computes and plots the number of times an item (variable) is estimated
in the same factor/dimension as originally estimated by \code{\link[EGAnet]{EGA}} (\code{item.replication}).
The output also contains each item's replication frequency (i.e., proportion of
bootstraps that an item appeared in each dimension; \code{item.dim.rep}) as well as the average
network loading for each item in each dimension (\code{item.loadings}).
}
\examples{

# Load data
wmt <- wmt2[,7:24]

\donttest{# Estimate EGA network
## plot.type = "qqraph" used for CRAN checks
## plot.type = "GGally" is the default
ega.wmt <- EGA(data = wmt, model = "glasso", plot.type = "qgraph")

# Estimate dimension stability
boot.wmt <- bootEGA(data = wmt, iter = 100, typicalStructure = TRUE,
plot.typicalStructure = TRUE, model = "glasso", plot.type = "qgraph",
type = "parametric", ncores = 2)
}

# Estimate item stability statistics
res <- itemStability(boot.wmt, orig.wc = ega.wmt$wc)

# Changing plot features (ggplot2)
## Changing colors (ignore warnings)
### qgraph Defaults
res$plot.itemStability + 
    ggplot2::scale_color_manual(values = rainbow(max(res$uniq.num)))

### Pastel
res$plot.itemStability + 
    ggplot2::scale_color_brewer(palette = "Pastel1")
    
## Changing Legend (ignore warnings)
res$plot.itemStability + 
    ggplot2::scale_color_discrete(labels = "Intelligence")

}
\references{
Christensen, A. P., & Golino, H. (2019).
Estimating the stability of the number of factors via Bootstrap Exploratory Graph Analysis: A tutorial.
\emph{PsyArXiv}.
\doi{10.31234/osf.io/9deay}

Christensen, A. P., Golino, H., & Silvia, P. J. (2020).
A psychometric network perspective on the validity and validation of personality trait questionnaires.
\emph{European Journal of Personality}.
\doi{10.1002/per.2265}
}
\seealso{
\code{\link[EGAnet]{EGA}} to estimate the number of dimensions of an instrument using EGA and
\code{\link[EGAnet]{CFA}} to verify the fit of the structure suggested by EGA using confirmatory factor analysis.
}
\author{
Hudson Golino <hfg9s at virginia.edu> and Alexander P. Christensen <alexpaulchristensen@gmail.com>
}
