% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cas_handlers.r
\name{cas}
\alias{cas}
\alias{is.cas}
\alias{as.cas}
\alias{[[.cas}
\alias{[.cas}
\alias{[[<-.cas}
\alias{[<-.cas}
\alias{format.cas}
\alias{as.character.cas}
\alias{show.cas}
\alias{print.cas}
\alias{as.list.cas}
\alias{as.double.cas}
\alias{as.integer.cas}
\alias{c.cas}
\alias{as.data.frame.cas}
\title{Functions for handling chemical abstract service (CAS) registry numbers}
\usage{
cas(length = 0L)

is.cas(x)

as.cas(x)

\method{[[}{cas}(x, i)

\method{[}{cas}(x, i)

\method{[[}{cas}(x, i) <- value

\method{[}{cas}(x, i) <- value

\method{format}{cas}(x, hyphenate = TRUE, ...)

\method{as.character}{cas}(x, ...)

show.cas(x, ...)

\method{print}{cas}(x, ...)

\method{as.list}{cas}(x, ...)

\method{as.double}{cas}(x, ...)

\method{as.integer}{cas}(x, ...)

\method{c}{cas}(...)

\method{as.data.frame}{cas}(...)
}
\arguments{
\item{length}{A non-negative \code{integer} specifying the desired length. Double values will be coerced to
\code{integer}: supplying an argument of length other than one is an error.}

\item{x}{Object from which data needs to be extracted or replaced, or needs to be coerced into a specific
format. For nearly all of the functions documented here, this needs to be an object of the S3 class 'cas',
which can be created with \code{as.cas}. For \code{as.cas}, \code{x} can be a \code{character} (CAS registry number
with or without hyphenation) or a \code{numeric} value. Note that \code{as.cas} will only accept correctly
formatted and valid CAS registry numbers.}

\item{i}{Index specifying element(s) to extract or replace. See also \code{\link[base:Extract]{Extract}}.}

\item{value}{A replacement value, can be anything that can be converted into an S3 cas-class object with \code{as.cas}.}

\item{hyphenate}{A \code{logical} value indicating whether the formatted CAS number needs to be hyphenated.
Default is \code{TRUE}.}

\item{...}{Arguments passed to other functions}
}
\value{
Functions \code{cas}, \code{c} and \code{as.cas} return S3 class 'cas' objects. Coercion functions
(starting with 'as') return the object as specified by their respective function names (i.e., \code{integer},
\code{double}, \code{character}, \code{list} and \code{data.frame}). The \code{show.cas} and \code{print} functions
also return formatted \code{charater}s. The function \code{is.cas} will return a single \code{logical} value,
indicating whether \code{x} is a valid S3 cas-class object. The square brackets return the selected index/indices,
or the \code{vector} of cas objects where the selected elements are replaced by \code{value}.
}
\description{
Functions for handling chemical abstract service (CAS) registry numbers
}
\details{
In the database \href{https://en.wikipedia.org/wiki/Chemical_Abstracts_Service}{CAS registry} numbers are stored
as text (type \code{character}). As CAS numbers can consist of a maximum of 10 digits (plus two hyphens) this means
that each CAS number can consume up to 12 bytes of memory or disk space. By storing the data numerically, only
5 bytes are required. These functions provide the means to handle CAS registry numbers and coerce from and to
different formats and types.
}
\examples{
## This will generate a vector of cas objects containing 10
## fictive (0-00-0), but valid registry numbers:
cas(10)

## This is a cas-object:
is.cas(cas(0L))

## This is not a cas-object:
is.cas(0L)

## Three different ways of creating a cas object from
## Benzene's CAS registry number (the result is the same)
as.cas("71-43-2")
as.cas("71432")
as.cas(71432L)

## This is one way of creating a vector with multiple CAS registry numbers:
cas_data <- as.cas(c("64175", "71432", "58082"))

## This is how you select a specific element(s) from the vector:
cas_data[2:3]
cas_data[[2]]

## You can also replace specific elements in the vector:
cas_data[1] <- "7440-23-5"
cas_data[[2]] <- "129-00-0"

## You can format CAS numbers with or without hyphens:
format(cas_data, TRUE)
format(cas_data, FALSE)

## The same can be achieved using as.character
as.character(cas_data, TRUE)
as.character(cas_data, FALSE)

## There are also show and print methods available:
show(cas_data)
print(cas_data)

## Numeric values can be obtained from CAS using as.numeric, as.double or as.integer
as.numeric(cas_data)

## Be careful, however. Some CAS numbers cannot be represented by R's 32 bit integers
## and will produce NA's. This will work OK:
huge_cas <- as.cas("9999999-99-5")

\dontrun{
## This will not:
as.integer(huge_cas)
}

## The trick applied by this package is that the final
## validation digit is stored separately as attribute:
unclass(huge_cas)

## This is how cas objects can be concatenated:
cas_data <- c(huge_cas, cas_data)

## This will create a data.frame
as.data.frame(cas_data)

## This will create a list:
as.list(cas_data)
}
\author{
Pepijn de Vries
}
