#' dt.quantile.numerics
#'
#' @description wrapper function that computes the quantiles for each selected quantitative variable in each group after applying a filter.
#'
#' @param dt.name a character value specifying the name of a data.frame or data.table object.

#' @param the.variables a character vector specifying the variables that we want to apply a function to.  Only values that exist in names(dat) will be used; other values in the.variables will be excluded from the calculation.  When the.variables includes ".", then all values in names(dat) will be selected.  Values of the.variables that also exist in grouping.variables will be excluded from the.variables (but grouped by these values).

#' @param the.filter a character value, logical value, or expression stating the logical operations to be performed in filtering the data prior to calculating the.function.

#' @param probs the range specifying the upper and lower quartiles
#'
#' @param grouping.variables  a character vector specifying variables to group by in performing the computation.  Only values that exist in names(dat) will be used.
#'
#' @param envir the environment in which the code would be evaluated; parent.frame() by default.

#' @param sortby.group a logical value to specify if the sorting functionality needs to be applied or not
#'
#' @param table.format a character vector specifying if table should be in a wide format or a tall format
#'
#' @param return.as describes whether return should be result, code or mixture of both
#'
#' @param  ... additional arguments to be passed
#' @return The function outputs depend on the specified 'return.as' parameter:
#' - If 'return.as' is "result", it returns a modified version of the input data frame or data table, including the calculated quantiles for each specified numeric variable. These quantiles are computed based on the provided 'probs' array, after applying any specified filters and subgroupings. The results are structured according to the 'table.format', which can be wide or long, reflecting any group-based sorting if 'sortby.group' is enabled.
#' - If 'return.as' is "code", the function will return the R code or expressions capable of computing the quantiles, allowing users to review or execute the calculations separately.
#' - For other specified values of 'return.as', the function may provide both the quantile calculations and the corresponding R code, depending on how the function is implemented.
#'
#' This function facilitates a comprehensive analysis of the data's distribution by calculating specific quantiles, aiding in the statistical examination of the dataset's quantitative attributes, particularly after implementing specific filters and groupings.

#' @export



dt.quantile.numerics <- function(dt.name, the.variables = ".", probs = c(0.25, 0.75),
                                 the.filter = NULL, grouping.variables = NULL,
                                 sortby.group = TRUE, table.format = "long",
                                 return.as = "result", envir = parent.frame(), ...){

  other.params = sprintf("probs = c(%s)", paste(sort(unique(probs)), collapse = ", "))

  return(dt.calculate(dt.name = dt.name, the.functions = "quantile.numerics",
                      the.variables = the.variables, the.filter = the.filter,
                      grouping.variables = grouping.variables, sortby.group = sortby.group,
                      other.params = other.params, table.format = table.format,
                      add.function.name = FALSE, individual.variables = TRUE,
                      output.as.table = TRUE, return.as = return.as, envir = envir))
}
