% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/D3mirt.R
\name{D3mirt}
\alias{D3mirt}
\title{3D DMIRT Model Estimation}
\usage{
D3mirt(x, constructs = NULL)
}
\arguments{
\item{x}{Data frame with rows for items and columns for model parameters. The number of columns must be more than or equal to 4, i.e., three columns with(\emph{a}) parameters and at least one column for difficulty (\emph{d}) parameters.}

\item{constructs}{Optional. Nested lists with integer values indicating construct. Default is \code{constructs = NULL}.}
}
\value{
S3 object of class \code{D3mirt} with lists of \emph{a} and \emph{d} parameters from the 2PL or GRM estimation, multidimensional difficulty (MDIFF), multidimensional discrimination (MDISC), direction cosines and degrees for vector angles, construct lists, and vector coordinates.
}
\description{
Descriptive multidimensional item response theory modeling (DMIRT; Reckase, 2009, 1985, Reckase & McKinley, 1991), for dichotomous and polytomous items restricted to three dimensions.
}
\details{
The \code{D3mirt()}function takes in a data frame of factor slopes and difficulty parameters from a compensatory three-dimensional two-parameter logistic model (2PL) or a graded
response model (GRM; Samejima, 1969), fitted in accordance with descriptive item response theory model specifications described below.
The function returns an S3 object containing descriptive multidimensional item response theory estimates that can be graphed as vector arrows representing item response functions in a three dimensional space with \link{plotD3mirt}.

Note, model parameters from the multidimensional 2PL or GRM must be assessed prior to using the \code{D3mirt()} function (see examples section below or the vignette included in this package).
This means that the model must first be identified (see \link{modid}.
}
\section{DMIRT Theory}{
In DMIRT analysis, also called within multidimensional modeling, it is assumed that items in a multidimensional ability space can measure single or multiple latent traits (Reckase, 2009, 1985; Reckase & McKinley, 1991).
The methodology is a type of data reduction technique that uses a compensatory type of model (Reckase, 2009), i.e., a model that assumes the existence of a multidimensional trait space forming an interconnected latent ability field.
This means that if one trait level is low, some other trait in the model can compensate to create the same predictive score.
Probability estimates are therefore based on linear combinations of trait scores in the model.

Moreover, the DMIRT analysis is said to be descriptive because the results describe the extent to which items in a test are unidimensional,
i.e., that the items describe one dimension only, or are within-multidimensional, i.e., that the items describe more than one dimension.

The DMIRT estimation process consists of first fitting a compensatory multidimensional two-parameter model so that discrimination and difficulty parameters can be extracted.
For \code{D3mirt}, this implies that a compensatory three-dimensional 2PL or GRM must be fitted.
Next, using these parameters the DMIRT computation finds the direction of the highest possible discrimination slope for each item.
The direction, in turn, shows what latent trait the item can be said to measure in the multidimensional latent model.
The output can be visualized as vector arrows representing item response functions located in a multidimensional space pointing in the direction of the maximum slope.
The \code{D3mirt} can be used to calculate all necessary estimates for this purpose, as well as to plot the results (albeit limited to three dimensions).

The most central estimates, in DMIRT analysis are the single multidimensional discrimination (MDISC) parameter, the multidimensional difficulty (MDIFF) parameters, and the directional discrimination (DDISC) parameters (Reckase2009, 1985; Reckase & McKinley, 1991).
In addition, if constructs are used in the function call the output will also contain the directional discrimination (DDISC) parameters for the items on each construct vector orientation.
This makes it possible to compare item discrimination under the assumption that they measure in the same direction as each constructs.
}

\section{Constructs}{
The user has the option of including constructs in the estimation, by creating one or more nested lists that indicate what items belong to what construct (see the examples section below).
From this, the \code{D3mirt()} function calculates direction cosines for the constructs by normalizing the direction cosines for the items contained in each construct list.
Note, the length of the construct vector arrows is arbitrary.
}

\examples{
\donttest{
# Load data
data("anes0809offwaves")
x <- anes0809offwaves
x <- x[,3:22] # Remove columns for age and gender

# Fit a three-dimensional graded response model with orthogonal factors
# Example below uses Likert items from the built-in data set "anes0809offwaves"
# Item W7Q3 and item W7Q20 was selected with `modid()`
# The model specification set all items in the data set (1-20)
# to load on all three factors (F1-F3)
# The START and FIXED commands are used on the two items to identify the DMIRT model
spec <- '  F1 = 1-20
           F2 = 1-20
           F3 = 1-20

           START=(W7Q3,a2,0)
           START=(W7Q3,a3,0)

           START=(W7Q20,a3,0)

           FIXED=(W7Q3,a2)
           FIXED=(W7Q3,a3)

           FIXED=(W7Q20,a3) '


mod1 <- mirt::mirt(x,
                   spec,
                   itemtype = 'graded',
                   SE = TRUE,
                   method = 'QMCEM')

# Assign a data frame with factor loadings (located in the first three columns in mod1),
# and difficulty parameters (columns 4-8 in mod1)
d <- data.frame(mirt::coef(mod1,
                           simplify=TRUE)$'items'[,1:8])

# Call D3mirt() with data frame d
g <- D3mirt(d)
summary(g) # Show summary of results

# Call to D3mirt(), including optional nested lists for three constructs
# Item W7Q16 is not included in any construct because of model violations
# The model violations for the W7Q16 item can be seen when plotting the model
c <- list(list (1,2,3,4),
          list(5,7,8,9,10),
          list(11,12,13,14,15,15,16,17,18,19,20))
g <- D3mirt(d, c)
summary(g)
}
}
\references{
Chalmers, R., P. (2012). mirt: A Multidimensional Item Response Theory Package for the R Environment. \emph{Journal of Statistical Software, 48}(6), 1-29.

Reckase, M. D. (2009). \emph{Multidimensional Item Response Theory}. Springer.

Reckase, M. D. (1985). The Difficulty of Test Items That Measure More Than One Ability. \emph{Applied Psychological Measurement, 9}(4), 401-412–412. https://doi-org.ezp.sub.su.se/10.1177/014662168500900409

Reckase, M. D., & McKinley, R. L. (1991). The Discriminating Power of Items That Measure More Than One Dimension. \emph{Applied Psychological Measurement, 15}(4), 361-373–373. https://doi-org.ezp.sub.su.se/10.1177/014662169101500407

Samejima, F. (1969). Estimation of latent ability using a response pattern of graded scores. \emph{Psychometrika 34}, 1–97. https://doi.org/10.1007/BF03372160
}
\author{
Erik Forsberg
}
