% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotgg.R
\name{plotgg}
\alias{plotgg}
\title{Heatmap plot of affinity() output}
\usage{
plotgg(
  data,
  variable,
  legendlimit,
  col = NULL,
  show.value = NULL,
  value.digit = NULL,
  text.size = NULL,
  plot.margin = NULL,
  ...
)
}
\arguments{
\item{data}{the output of affinity()}

\item{variable}{a column name in $all dataframe in affinity() output; should be a quantitative column;
can be one of the following: "entity_1_count_mA", "entity_2_count_mB", "obs_cooccur_X", "sites_total_N",
"p_value", "exp_cooccur", "alpha_mle", "jaccard", "sorensen", "simpson"}

\item{legendlimit}{"datarange" or "balanced"; if "datarange", the legend spans to the range of data,
if "balanced, the legend spans in equal magnitude from the center (= white by default) in both directions;
note that, irrespective of the value-span in legend, the color spectrum of the plot and legend always goes from the center (=white by default)
to two directions in equal magnitude. See details for more information.}

\item{col}{a set of three colors c("#87beff", "white", "#fd6a6c") by default to represent low, mid and high values in the plot;
these colors are applied with ggplot::scale_fill_gradient2()}

\item{show.value}{a boolean to show ("TRUE") or hide ("FALSE") values in the plot; "TRUE" by default if <=20 entities to compare, otherwise "FALSE" by default}

\item{value.digit}{the number of digits in values if they are printed; default 2}

\item{text.size}{the size of values if they are printed; default 2.5}

\item{plot.margin}{same as ggplot's plot.margin which includes top, right, bottom and left margins as "margin(1,1,5,2, "cm")"}

\item{...}{Additional arguments to control behavior of the function.}
}
\value{
This function returns a heatmap plot generated with ggplot() behind the scene.
}
\description{
This function works on the output of affinity() and uses ggplot2::ggplot() to plot a heatmap plot for the numeric columns of $all dataframe except
the interval columns (median interval and confidence intervals) and confidence level (which is a constant for all pairs in one run of the code)
}
\details{
This function is really a ggplot behind the scene where I have taken care of the default value of many arguments for generating a useful plot.
It generates a plot for the lower triangle of an NxN square matrix where both row and columns carry the same set of entities,
such that all pairwise analyses are shown in the plot (upper triangle is the mirror image of the lower triange
and diagonals are the relation to the self which are excluded).

The plots can be requested using the column names of $all of the main output of affinity(). The function can include additional arguments
either inside plot.gg() or by appending to it with a "+" that is characteristic of ggplot().

"legendlimit" centers the legend color to white by default at null expectation in the case of alpha_mle,
and negative and positive values stretch between pastel blue and pastel red colors, respectively
such that the color spectrum is applied NOT to the range of data but to the same extent of values
on both sides of zero, which is max(abs(valrange)) and -(max(abs(valrange))). For example, if alpha_mle
ranges between -1.25 to 2.0, then the color spectrum always ranges between -2.0 and 2.0 but the legend can be printed
to span between -1.25 and 2.0 with "dataframe" and -2.0 and 2.0 with "balanced".

For "entity_1_count_mA", "entity_2_count_mB", and "sites_total_N", there is no natural midpoint.
So, "balanced" and "datarange" both use the natural behavior of ggplot in creating the color spectrum that spans between the extremes of the data.

For "obs_cooccur_X", and "exp_cooccur" also, there is no natural midpoint.
To make the two plots of observed and expected cooccurrence counts comparable visually, one color scale has been applied in these two plots
such that the spectrum ranges between the extremes of "obs_cooccur_X", and "exp_cooccur" collectively.
}
\examples{
if(requireNamespace("cooccur", quietly = TRUE)) {
  data(finches, package = "cooccur")
} else {
  message("The cooccur package is not installed.
          You can install it with install.packages('cooccur').")
}

# require(cooccur)
# data(finches)
data(finches, package = "cooccur")
head(finches)
require(ggplot2)



# the remainder of the script has been enclosed under \donttest{}
# to bypass the CRAN's 5 second limit on example files
# --------------------------------------------------------------

\donttest{

# plotting various variables
# ---------------------------------------------
# compute alpha and other quantities for island-pair affinity (beta diversity)
# the square matrices are not used for plotting
myout <- affinity(data = finches, row.or.col = "col")
# myout

plotgg(data = myout, variable = "alpha_mle", legendlimit = "datarange")
# in the example above, null expectation of the alpha_mle (=0) has white color,
# and negative and positive values stretch between "#87beff" and "#fd6a6c", respectively
# so that the color spectrum is applied NOT to the range of data but to the same extent of values
# on both sides of zero, which is max(abs(valrange)) and -(max(abs(valrange))).
# however, the legend can be printed to show the extent of data with "datarange"
# or the entire spectrum where the color is applied with "balanced".
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced")
# notice that the two plots above are identical but the legend has
# different range with the same color scale.


plotgg(data = myout, variable = "sorensen", legendlimit = "balanced")
plotgg(data = myout, variable = "jaccard", legendlimit = "balanced")

# in the case of observed and expected cooccurrences, one color scale is applied for both plots
# so that the shades of color across plots can be visually compared
plotgg(data = myout, variable = "exp_cooccur", legendlimit = "datarange")
plotgg(data = myout, variable = "exp_cooccur", legendlimit = "balanced")
plotgg(data = myout, variable = "obs_cooccur_X", legendlimit = "balanced")

plotgg(data = myout, variable = "entity_1_count_mA", legendlimit = "datarange")
plotgg(data = myout, variable = "entity_2_count_mB", legendlimit = "datarange")
plotgg(data = myout, variable = "total_N", legendlimit = "datarange")
# for "entity_1_count_mA", "entity_2_count_mB", "sites_total_N",
# if legendlimit is set to "balanced", it will be changed to "datarange"
plotgg(data = myout, variable = "entity_2_count_mB", legendlimit = "balanced")


# change color of the plot
# ---------------------------------------------
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced")
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced",
       col = c('#99cc33', 'white', '#ff9933'))

plotgg(data = myout, variable = "obs_cooccur_X", legendlimit = "balanced")
plotgg(data = myout, variable = "obs_cooccur_X", legendlimit = "balanced",
       col = c('#99cc33', 'white', '#ff9933'))


# change the characteristics of text printed in the plot
# ------------------------------------------------------
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced")

# change the number of digits; the default is 2
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced", value.digit = 3)

# make the fonts bigger; the default is 2.5
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced", text.size = 3.5)



# hide values from the plot
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced", show.value = FALSE)


# increase or decrease margin
# ---------------------------------------------

myout <- affinity(data = finches, row.or.col = "row")
# myout

plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced")
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced",
       plot.margin = ggplot2::margin(1,1,5,2, "cm"))


# change angle of x-axis tick label; the default is 35 degrees
# ------------------------------------------------------------
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced")
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced") +
  ggplot2::theme(axis.text.x = element_text(angle = 45))

# to change to 90 degrees, adjust vjust
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced") +
  ggplot2::theme(axis.text.x = element_text(angle = 90))
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced") +
  ggplot2::theme(axis.text.x = element_text(angle = 90, vjust = 0.5))


# additional elements in the plot
# ----------------------------------
# because it is ggplot output, you can use the arguments of ggplot() to make changes

# add plot title and change legend title
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced") +
  ggplot2::theme(axis.text.x = element_text(angle = 90, vjust = 0.5)) +
  ggplot2::ggtitle("Affinity of island pairs measured with Alpha MLE") +
  ggplot2::labs(fill = 'Alpha MLE')


# an example of much bigger dataset
# -------------------------------------
data("beetles", package = "cooccur")
dim(beetles)
myout <- affinity(data = beetles, row.or.col = "row")

plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced") +
  theme(axis.text.x = element_text(angle = 90, vjust = 0.5))
plotgg(data = myout, variable = "alpha_mle", legendlimit = "balanced",
       show.value = TRUE, text.size = 1.5, value.digit = 1) +
  theme(axis.text.x = element_text(angle = 90, vjust = 0.5))

} #end of \donttest{}



}
\author{
Kumar Mainali
}
