% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimationCKT.KendallReg.R
\name{CKT.KendallReg.LambdaCV}
\alias{CKT.KendallReg.LambdaCV}
\title{Kendall's regression: choice of the penalization parameter by K-folds cross-validation}
\usage{
CKT.KendallReg.LambdaCV(
  X1 = NULL,
  X2 = NULL,
  Z = NULL,
  ZToEstimate,
  designMatrixZ = cbind(ZToEstimate, ZToEstimate^2, ZToEstimate^3),
  typeEstCKT = 4,
  h_lambda,
  Lambda = identity,
  kernel.name = "Epa",
  Kfolds_lambda = 10,
  l_norm = 1,
  matrixSignsPairs = NULL,
  progressBars = "global",
  observedX1 = NULL,
  observedX2 = NULL,
  observedZ = NULL
)
}
\arguments{
\item{X1}{a vector of n observations of the first variable \eqn{X_1}.}

\item{X2}{a vector of n observations of the second variable \eqn{X_2}.}

\item{Z}{a vector of n observations of the conditioning variable,
or a matrix with n rows of observations of the conditioning vector
(if \eqn{Z} is multivariate).}

\item{ZToEstimate}{the new data of observations of Z at which
the conditional Kendall's tau should be estimated.}

\item{designMatrixZ}{the transformation of the ZToEstimate that
will be used as predictors. By default, no transformation is applied.}

\item{typeEstCKT}{type of estimation of the conditional Kendall's tau.}

\item{h_lambda}{the smoothing bandwidth used in the cross-validation
procedure to choose \code{lambda}.}

\item{Lambda}{the function to be applied on conditional Kendall's tau.
By default, the identity function is used.}

\item{kernel.name}{name of the kernel. Possible choices are
"Gaussian" (Gaussian kernel) and "Epa" (Epanechnikov kernel).}

\item{Kfolds_lambda}{the number of folds used in the cross-validation
procedure to choose \code{lambda}.}

\item{l_norm}{type of norm used for selection of the optimal lambda.
l_norm=1 corresponds to the sum of absolute values of differences
between predicted and estimated conditional Kendall's tau
while l_norm=2 corresponds to the sum of squares of differences.}

\item{matrixSignsPairs}{the results of a call to
\code{\link{computeMatrixSignPairs}} (if already computed).
If \code{NULL} (the default value), the \code{matrixSignsPairs}
will be computed again from the data.}

\item{progressBars}{should progress bars be displayed?
Possible values are
\itemize{
   \item \code{"none"}: no progress bar at all.

   \item \code{"global"}: only one global progress bar (default behavior)

   \item \code{"eachStep"}: uses a global progress bar + one progress bar
   for each kernel smoothing step.
}}

\item{observedX1, observedX2, observedZ}{old parameter names for \code{X1},
\code{X2}, \code{Z}. Support for this will be removed at a later version.}
}
\value{
A list with the following components
\itemize{
  \item \code{lambdaCV}: the chosen value of the
  penalization parameters \code{lambda}.

  \item \code{vectorLambda}: a vector containing the values of
  \code{lambda} that have been compared.

  \item \code{vectorMSEMean}: the estimated MSE for each value of
  \code{lambda} in \code{vectorLambda}

  \item \code{vectorMSESD}: the estimated standard deviation of the
  MSE for each \code{lambda}. It can be used to construct confidence
  intervals for estimates of the MSE given by \code{vectorMSEMean}.
}
}
\description{
In this model, three variables \eqn{X_1}, \eqn{X_2} and \eqn{Z} are observed.
We try to model the conditional Kendall's tau between \eqn{X_1} and \eqn{X_2} conditionally
to \eqn{Z=z}, as follows:
\deqn{\Lambda(\tau_{X_1, X_2 | Z = z})
= \sum_{i=1}^{p'} \beta_i \psi_i(z),}
where \eqn{\tau_{X_1, X_2 | Z = z}} is the conditional Kendall's tau
between \eqn{X_1} and \eqn{X_2} conditionally to \eqn{Z=z},
\eqn{\Lambda} is a function from \eqn{]-1, 1[]} to \eqn{R},
\eqn{(\beta_1, \dots, \beta_p)} are unknown coefficients to be estimated
and \eqn{\psi_1, \dots, \psi_{p'})} are a dictionary of functions.
To estimate \eqn{beta}, we used the penalized estimator which is defined
as the minimizer of the following criteria
\deqn{\frac{1}{2n'} \sum_{i=1}^{n'} [\Lambda(\hat\tau_{X_1, X_2 | Z = z})
- \sum_{j=1}^{p'} \beta_j \psi_j(z)]^2 + \lambda * |\beta|_1.}
This function chooses the penalization parameter \eqn{lambda}
by cross-validation.
}
\examples{
# We simulate from a conditional copula
set.seed(1)
N = 400
Z = rnorm(n = N, mean = 5, sd = 2)
conditionalTau = -0.9 + 1.8 * pnorm(Z, mean = 5, sd = 2)
simCopula = VineCopula::BiCopSim(N=N , family = 1,
    par = VineCopula::BiCopTau2Par(1 , conditionalTau ))
X1 = qnorm(simCopula[,1])
X2 = qnorm(simCopula[,2])

newZ = seq(2, 10, by = 0.1)
result <- CKT.KendallReg.LambdaCV(X1 = X1, X2 = X2, Z = Z,
                                  ZToEstimate = newZ, h_lambda = 2)

plot(x = result$vectorLambda, y = result$vectorMSEMean,
     type = "l", log = "x")

}
\references{
Derumigny, A., & Fermanian, J. D. (2020).
On Kendall’s regression.
Journal of Multivariate Analysis, 178, 104610.
}
\seealso{
the main fitting function \code{\link{CKT.kendallReg.fit}}.
}
