% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimationCKT.classif.kNN.R
\name{CKT.predict.kNN}
\alias{CKT.predict.kNN}
\title{Prediction of conditional Kendall's tau using nearest neighbors}
\usage{
CKT.predict.kNN(
  datasetPairs,
  designMatrix = datasetPairs[, 2:(ncol(datasetPairs) - 3), drop = FALSE],
  newZ,
  number_nn,
  weightsVariables = 1,
  normLp = 2,
  constantA = 1,
  partition = NULL,
  verbose = 1,
  lengthVerbose = 100,
  methodSort = "partial.sort"
)
}
\arguments{
\item{datasetPairs}{the matrix of pairs and corresponding values of the kernel
as provided by \code{\link{datasetPairs}}.}

\item{designMatrix}{the matrix of predictors.
They must have the same number of variables as \code{newZ} and
the same number of observations as \code{inputMatrix},
i.e. there should be one "multivariate observation" of the predictor for each pair.}

\item{newZ}{the matrix of predictors for which we want to
estimate the conditional Kendall's taus at these values.}

\item{number_nn}{vector of numbers of nearest neighbors to use.
If several number of neighbors are given (local) aggregation is performed
using Lepski's method on the subset determined by the \code{partition}.}

\item{weightsVariables}{optional argument to give
different weights \eqn{w_j} to each variable.}

\item{normLp}{the p in the weighted p-norm \eqn{|| x ||_p = \sum_j w_j * x_j^p}
used to determine the distance in the computation of the nearest neighbors.}

\item{constantA}{a tuning parameter that controls the adaptation.
The higher, the smoother it is; while the smaller, the least smooth it is.}

\item{partition}{used only if \code{length(number_nn) > 1}.
It is the number of subsets to consider for the local choice of the
number of nearest neighbors ; or a vector giving the id of each observations
among the subsets.
If \code{NULL}, only one set is used.}

\item{verbose}{if TRUE, this print information each \code{lengthVerbose} iterations}

\item{lengthVerbose}{number of iterations at each time for which progress is printed.}

\item{methodSort}{is the sorting method used to find the nearest neighbors.
Possible choices are
\code{ecdf} (uses the ecdf to order the points to find the neighbors)
and \code{partial.sort} uses a partial sorting algorithm.
This parameter should not matter except for the computation time.}
}
\value{
a list with two components
\itemize{
    \item \code{estimatedCKT} the estimated conditional Kendall's tau, a vector of
    the same size as the number of rows in \code{newZ};
    \item \code{vect_k_chosen} the locally selected number of nearest neighbors,
    a vector of the same size as the number of rows in \code{newZ}.
}
}
\description{
Let \eqn{X_1} and \eqn{X_2} be two random variables.
The goal of this function is to estimate the conditional Kendall's tau
(a dependence measure) between \eqn{X_1} and \eqn{X_2} given \eqn{Z=z}
for a conditioning variable \eqn{Z}.
Conditional Kendall's tau between \eqn{X_1} and \eqn{X_2} given \eqn{Z=z}
is defined as:
\deqn{P( (X_{1,1} - X_{2,1})(X_{1,2} - X_{2,2}) > 0 | Z_1 = Z_2 = z)}
\deqn{- P( (X_{1,1} - X_{2,1})(X_{1,2} - X_{2,2}) < 0 | Z_1 = Z_2 = z),}
where \eqn{(X_{1,1}, X_{1,2}, Z_1)} and \eqn{(X_{2,1}, X_{2,2}, Z_2)}
are two independent and identically distributed copies of \eqn{(X_1, X_2, Z)}.
In other words, conditional Kendall's tau is the difference
between the probabilities of observing concordant and discordant pairs
from the conditional law of \deqn{(X_1, X_2) | Z=z.}
This function estimates conditional Kendall's tau using a
\strong{nearest neighbors}. This is possible by the relationship between
estimation of conditional Kendall's tau and classification problems
(see Derumigny and Fermanian (2019)): estimation of conditional Kendall's tau
is equivalent to the prediction of concordance in the space of pairs
of observations.
}
\examples{
# We simulate from a conditional copula
set.seed(1)
N = 800
Z = rnorm(n = N, mean = 5, sd = 2)
conditionalTau = -0.9 + 1.8 * pnorm(Z, mean = 5, sd = 2)
simCopula = VineCopula::BiCopSim(N=N , family = 1,
    par = VineCopula::BiCopTau2Par(1 , conditionalTau ))
X1 = qnorm(simCopula[,1])
X2 = qnorm(simCopula[,2])

newZ = seq(2,10,by = 0.1)
datasetP = datasetPairs(X1 = X1, X2 = X2, Z = Z, h = 0.07, cut = 0.9)
estimatedCKT_knn <- CKT.predict.kNN(
  datasetPairs = datasetP,
  newZ = matrix(newZ,ncol = 1),
  number_nn = c(50,80, 100, 120,200),
  partition = 8)

# Comparison between true Kendall's tau (in black)
# and estimated Kendall's tau (in red)
trueConditionalTau = -0.9 + 1.8 * pnorm(newZ, mean = 5, sd = 2)
plot(newZ, trueConditionalTau , col="black",
   type = "l", ylim = c(-1, 1))
lines(newZ, estimatedCKT_knn$estimatedCKT, col = "red")

}
\references{
Derumigny, A., & Fermanian, J. D. (2019).
A classification point-of-view about conditional Kendall’s tau.
Computational Statistics & Data Analysis, 135, 70-94.
(Algorithm 5)
\doi{10.1016/j.csda.2019.01.013}
}
\seealso{
See also other estimators of conditional Kendall's tau:
\code{\link{CKT.fit.tree}}, \code{\link{CKT.fit.randomForest}},
\code{\link{CKT.fit.nNets}}, \code{\link{CKT.fit.randomForest}},
\code{\link{CKT.fit.GLM}}, \code{\link{CKT.kernel}},
\code{\link{CKT.kendallReg.fit}},
and the more general wrapper \code{\link{CKT.estimate}}.
}
