library(shinytest2)
test_that("Launch Shiny App without error", {
  result <- readRDS(testthat::test_path("previous-results/lm_result.rds"))
  # Launch the Shiny app
  port <- httpuv::randomPort()
  app <- AppDriver$new(comfam_shiny(result), name = "comfam_shiny", variant = platform_variant(), expect_values_screenshot_args = FALSE, load_timeout = 30000,
                       shiny_args = list(port = port))

  # Check if the app launched without errors
  expect_true(!is.null(app))

  # Data Overview
  app$set_window_size(width = 1619, height = 992)
  app$expect_screenshot()
  app$set_inputs(data_view = "Exploratory Analysis")
  app$expect_values(output = "explore_bar")
  app$set_inputs(single_feature = "thickness.left.caudal.middle.frontal")
  app$set_inputs(num_var_control_batch = "No")
  app$wait_for_value(input = "single_feature", ignore = NULL)
  app$wait_for_value(output = "batch_vi", timeout = 10)
  app$expect_values(output = c("batch_vi", "cov_vi"))
  app$expect_screenshot()

  app$set_inputs(single_cov = "AGE")
  app$set_inputs(num_var_control_batch = "Yes")
  app$wait_for_value(output = "batch_sep_control")
  app$set_inputs(overview_batch_select = c("Philips", "Siemens"))
  expect_true("Philips" %in% app$get_value(input = "overview_batch_select"))
  expect_true("Siemens" %in% app$get_value(input = "overview_batch_select"))
  app$set_inputs(num_var_control = "loess")
  app$wait_for_value(output = "cov_vi", timeout = 10)
  app$expect_values(output = "cov_vi")
  app$expect_screenshot()


  app$set_inputs(single_cov = "SEX")
  app$set_inputs(char_var_control = "boxplot with points")
  app$wait_for_value(output = "cov_vi", timeout = 10)
  app$expect_values(output = "cov_vi")
  app$expect_screenshot()

  app$set_inputs(char_var_control = "density plot")
  app$wait_for_value(output = "cov_vi", timeout = 10)
  app$expect_values(output = "cov_vi")
  app$expect_screenshot()

  # Summary
  app$set_inputs(tabselected = "2")
  app$set_inputs(type = "Plot")
  app$wait_for_value(output = "output", timeout = 10)
  app$set_inputs(text_status = "Yes")
  app$set_inputs(cov = "AGE")
  app$wait_for_value(output = "plot", timeout = 10)
  app$expect_screenshot()
  app$set_inputs(type = "Table")
  app$wait_for_value(output = "output", timeout = 10)
  app$wait_for_value(output = "table", timeout = 10)
  app$expect_screenshot()

  app$set_inputs(cov = "SEX")
  app$wait_for_value(output = "cov_table", timeout = 10)
  app$expect_screenshot()

  # Residual Plot
  app$set_inputs(tabselected = "3")
  app$set_inputs(resid_all = "No")
  app$wait_for_value(input = "resid_batch_select", ignore = NULL)
  app$set_inputs(resid_color = "Yes")
  app$set_inputs(resid_label = "Yes")
  app$set_inputs(label_angle = 18)
  app$wait_for_value(input = "label_angle", ignore = NULL)
  app$wait_for_value(output = "res_add")
  app$wait_for_value(output = "res_ml")
  app$expect_screenshot()

  app$set_inputs(resid_all = "Yes")
  app$set_inputs(resid_color = "No")
  app$set_inputs(resid_label = "No")
  app$wait_for_value(output = "res_add", timeout = 10)
  app$wait_for_value(output = "res_ml", timeout = 10)
  app$expect_screenshot()

  # Global Diagnosis
  app$set_inputs(tabselected = "4")
  app$set_inputs(PC1 = "PC1", PC2 = "PC2")
  app$set_inputs(pca_label = "Yes")
  app$set_inputs(pca_all = "No")
  app$wait_for_value(input = "pca_batch_select", ignore = NULL)
  app$wait_for_value(output = "pca")
  app$expect_screenshot()

  app$set_inputs(pca_all = "Yes")
  app$set_inputs(pca_label = "No")
  app$wait_for_value(output = "pca", timeout = 10)
  app$expect_screenshot()

  # Individual Diagnosis
  app$set_inputs(tabselected = "5")
  app$wait_for_value(output = "test_batch_table")
  app$expect_screenshot()

  app$set_inputs(test_batch = "Kruskal-Wallis")
  app$wait_for_value(output = "test_batch_table", timeout = 10)
  app$expect_screenshot()

  app$set_inputs(test_variance = "Levene's Test")
  app$wait_for_value(output = "test_variance", timeout = 10)
  app$expect_screenshot()

  app$set_inputs(test_variance = "Bartlett's Test")
  app$wait_for_value(output = "test_variance", timeout = 10)
  app$expect_screenshot()

  app$set_inputs(diag_save_path = "~/Desktop")
  app$wait_for_value(input = "diag_save_path", ignore = NULL)
  app$click("Diagnosis")
  Sys.sleep(1)
  app$expect_screenshot()

  # Harmonization
  app$set_inputs(tabselected = "6")
  Sys.sleep(1)
  app$expect_screenshot()
  app$set_inputs(com_type = "covfam")
  app$set_inputs(com_model = "gam")
  app$expect_screenshot()

  app$set_inputs(eb_control = "Yes")
  app$click("eb_check_button")
  Sys.sleep(1.8)
  app$expect_screenshot()


  app$set_inputs(save_path = "~/Desktop/harm.csv")
  app$click("ComBat")
  Sys.sleep(1.8)
  app$expect_screenshot()

  app$set_inputs(model_save_path = "~/Desktop/model.rds")
  app$click("ComBat_model")
  Sys.sleep(1.8)
  app$expect_screenshot()

})

test_that("Batch effect diagnostics plot generated correctly", {
  result <- readRDS(testthat::test_path("previous-results/lm_result.rds"))
  features <- result$info$features
  batch_density_plot <- combat_plot_gen(result, features[1], plot_name = "batch_density")
  batch_density_plot_se <- combat_plot_gen(result, features[1], batch_control = "Yes", batch_level = c("GE", "Philips", "Siemens"), plot_name = "batch_density")
  expect_true(inherits(batch_density_plot, "ggplot"))
  expect_true(inherits(batch_density_plot_se, "ggplot"))

  cov_feature_plot_num <- combat_plot_gen(result, features[1], batch_control = "No", plot_name = "cov_feature", c = "AGE")
  cov_feature_plot_num_se <- combat_plot_gen(result, features[1], batch_control = "Yes", batch_level = c("GE", "Philips", "Siemens"), plot_name = "cov_feature", c = "AGE")
  cov_feature_plot_char_1 <- combat_plot_gen(result, features[1], batch_control = "No", plot_name = "cov_feature", c = "SEX")
  cov_feature_plot_char_2 <- combat_plot_gen(result, features[1], batch_control = "No", plot_name = "cov_feature", c = "SEX", char_plot_type = "boxplot with points")
  cov_feature_plot_char_3 <- combat_plot_gen(result, features[1], batch_control = "No", plot_name = "cov_feature", c = "SEX", char_plot_type = "density plot")
  cov_feature_plot_char_1_se <- combat_plot_gen(result, features[1], batch_control = "Yes", batch_level = c("GE", "Philips", "Siemens"), plot_name = "cov_feature", c = "SEX")
  cov_feature_plot_char_2_se <- combat_plot_gen(result, features[1], batch_control = "Yes", batch_level = c("GE", "Philips", "Siemens"), plot_name = "cov_feature", c = "SEX", char_plot_type = "boxplot with points")
  expect_true(inherits(cov_feature_plot_num, "ggplot"))
  expect_true(inherits(cov_feature_plot_num_se, "ggplot"))
  expect_true(inherits(cov_feature_plot_char_1, "ggplot"))
  expect_true(inherits(cov_feature_plot_char_2, "ggplot"))
  expect_true(inherits(cov_feature_plot_char_3, "ggplot"))
  expect_true(inherits(cov_feature_plot_char_1_se, "ggplot"))
  expect_true(inherits(cov_feature_plot_char_1_se, "ggplot"))

  batch_summary_plot <- combat_plot_gen(result, plot_name = "batch_summary")
  batch_summary_plot_text <- combat_plot_gen(result, plot_name = "batch_summary", text_status = "Yes")
  expect_true(inherits(batch_summary_plot, "ggplot"))
  expect_true(inherits(batch_summary_plot_text, "ggplot"))

  cov_distribution_plot_num <- combat_plot_gen(result, plot_name = "cov_distribution", c = "AGE")
  cov_distribution_plot_char <- combat_plot_gen(result, plot_name = "cov_distribution", c = "SEX")
  cov_distribution_plot_char_text <- combat_plot_gen(result, plot_name = "cov_distribution", c = "SEX", text_status = "Yes")
  expect_true(inherits(cov_distribution_plot_num, "ggplot"))
  expect_true(inherits(cov_distribution_plot_char, "ggplot"))
  expect_true(inherits(cov_distribution_plot_char_text, "ggplot"))

  resid_add_plot <- combat_plot_gen(result, f = features[1], batch_control = "No", plot_name = "resid_add", color = "No", label = "No", angle = 0)
  resid_add_plot_color <- combat_plot_gen(result, f = features[1], batch_control = "No", plot_name = "resid_add", color = "Yes", label = "No", angle = 0)
  resid_add_plot_label <- combat_plot_gen(result, f = features[1], batch_control = "No", plot_name = "resid_add", color = "Yes", label = "Yes", angle = 0)
  resid_add_plot_label_se <- combat_plot_gen(result, f = features[1], batch_control = "Yes", batch_level = c("Philips", "Siemens"), plot_name = "resid_add", color = "Yes", label = "Yes", angle = 0)
  expect_true(inherits(resid_add_plot, "ggplot"))
  expect_true(inherits(resid_add_plot_color, "ggplot"))
  expect_true(inherits(resid_add_plot_label, "ggplot"))
  expect_true(inherits(resid_add_plot_label_se, "ggplot"))

  resid_mul_plot <- combat_plot_gen(result, f = features[1], batch_control = "No", plot_name = "resid_mul", color = "No", label = "No", angle = 0)
  resid_mul_plot_color <- combat_plot_gen(result, f = features[1], batch_control = "No", plot_name = "resid_mul", color = "Yes", label = "No", angle = 0)
  resid_mul_plot_label <- combat_plot_gen(result, f = features[1], batch_control = "No", plot_name = "resid_mul", color = "Yes", label = "Yes", angle = 0)
  resid_mul_plot_label_se <- combat_plot_gen(result, f = features[1], batch_control = "Yes", batch_level = c("Philips", "Siemens"), plot_name = "resid_mul", color = "Yes", label = "Yes", angle = 0)
  expect_true(inherits(resid_mul_plot, "ggplot"))
  expect_true(inherits(resid_mul_plot_color, "ggplot"))
  expect_true(inherits(resid_mul_plot_label, "ggplot"))
  expect_true(inherits(resid_mul_plot_label_se, "ggplot"))

  pca_plot <- combat_plot_gen(result, batch_control = "No", plot_name = "pca", PC1 = "PC1", PC2 = "PC2")
  pca_plot_label <- combat_plot_gen(result, batch_control = "No", plot_name = "pca", PC1 = "PC1", PC2 = "PC2", label = "Yes")
  pca_plot_label_se <- combat_plot_gen(result, batch_control = "Yes", batch_level = c("Philips", "Siemens"), plot_name = "pca", PC1 = "PC1", PC2 = "PC2", label = "Yes")
  expect_true(inherits(pca_plot, "ggplot"))
  expect_true(inherits(pca_plot_label, "ggplot"))
  expect_true(inherits(pca_plot_label_se, "ggplot"))

  tsne_plot <- combat_plot_gen(result, batch_control = "No", plot_name = "tsne")
  tsne_plot_label <- combat_plot_gen(result, batch_control = "No", plot_name = "tsne", label = "Yes")
  tsne_plot_label_se <- combat_plot_gen(result, batch_control = "Yes", batch_level = c("Philips", "Siemens"), plot_name = "tsne", label = "Yes")
  expect_true(inherits(tsne_plot, "ggplot"))
  expect_true(inherits(tsne_plot_label, "ggplot"))
  expect_true(inherits(tsne_plot_label_se, "ggplot"))

  eb_df <- combat_harm(eb_check = TRUE, result, type = "lm", interaction = "timedays:DIAGNOSIS")
  eb_location_plot <- combat_plot_gen(result, eb = TRUE, eb_df = eb_df, batch_control = "No", plot_name = "eb_location")
  eb_location_plot_non_eb <- combat_plot_gen(result, eb = FALSE, eb_df = eb_df, batch_control = "No", plot_name = "eb_location")
  eb_location_plot_se <- combat_plot_gen(result, eb = TRUE, eb_df = eb_df, batch_control = "Yes", batch_level = "GE", plot_name = "eb_location")
  expect_true(inherits(eb_location_plot, "ggplot"))
  expect_true(inherits(eb_location_plot_non_eb, "ggplot"))
  expect_true(inherits(eb_location_plot_se, "ggplot"))

  eb_scale_plot <- combat_plot_gen(result, eb = TRUE, eb_df = eb_df, batch_control = "No", plot_name = "eb_scale")
  eb_scale_plot_non_eb <- combat_plot_gen(result, eb = FALSE, eb_df = eb_df, batch_control = "No", plot_name = "eb_scale")
  eb_scale_plot_se <- combat_plot_gen(result, eb = TRUE, eb_df = eb_df, batch_control = "Yes", batch_level = "GE", plot_name = "eb_scale")
  expect_true(inherits(eb_scale_plot, "ggplot"))
  expect_true(inherits(eb_scale_plot_non_eb, "ggplot"))
  expect_true(inherits(eb_scale_plot_se, "ggplot"))
})


test_that("Batch effect diagnostic table generated correctly", {
  result <- readRDS(testthat::test_path("previous-results/lm_result.rds"))
  features <- result$info$features
  data_overview_table <- combat_table_gen(result, table_name = "data_overview")
  expect_true("datatables" %in% class(data_overview_table))
  expect_true(inherits(data_overview_table, "htmlwidget"))
  exploratory_table <- combat_table_gen(result, table_name = "exploratory_analysis", f = features[1])
  expect_true("datatables" %in% class(exploratory_table))
  expect_true(inherits(exploratory_table, "htmlwidget"))
  summary_table <- combat_table_gen(result, table_name = "summary_df")
  expect_true("datatables" %in% class(summary_table))
  expect_true(inherits(summary_table, "htmlwidget"))
  covariate_table_num <- combat_table_gen(result, table_name = "cov_table", c = "AGE")
  expect_true("datatables" %in% class(covariate_table_num))
  expect_true(inherits(covariate_table_num, "htmlwidget"))
  covariate_table_char <- combat_table_gen(result, table_name = "cov_table", c = "SEX")
  expect_true("datatables" %in% class(covariate_table_char))
  expect_true(inherits(covariate_table_char, "htmlwidget"))
  pca_table <- combat_table_gen(result, table_name = "pc_variance", PC1 = "PC1", PC2 = "PC2")
  expect_true("datatables" %in% class(pca_table))
  expect_true(inherits(pca_table, "htmlwidget"))
  mdmr_table <- combat_table_gen(result, table_name = "mdmr")
  expect_true("datatables" %in% class(mdmr_table))
  expect_true(inherits(mdmr_table, "htmlwidget"))
  kr_table <- combat_table_gen(result, table_name = "kenward_roger")
  expect_true("datatables" %in% class(kr_table))
  expect_true(inherits(kr_table, "htmlwidget"))
  anova_table <- combat_table_gen(result, table_name = "anova")
  expect_true("datatables" %in% class(anova_table))
  expect_true(inherits(anova_table, "htmlwidget"))
  kw_table <- combat_table_gen(result, table_name = "kruskal_wallis")
  expect_true("datatables" %in% class(kw_table))
  expect_true(inherits(kw_table, "htmlwidget"))
  fk_table <- combat_table_gen(result, table_name = "fligner_killeen")
  expect_true("datatables" %in% class(fk_table))
  expect_true(inherits(fk_table, "htmlwidget"))
  levenes_table <- combat_table_gen(result, table_name = "levenes")
  expect_true("datatables" %in% class(levenes_table))
  expect_true(inherits(levenes_table, "htmlwidget"))
  bartletts_table <- combat_table_gen(result, table_name = "bartletts")
  expect_true("datatables" %in% class(bartletts_table))
  expect_true(inherits(bartletts_table, "htmlwidget"))
})
