#' Selecting optimal hyperparameters for the conformal prediction set
#'
#' \code{hyperparam.torus} selects optimal hyperparamters for constructing the conformal prediction
#'   set, based on the type of postulated model and the criterion.
#'
#' @param data n x d matrix of toroidal data on \eqn{[0, 2\pi)^d}
#'   or \eqn{[-\pi, \pi)^d}
#' @param icp.torus.objects list whose elements are icp.torus objects, generated by
#'   \code{icp.torus.score}
#' @param option A string. One of "elbow", "risk", "AIC", or "BIC", which determines the
#' criterion for the model selection. "risk" is based on the negative log-likelihood, "AIC"
#' for the Akaike Information Criterion, and "BIC" for the Bayesian Infor- mation Criterion.
#' "elbow" is based on minimizing the criterion used in Jung, et. al.(2021).
#' @param split.id a n-dimensinal vector consisting of values 1 (estimation)
#'   and 2(evaluation)
#' @param Jvec either a scalar or a vector for the number of mixture components. Default value is 3:35.
#' @param kvec either a scalar or a vector for the concentration parameter. Default value is 20:100.
#' @param alphavec either a scalar or a vector, or even \code{NULL} for the levels.
#'   Default value is \code{NULL}. If \code{NULL}, then \code{alphavec} is
#'   automatically generated as a sequence from 0 to \code{alpha.lim}.
#' @param alpha.lim a positive number lower than 1, which is the upper bound of
#'   Default is 0.15.
#' @param method A string. One of "all", "kde", "mixture", and "kmeans" which
#'   determines the model or estimation methods. If "kde", the model is based
#'   on the kernel density estimates. It supports the kde-based conformity score
#'   only. If "mixutre", the model is based on the von Mises mixture, fitted
#'   with an EM algorithm. It supports the von Mises mixture and its variants
#'   based conformity scores. If "kmeans", the model is also based on the von
#'   Mises mixture, but the parameter estimation is implemented with the
#'   elliptical k-means algorithm illustrated in Appendix. It supports the
#'   log-max-mixture based conformity score only. Default is "all". If the
#'   dimension of data space is greater than 2, only "kmeans" is supported.
#' @param mixturefitmethod A string. One of "circular", "axis-aligned", and
#'   "general" which determines the constraint of the EM fitting. Default is
#'   "axis-aligned". This argument only works for \code{method = "mixture"}.
#' @param kmeansfitmethod A string. One of "general", ellipsoids",
#'   "heterogeneous-circular" or "homogeneous-circular". If "general", the
#'   elliptical k-means algorithm with no constraint is used. If "ellipsoids",
#'   only the one iteration of the algorithm is used. If"heterogeneous-circular",
#'   the same as above, but with the constraint that ellipsoids must be spheres.
#'   If "homogeneous-circular", the same as above but the radii of the spheres are
#'   identical. This argument only works for method = "kmeans".
#' @param init determine the initial parameter of "kmeans" method,
#'   for option "general". Must be "kmeans" or "hierarchical".
#'   If "kmeans", the initial parameters are obtained with extrinsic kmeans
#'   method.
#'   If "hierarchical", the initial parameters are obtained with hierarchical
#'   clustering method.
#'   Default is "kmeans".
#' @param eval.point N x N numeric matrix on \eqn{[0, 2\pi)^2}. Default input is
#'   \code{grid.torus}.
#' @param additional.condition boolean index.
#'   If \code{TRUE}, a singular matrix will be altered to the scaled identity.
#' @param kmax the maximal number of kappa. If estimated kappa is
#'   larger than \code{kmax}, then put kappa as \code{kmax}.
#' @param THRESHOLD number for difference between updating and
#'   updated parameters. Default is 1e-10.
#' @param maxiter the maximal number of iteration. Default is 200.
#' @param verbose boolean index, which indicates whether display
#'   additional details as to what the algorithm is doing or
#'   how many loops are done. Moreover, if \code{additional.condition} is
#'   \code{TRUE}, the warning message will be reported.
#' @return returns a list object which contains \code{data.frame} objects for
#'   the evaluated criterion corresponding to each hyperparameter,
#'   selected hyperparameters based on the designated criterion, and
#'   an \code{icp.torus} object based the selected hyperparameters.
#' @export
#' @references S. Jung, K. Park, and B. Kim (2021),
#'   "Clustering on the torus by conformal prediction",
#'   Akaike (1974), "A new look at the statistical model identification",
#'   Schwarz, Gideon E. (1978), "Estimating the dimension of a model"
#' @examples
#' \donttest{
#' data <- toydata2[, 1:2]
#' n <- nrow(data)
#' split.id <- rep(2, n)
#' split.id[sample(n, floor(n/2))] <- 1
#' Jvec <- 3:35
#' icp.torus.objects <- list()
#' for (j in Jvec){
#'   icp.torus.objects[[j]] <- icp.torus.score(data, split.id = split.id, method = "kmeans",
#'                                             kmeansfitmethod = "ge", init = "h",
#'                                             param = list(J = j), verbose = TRUE)
#' }
#' hyperparam.torus(data, icp.torus.objects, option = "risk")
#' }

hyperparam.torus <- function(data, icp.torus.objects = NULL,
                             option = c("elbow", "risk", "AIC", "BIC"),
                             split.id = NULL, Jvec = 3:35, kvec = 20:100, alphavec = NULL,
                             alpha.lim = 0.15 ,method = c("kde", "mixture", "kmeans"),
                             mixturefitmethod = c("circular", "axis-aligned", "general", "Bayesian"),
                             kmeansfitmethod = c("homogeneous-circular", "heterogeneous-circular",
                                                 "ellipsoids", "general"),
                             init = c("kmeans", "hierarchical"),
                             eval.point = NULL, additional.condition = TRUE,
                             kmax = 500, THRESHOLD = 1e-10, maxiter = 200,
                             verbose = FALSE){

  if (is.null(data)) {stop("data must be input.")}

  # designating the split.id

  if (!is.null(icp.torus.objects)) {
    Jvec <- c()
    for (j in 1:length(icp.torus.objects)){
      if (!is.null(icp.torus.objects[[j]])){
        Jvec <- c(Jvec, j)
      }
    }
    split.id <- icp.torus.objects[[Jvec[1]]]$split.id

    if(!is.null(icp.torus.objects[[Jvec[1]]]$mixture)) {
      method <- "mixture"
      mixturefitmethod <- icp.torus.objects[[Jvec[1]]]$mixture$fittingmethod
    } else if(!is.null(icp.torus.objects[[Jvec[1]]]$kmeans)) {
      method <- "kmeans"
      kmeansfitmethod <- icp.torus.objects[[Jvec[1]]]$kmeans$fittingmethod
    } else {method <- "kde"}
  }

  # d : dimension
  d <- ncol(data)

  # if d <= 2, default is elbow based criterion. If d > 2, default is risk.
  if (is.null(option)) {
    option <- ifelse(d <= 2, "elbow", "risk")
  }

  if (!is.matrix(data)) {data <- as.matrix(data)}
  data <- on.torus(data)

  if (is.null(method)) {stop("method must be designated.")}
  if (is.null(mixturefitmethod)) {mixturefitmethod <- "axis-aligned"}
  if (is.null(kmeansfitmethod)) {kmeansfitmethod <- "general"}
  if (is.null(init)) {init <- "kmeans"}
  if (alpha.lim > 1) {stop("alpha.lim must be less than 1.")}

  option <- match.arg(option)
  method <- match.arg(method)
  mixturefitmethod <- match.arg(mixturefitmethod)
  kmeansfitmethod <- match.arg(kmeansfitmethod)
  init <- match.arg(init)

  if (option != "elbow" && method == "kde")
    {stop("method kde is only supported by option elbow.")}
  if (method != "kmeans" && d > 2) {stop("The method is not supported in high(d >= 3) dimensional cases.")}
  # preparing output : containing methods and fitting methods
  output <- list()

  if (method == "kde") {fitmethod <- NULL}
  else if (method == "mixture") {fitmethod <- mixturefitmethod}
  else {fitmethod <- kmeansfitmethod}
  output$method <- c(method, fitmethod)
  output$option <- option

  # initializing icp.torus.objects for given Jvec
  if (is.null(icp.torus.objects)){
    # preparing conformity scores for different J
    icp.torus.objects <- list()
    if (method != "kde"){
      for (j in Jvec){
        icp.torus.objects[[j]] <- icp.torus.score(data, split.id = split.id,
                                                  method = method,
                                                  mixturefitmethod = mixturefitmethod,
                                                  kmeansfitmethod = kmeansfitmethod,
                                                  init = init,
                                                  additional.condition = additional.condition,
                                                  param = list(J = j), THRESHOLD = THRESHOLD,
                                                  maxiter = maxiter, verbose = verbose)
      }
      n2 <- icp.torus.objects[[j]]$n2
    } else {
      # preparing conformity scores for different concentrations
      for (k in kvec){
        icp.torus.objects[[k]] <- icp.torus.score(data, split.id = split.id, method = method,
                                                  param = list(concentration = k))
      }
      n2 <- icp.torus.objects[[k]]$n2
    }
  } else { n2 <- icp.torus.objects[[Jvec[1]]]$n2 }

  # initializing alphavec
  if (is.null(alphavec)) {alphavec <- 1:floor(n2 * alpha.lim) / n2}

  # criterion based on elbow -----------------------------------
  if (option == "elbow"){
    if (d > 3) {warning("Highly drepreciated for high dimensional case (d >= 3).", immediate. = TRUE)}

    # generating grid points if eval.point == NULL : sparse when d is large.
    grid.size <- ifelse(d == 2, 100, floor(10^(6/d)))
    if (is.null(eval.point)) { eval.point <- grid.torus(d = d, grid.size = grid.size)}

    # 1. kmeans -----------------------------------------------------
    if (method == "kmeans"){
      N <- length(alphavec)
      Mvec <- vector("numeric", length = N)
      out <- data.frame()

      for (j in 1:length(icp.torus.objects)){
        if (is.null(icp.torus.objects[[j]])) {next}
        inclusion.test <- icp.torus.eval(icp.torus.objects[[j]], level = alphavec, eval.point = eval.point)
        Mvec <- colSums(inclusion.test$Chat_kmeans)/nrow(eval.point)

        out <- rbind(out, data.frame(J = j, alpha = alphavec, mu = Mvec, criterion = alphavec +  Mvec))
      }

      out.index <- which.min(out$criterion)
      output$results <- out
      output$optim$icp.torus <- icp.torus.objects[[out[out.index, 1]]]
      output$optim$hyperparam <- unlist(out[out.index, 1:2])
      return(output)

      # 2. mixture ----------------------------------------------------------------
    } else if (method == "mixture") {
      N <- length(alphavec)
      Mvec <- vector("numeric", length = N)
      out <- data.frame()

      for (j in 1:length(icp.torus.objects)){
        if (is.null(icp.torus.objects[[j]])) {next}
        inclusion.test <- icp.torus.eval(icp.torus.objects[[j]], level = alphavec, eval.point = eval.point)
        Mvec <- colSums(inclusion.test$Chat_mix)/nrow(eval.point)

        out <- rbind(out, data.frame(J = j, alpha = alphavec, mu = Mvec, criterion = alphavec +  Mvec))
      }

      out.index <- which.min(out$criterion)
      output$results <- out
      output$optim$icp.torus <- icp.torus.objects[[out[out.index, 1]]]
      output$optim$hyperparam <- unlist(out[out.index, 1:2])
      return(output)
    }
    # 3. kde --------------------------------------------------
    else {
      N <- length(alphavec)
      Mvec <- vector("numeric", length = N)
      out <- data.frame()

      for (k in 1:length(icp.torus.objects)){
        if (is.null(icp.torus.objects[[k]])) {next}
        inclusion.test <- icp.torus.eval(icp.torus.objects[[k]], level = alphavec, eval.point = eval.point)
        Mvec <- colSums(inclusion.test$Chat_kde)/nrow(eval.point)

        out <- rbind(out, data.frame(k = k, alpha = alphavec, mu = Mvec, criterion = alphavec + Mvec))
      }

      out.index <- which.min(out$criterion)
      output$results <- out
      output$optim$icp.torus <- icp.torus.objects[[out[out.index, 1]]]
      output$optim$hyperparam <- unlist(out[out.index, 1:2])
      return(output)
    }
  }

  # criterion based on information criteria ----------------------
  if (sum(option == c("AIC", "BIC", "risk")) == 1){

    results.J <- hyperparam.J(data = data, icp.torus.objects = icp.torus.objects,
                              option = option)

    output$IC.results <- results.J$IC.results
    Jhat <- results.J$Jhat
    icp.torus <- results.J$icp.torus

    results.alpha <- hyperparam.alpha(icp.torus, alphavec = alphavec, alpha.lim = alpha.lim)
    output$alpha.results <- results.alpha$alpha.results
    alphahat <- results.alpha$alphahat

    hyperparam <- c(Jhat, alphahat)
    names(hyperparam) <- c("J", "alpha")

    output$optim$hyperparam <- hyperparam
    output$optim$icp.torus <- icp.torus

    return(output)
  }
}
