\name{ahist}
\alias{ahist}
\title{Adaptive Histograms by Optimal Univariate k-Means Clustering}
\description{
Generate histograms that are adaptive to patterns in data. The number and widths of histogram bins are automatically adjusted to the data and thus the bins are unlikely of equal width. The breaks are computed from optimal univariate \var{k}-means clusters of the input data.
}
\usage{
ahist(x, k=c(1,9), plot = TRUE, xlab = deparse(substitute(x)),
  main = paste("Adaptive histogram of", deparse(substitute(x))),
  col = NULL, lwd = graphics::par("lwd"), col.stick = "gray", lwd.stick = 1,
  \dots)
}
\arguments{
  \item{x}{a numeric vector of data. All \code{NA} elements must be removed from \code{x} before calling this function. The function will run faster on sorted \code{x} (in non-decreasing order) than an unsorted input.}

  \item{k}{either an exact integer number of bins/clusters, or a vector of length two specifying the minimum and maximum numbers of bins/clusters to be examined. The default is \code{c(1,9)}. When \code{k} is a range, the actual number of clusters is determined by Bayesian information criterion.}

 \item{plot}{a logical. If \code{TRUE}, the histogram will be plotted.}

 \item{xlab}{a character string. The x-axis label for the plot.}

 \item{main}{a character string. The title for the plot.}

 \item{col}{a character string. The fill color of the histogram bars.}

 \item{lwd}{a numeric value. The line width of the border of the histogram bars}

 \item{col.stick}{a character string. The color of the sticks above the x-axis. See Details.}

 \item{lwd.stick}{a numeric value. The line width of the sticks above the x-axis. See Details.}

 \item{...}{additional arguments to be passed to \code{\link{hist}} or \code{\link{plot.histogram}}.}
}

\author{
	Joe Song
}

\details{
The number of histogram bins is the optimal number of univariate k-means clusters estimated using Bayesian information criterion evaluated on Gaussian mixture models fitted to the input data in \code{x}. Breaks in the histogram are the midpoints between clusters identified by optimal univariate k-means (\code{\link{Ckmeans.1d.dp}}). The histogram is by default plotted using the \code{\link{plot.histogram}} method. The plot can be optionally disabled with the \code{plot=FALSE} argument. The original input data are shown as sticks just above the x-axis.
}

\value{
An object of class \code{histogram} defined in \code{\link{hist}}. It has a S3 \code{plot} method \code{\link{plot.histogram}}.
}

\references{
Wang, H. and Song, M. (2011) Ckmeans.1d.dp: optimal \var{k}-means clustering in one dimension by dynamic programming. \emph{The R Journal} \bold{3}(2), 29--33. Retrieved from \url{http://journal.r-project.org/archive/2011-2/RJournal_2011-2_Wang+Song.pdf}
}

\seealso{
  \code{\link{hist}} in package \pkg{\link{graphics}}.
}

\examples{
# Example 1: plot an adaptive histogram from data generated by
#   a Gaussian mixture model with three components using a given
#   number of bins
x <- c(rnorm(40, mean=-2, sd=0.3),
       rnorm(45, mean=1, sd=0.1),
       rnorm(70, mean=3, sd=0.2))
ahist(x, k=9, col="lavender", col.stick="salmon",
      sub=paste("n =", length(x)), lwd=2,
      main="Example 1. Gaussian mixture model with 3 components\n(on average 3 bins per component)")

# Example 2: plot an adaptive histogram from data generated by
#   a Gaussian mixture model with three components
ahist(x, col="lightblue", sub=paste("n =", length(x)),
      col.stick="salmon", lwd=2,
      main="Example 2. Gaussian mixture model with 3 components\n(one bin per component)")

# Example 3: The DNase data frame has 176 rows and 3 columns of
#   data obtained during development of an ELISA assay for the
#   recombinant protein DNase in rat serum.

data(DNase)
ahist(DNase$density, col="gray", col.stick="salmon",
      sub=paste("n =", length(x)), lwd=2,
      xlab="Optical density of protein DNase",
      main="Example 3. Elisa assay of DNase in rat serum")
}
