% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/myfunctions.R
\name{SimulateCVR}
\alias{SimulateCVR}
\title{Generate simulation data.}
\usage{
SimulateCVR(family = c("gaussian", "binomial", "poisson"), n = 100,  
         rank = 4, p1 = 50, p2 = 70, pnz = 10, sigmax = 0.2,   
         sigmay = 0.5, beta = c(2, 1, 0, 0), standardization = TRUE)
}
\arguments{
\item{family}{Type of response. \code{"gaussian"} for continuous response, \code{"binomial"}
for binary response, and \code{"poisson"} for Poisson response. The default is \code{"gaussian"}.}

\item{n}{Number of rows. The default is 100.}

\item{rank}{Number of latent factors generating the covariates.  The default is 4.}

\item{p1}{Number of variables in X1. The default is 50.}

\item{p2}{Number of variables in X2. The default is 70.}

\item{pnz}{Number of variables in X1 and X2 related to the signal. The default is 10.}

\item{sigmax}{Standard deviation of normal noise in X1 and X2. The default is 0.2.}

\item{sigmay}{Standard deviation of normal noise in Y. Only used when the response is Gaussian. The default is 0.5.}

\item{beta}{Numeric vector, the coefficients used to generate respose from the latent factors. The default is c(2, 1, 0, 0).}

\item{standardization}{Logical. If TRUE, standardize X1 and X2 before output. The default is TRUE.}
}
\value{
\item{X1, X2}{The two sets of covariates with dimensions  n*p1 and n*p2 respectively.}

\item{y}{The  response vector with length   n.}

\item{U}{The true latent factor matrix with dimension  n*rank.}

\item{beta}{The coefficients used to generate response from \code{U}. The length is rank.}

\item{V1, V2}{The true loading matrices for X1 and X2 with dimensions  p1*rank and p2*rank. The first \code{pnz} rows are nonzero.}
}
\description{
Generate two sets of covariates and an univariate response driven by several latent factors.
}
\details{
The latent factors in U are randomly generated normal vectors, 

         \eqn{X_1 = U*V_1 + \sigma_x*E_1, X_2 = U*V_2 + \sigma_x*E_2, E_1, E_2}{%
         X1 = U*V1 + sigmax*E1, X2 = U*V2 + sigmax*E2, E1, E2} are N(0,1) noise matrices.
         
         The nonzero entries of \eqn{V_1}{%
         V1} and \eqn{V_2}{%
         V2} are generated from Uniform([-1,-0.5]U[0.5,1]).
                  
         For Gaussian response, 
         
         \eqn{y = U*\beta + \sigma_y*e_y, e_y}{%
         y = U*\beta + sigmay*ey, ey} is N(0,1) noise vector,
         
         for binary response, 
         
         \eqn{y \sim rbinom(n, 1, 1/(1 + \exp(-U*\beta)))},
         
         and for Poisson response,
         
         \eqn{y \sim rpois(n, \exp(U*\beta))}.
         
         See the reference for more details.
}
\examples{
 set.seed(42)
 mydata <- SimulateCVR(family = "g", n = 100, rank = 4, p1 = 50, p2 = 70, 
                   pnz = 10, beta = c(2, 1, 0, 0))
 X1 <- mydata$X1
 X2 <- mydata$X2
 Xlist <- list(X1 = X1, X2 = X2); 
 Y <- mydata$y
 opts <- list(standardization = FALSE, maxIters = 300, tol = 0.005)
 ## use sparse CCA solution as initial values, see SparseCCA()
 Wini <- SparseCCA(X1, X2, 4, 0.7, 0.7) 
 ## perform CVR with fixed eta and lambda, see cvrsolver()
 fit <- cvrsolver(Y, Xlist, rank = 4, eta = 0.5, Lam = c(1, 1), 
                 family = "gaussian", Wini, penalty = "GL1", opts)
 ## check sparsity recovery
 fit$W[[1]]; 
 fit$W[[2]];
 ## check orthogonality
 X1W1 <- X1 \%*\% fit$W[[1]]; 
 t(X1W1) \%*\% X1W1
}
\references{
Chongliang Luo, Jin Liu, Dipak D. Dey and Kun Chen (2016) Canonical variate regression. 
              Biostatistics, doi: 10.1093/biostatistics/kxw001.
}
\seealso{
\code{\link{CVR}}, \code{\link{cvrsolver}}.
}
\author{
Chongliang Luo, Kun Chen.
}
