\name{clme.em}
\alias{clme.em}
\alias{clme.em.all}
\alias{clme.em.fixed}
\alias{clme.em.mixed}

\title{ Constrained Expectation-Maximization Algorithm }
\description{ Implements an expectation-maximization (EM) algorithm under inequality constraints to estimate parameters and compute a test statistic.  }

\usage{
clme.em.all(method, Y, X1, X2 = NULL, U = NULL, Nks = dim(X1)[1],
        Qs = dim(U)[2] , constraints, mq.phi = NULL,
        tsf=lrt.stat , tsf.ind=w.stat.ind , pav.alg, hp=FALSE , em.iter=500,
        em.eps = sqrt(.Machine$double.eps), verbose = FALSE)
clme.em.fixed(method, Y, X1, X2 = NULL, U = NULL, Nks = dim(X1)[1],
        Qs = dim(U)[2] , constraints, mq.phi = NULL,
        tsf=lrt.stat , tsf.ind=w.stat.ind , pav.alg, hp=FALSE , em.iter=500,
        em.eps = sqrt(.Machine$double.eps), verbose = FALSE)
clme.em.mixed(method, Y, X1, X2 = NULL, U = NULL, Nks = dim(X1)[1],
        Qs = dim(U)[2] , constraints, mq.phi = NULL,
        tsf=lrt.stat , tsf.ind=w.stat.ind , pav.alg, hp=FALSE , em.iter=500,
        em.eps = sqrt(.Machine$double.eps), verbose = FALSE)
}

\arguments{
  \item{method}{ method for applying order constraints. Supported are \dQuote{QPE} and \dQuote{PAVA} }
  \item{Y}{ \eqn{N \times 1}{Nx1} vector of response data. }
  \item{X1}{ \eqn{N \times p_1}{Nxp1} design matrix. }
  \item{X2}{ optional. \eqn{N \times p_2}{Nxp2} matrix of covariates. }
  \item{U}{ optional. \eqn{N \times c}{Nxc} matrix of random effects. }
  \item{Nks}{ optional. \eqn{K \times 1}{Kx1} vector of group sizes. }
  \item{Qs}{ optional. \eqn{Q \times 1}{Qx1} vector of group sizes for random effects.}
  \item{constraints}{optional. List containing the constraints, defaults to empty list. See Details for further information. }
  \item{mq.phi}{ optional. MINQUE estimates of variance parameters.}
  \item{tsf}{ Function to calculate the test statistic. Defaults to likelihood ratio test statistic (\code{\link{lrt.stat}}). See Details for further information. }
  \item{tsf.ind}{ Function to calculate the test statistic for individual constrats. Defaults to Williams type test statistic (\code{\link{w.stat.ind}}). See Details for further information. }
  \item{pav.alg}{ algorithm to apply PAVA constraints. Only required when custom \code{constraints} are specified and \code{method=}\dQuote{PAVA}. Ignored when \code{method=}\dQuote{QPE}. See Details for further information. }
  \item{hp}{logical to indicate whether weights for PAVA should be use the full covariance matrix or the diagonal elements. Ignored when \code{method=}\dQuote{QPE}.}
  \item{em.eps}{ optional. Criterion for convergence for the EM algorithm. }
  \item{em.iter}{ optional. Maximum number of iterations permitted for the EM algorithm. }
  \item{verbose}{ If \code{TRUE}, function prints messages on progress of the EM algorithm. }
}

\details{
Constraints can be implemented either through Restricted Maximum Likelihood Estimation (by setting \code{method=}\dQuote{QPE}) or the Pool Adjacent Violators Algorithm (\code{method=}\dQuote{PAVA}). For PAVA, the user must also specify an appropriate algorithm. Several default PAVA algorithms are included in the package \pkg{CLME}. See \code{\link{pava.functions}} for more details. 

Argument \code{constraints} is a list including at least the elements \code{A} and \code{B}. This argument can be generated by function \code{\link{create.constraints}}.

}

\value{
The function returns a list with the elements:
\item{\code{theta}}{ vector of theta.}
\item{\code{ssq}}{ vector of estimates of variance terms.}
\item{\code{tsq}}{ vector of estimates of variance terms for the random effects.}
\item{\code{cov.theta}}{ the covariance matrix of the unconstrained theta coefficients. }
\item{\code{ts.glb}}{ test statistic for the global hypothesis.}
\item{\code{ts.ind}}{ vector of test statistics for each of the constraints.}
}

\references{

Farnan, L., Ivanova, A., and Peddada, S. D. (2014).
  Linear Mixed Efects Models under Inequality Constraints with Applications.
  \emph{PLOS ONE}, 9(1). e84778. doi: 10.1371/journal.pone.0084778
  \url{http://www.plosone.org/article/info:doi/10.1371/journal.pone.0084778}

}

\author{ Casey M. Jelsema <casey.jelsema@nih.gov> }

\note{
There are few error catches in these functions. If only the EM estimates are desired, users are recommended to run \code{\link{constrained.lme}} setting \code{M=0}.

By default, homogeneous variances are assumed for the residuals and (if included) random effects. Heterogeneity can be induced using the arguments \code{Nks} and \code{Qs}, which refer to the vectors \eqn{ (n_{1}, n_{2}, \ldots, n_{k}) }{(n1, n2 ,... , nk)} and \eqn{ (c_{1}, c_{2}, \ldots, c_{q}) }{(c1, c2 ,... , cq)}, respectively. See \code{\link{CLME-package}} for further explanation the model and these values.

The function \code{clme.em.all} is the general case, \code{clme.em.fixed} and \code{clme.em.mixed} omit many evaluations of \code{if} commands to detect whether the model is fixed or mixed effects. All arguments are identical. In the process of running \code{\link{constrained.lme}}, the proper EM function will be automatically selected.
}

\seealso{
\code{\link{CLME-package}},
\code{\link{constrained.lme}},
\code{\link{pava.functions}},
\code{\link{create.constraints}},
\code{\link{w.stat}} 
}

\examples{

set.seed( 42 )

n  <- 5
P1 <- 5

X1 <- diag(P1) \%x\% rep(1,n)
X2 <- as.matrix( rep(1,P1) \%x\% runif(n , 0,2) )
U  <- rep(1,P1) \%x\% diag(n)
X  <- as.matrix( cbind(X1,X2) )

tsq <- 1
ssq <- 0.7

Nks <- dim(X1)[1]
Qs  <- dim(U)[2]

xi <- rnorm( sum(Qs)  , 0 , rep(sqrt(tsq) , Qs)  )
ep <- rnorm( sum(Nks) , 0 , rep(sqrt(ssq) , Nks) )  

thetas <- c(2 , 3 , 3, 3 , 4 , 2 )
Y      <- X\%*\%thetas + U\%*\%xi + ep
const  <- create.constraints( X1=X1 , X2=X2 , 
          constraints=list(order='simple' , decreasing=FALSE) )

## Two runs of clme.em with different options

clme.out1 <- clme.em.all( method='QPE', Y=Y, X1=X1, X2=X2, U=U, 
                     constraints=const )

clme.out2 <- clme.em.mixed( method='QPE', Y=Y, X1=X1, X2=X2, U=U, 
                     constraints=const )
                    
                    
clme.out3 <- clme.em.all( method='PAVA', Y=Y, X1=X1, X2=X2, U=U,
                     constraints=const , tsf=w.stat ,
                    pav.alg=pava.simple.order )


}

