\name{transfer}
\alias{transfer}
\alias{draw.transfer}
\alias{feldspar}
\alias{apc}
\title{Mass Transfer Calculations}
\description{
Simulate a mass transfer process such as mineral weathering or sequential formation of proteins.
}

\usage{
  transfer(nsteps = 500, dmode = "coupled", devmax = 0.1, plot = NULL,
    ibalance = 1, fmode = "one", buffers = NULL, alphamax = -2,
    alphastart = -10, T = 25, P = "Psat", do.title = TRUE, beta = 0)
  draw.transfer(t, ylim = c(-10, 1), ylimbasis = c(-12, -2), 
    logprogress = FALSE)
  feldspar(which = "closed", plot.it = FALSE)
  apc(which = "open", basis = "CO2", plot.it = FALSE)
}

\arguments{
  \item{nsteps}{numeric, maximum number of steps to run simulation.}
  \item{dmode}{character, destruction mode.}
  \item{devmax}{numeric, maximum deviation of logarithm of activity of basis species in any step.}
  \item{plot}{numeric, which basis species to use as plotting and coupling variables.}
  \item{ibalance}{numeric, which basis species is the primary conservant.}
  \item{fmode}{character, formation mode.}
  \item{buffers}{list, basis species to be buffered during the simulation.}
  \item{alphamax}{numeric, maximum value of the destruction exponent.}
  \item{alphastart}{numeric, initial value of the destruction exponent.}
  \item{T}{numeric, temperature.}
  \item{P}{numeric, pressure.}
  \item{do.title}{logical, plot a title?}
  \item{beta}{numeric, alpha + beta = buffer transfer exponent.}
  \item{t}{list, the output of \code{transfer}.}
  \item{ylim}{numeric, y-axis limits.}
  \item{ylimbasis}{numeric, y-axis limits for the logarithms of activities of basis species.}
  \item{logprogress}{logical, put reaction progress on a logarithmic scale?}
  \item{which}{character, type of system to simulate.}
  \item{plot.it}{logical, summarize the results using draw.transfer?}
  \item{basis}{character, type of basis definition to use.}
}

\details{

  The transfer function calculates a reaction path that is generated by incrementally reacting a starting composition into an aqueous system. Before calling this function, set up a system and define the starting material using \code{\link{species}}. 

  At each step, a small amount (\eqn{10^{\alpha}}{10^alpha}) of the starting composition is provisionally reacted and a relatively more stable product may be formed. The amount of product formed is such that the activity of the primary conservant (the basis species given in \code{ibalance}) is not changed. The changes in the activities of the other basis species are calculated, and the process is iterated until \code{nsteps} is reached or the value of \eqn{\alpha}{alpha} is driven to a very low value (\code{logpresent}, which is a constant set in the code to \eqn{-50}).

 If at a given step the most stable product is different from the one before, either the previous products are ignored (for \code{dmode} equal to \samp{none}, i.e. an open system) or the reaction of the starting material is coupled to that of the existing products (for \code{dmode} equal to \samp{coupled}, i.e. a closed system) through a secondary conservation constraint. The basis species that are candidates for the secondary conservation are identified in \code{iplot}.

 The initial value of alpha is given by \code{alphastart}. After successful steps, the function increases the value of alpha by \eqn{1}, and after failed steps decreases the value of alpha by \eqn{1}. One condition that can lead to a failed step is that the logarithm of activity of any basis species changes by more than \code{devmax}. Therefore, throughout the simulation the value of \eqn{\alpha}{alpha} dynamically adjusts based on the \code{devmax} set by the user.

  \code{buffers} is a list with elements \code{basis} indicating the basis species to be buffered and \code{buffer} naming the buffers to use for that basis species. If this argument is given, at each step the activity of the basis species \emph{in the buffer} is calculated. The difference between this activity and the current activity of the basis species in the system is then multiplied by 10 raised to the ( \code{alpha} + \code{beta} ) and this quantity added to the current activity of the basis species in the system. As a result, the value of \code{beta} modifies the strength of the buffer relative to the incremental reaction progress.

  \code{draw.transfer} is used to plot the logarithms of activities of basis species, and logarithms of activities (moles for solid species, molalities for aqueous species) of the minerals or proteins as a function of reaction progress, or logarithm of reaction progress if \code{logprogress} is set to \code{TRUE}. The y-limits of the plots can be set using \code{ylim} and \code{ylimbasis}.

  \code{feldspar} and \code{apc} encode examples for feldspar weathering and reactions among proteins in the anaphase-promoting complex of yeast.

}

\section{Warning}{
  \code{feldspar} loads the secondary database (using \code{\link{add.obigt}}); after using this function use \code{data(thermo)} to restore the primary database.
}

\value{
  \code{transfer} returns a list containing information about the conditions at each step: \code{basis}, data frame of the logarithms of activities of basis species, \code{species}, data frame of the logarithms of activities (moles for solids) of species, \code{alphas}, numeric vector of the values of the destruction exponent, \code{dmodes}, character vector of the destruction mode, \code{istables}, numeric vector of the index of the most stable product, \code{myaffs}, list of the affinities of the formation reactions of species, \code{didwork}, logical vector indicating whether the steps succeeded or failed.
}


\examples{\dontshow{data(thermo)}
## react potassium feldspar in a closed system
## after Steinmann et al., 1994 and Helgeson et al., 1969
# load the secondary database to use H4SiO4
add.obigt()
basis(c("Al+3", "H4SiO4", "K+", "H2O", "H+", "O2"), c(0, -6, -6, 0, 0, 0))
species(c("k-feldspar", "muscovite", "pyrophyllite", "kaolinite", "gibbsite"))
a <- affinity(H4SiO4=c(-6, -2), "K+"=c(-3, 8))
diagram(a, fill="heat")
basis("pH", 4)
species(1:5, c(-4, rep(-999, 4)))
tr <- transfer(190, dmode="coupled", plot=c(2, 3), devmax=1, alphamax=-1.5)
# plot the output from transfer
draw.transfer(tr)
# reset the plot layout
layout(matrix(1))
# restore the primary database
data(thermo)

## can also run the calculation above (at higher resolution) with
#feldspar("closed")
## or an example for proteins with
#apc("closed")
}

\references{
  Helgeson, H. C., Garrels, R. M. and Mackenzie, F. T. (1969) Evaluation of irreversible reactions in geochemical processes involving minerals and aqueous solutions. II. Applications. \emph{Geochim. Cosmochim. Acta} \bold{33}, 455--481. \url{http://dx.doi.org/10.1016/0016-7037(69)90127-6}

  Steinmann, P., Lichtner, P. C. and Shotyk, W. (1994) Reaction path approach to mineral weathering reactions. \emph{Clay Clay Min.} \bold{42}, 197--206. \url{http://ccm.geoscienceworld.org/cgi/content/abstract/42/2/197}
}

\concept{Extra thermodynamic modeling}
