CADFpvalues <- function(t0, rho2=0.5, trend=c("c", "nc", "ct", "none", "drift", "trend"))
{
  # This procedure computes the p-values of the CADF test developed by Hansen (1995)
  # t0:        sample statistic
  # rho2:      value of the parameter rho^2
  # trend:     CADF model type. No constant "nc", Constant "c", Constant plus trend "ct"
  # The procedure is described in Costantini, Lupi & Popp (2007)
  # Citation:
  # @TECHREPORT{,
  # author = {Costantini, Mauro and Lupi, Claudio and Popp, Stephan},
  # title = {A Panel-{CADF} Test for Unit Roots},
  # institution = {University of Molise},
  # year = {2007},
  # type = {Economics \& Statistics Discussion Paper},
  # number = {39/07},
  # timestamp = {2008.10.15},
  # url = {http://econpapers.repec.org/paper/molecsdps/esdp07039.htm}}

  trend <- match.arg(trend)
  switch(trend,
       "none" = trend <- "nc",
       "drift" = trend <- "c",
       "trend" = trend <- "ct")
  switch(trend,
         "c" = coeffs <- matrix(coeffs_c,  1005, 5, byrow=TRUE),
         "nc" = coeffs <- matrix(coeffs_nc, 1005, 5, byrow=TRUE),
         "ct" = coeffs <- matrix(coeffs_ct, 1005, 5, byrow=TRUE))

  # the first column of coefs are the probabilities, the other columns are beta_0, ..., beta_4
  # of Costantini, Lupi & Popp eqn (13).
    
  L <- dim(coeffs)[1]

  # compute the fitted quantiles for the given value of rho^2
  fitted.q <- coeffs[,2] + coeffs[,3]*rho2 + coeffs[,4]*rho2^2 + coeffs[,5]*rho2^3

  # find the position of the fitted quantile that is closest to the sample statistic
  difference <- abs(fitted.q - t0)
  position   <- which(difference==min(difference))
  if (length(position)>1) position <- position[1]
  
  # interpolate locally using eq (10) in Costantini, Lupi & Popp (2007) using l observations (l must be an integer odd number)
  l <- 11
  if ( (position > ((l-1)/2)) & (position < (L - (l-1)/2)) ) range <- (position - (l-1)/2):(position + (l-1)/2)
  if (position <= ((l-1)/2))                                 range <- 1:l
  if (position >= (L-(l-1)/2))                               range <- (L-(l-1)/2+1):L
  prob <- coeffs[range,1] 
  prob <- qnorm(prob)
  x1 <- fitted.q[range]
  x2 <- x1^2
  x3 <- x1^3
  local.interp <- lm(prob~x1+x2+x3)
  model.summary <- summary(local.interp)
  gamma  <- model.summary$coefficients
  vt0 <- c(1, t0, t0^2, t0^3)
  vt0 <- vt0[1:dim(gamma)[1]]
  p.value <- vt0%*%gamma[,1]
  p.value <- pnorm(p.value)
  return(p.value)
}

###############################################################################################################
#  Tables of coefficients computed by Costantini, Lupi & Popp (2007) ##########################################
###############################################################################################################
 
  coeffs_c <- c(0.00025,-3.65763549019237,-2.32467772521735,2.72415216250122,-1.14775083238335,
                5e-04,-3.49556626566443,-2.24211873026207,2.65462742426698,-1.17288545707193,
                0.00075,-3.40085379048553,-2.06958755455025,2.25138630803388,-0.935281770807006,
                0.001,-3.31306721587468,-2.15075621330517,2.43390648128164,-1.05480496613948,
                0.002,-3.09342116871037,-2.31133068688811,2.81686832170154,-1.34253030756012,
                0.003,-2.96863185471126,-2.29012076698092,2.72689160254652,-1.28710152880893,
                0.004,-2.87550041511816,-2.28809106374631,2.69145748900743,-1.26380016444477,
                0.005,-2.79561555457964,-2.32409395326508,2.71383817538055,-1.26473514969571,
                0.006,-2.7288499207719,-2.34953046036165,2.71982982953511,-1.25763298533048,
                0.007,-2.67638298441944,-2.36012733843633,2.73863734770900,-1.27099779690023,
                0.008,-2.63239062024450,-2.34404426818237,2.69739087389171,-1.24909556638222,
                0.009,-2.58362197446487,-2.38684474845532,2.76433624488199,-1.28664629076268,
                0.01,-2.54903232889016,-2.362369689475,2.69716227284464,-1.24117086681383,
                0.011,-2.51091627735069,-2.37553749505315,2.70977052303927,-1.25038295578942,
                0.012,-2.48208896385183,-2.34628815501374,2.63598891659879,-1.20414462795801,
                0.013,-2.44790274189907,-2.38281651110711,2.70032430889415,-1.23978184652741,
                0.014,-2.42267310340118,-2.35656453864271,2.63037910440235,-1.19401153521811,
                0.015,-2.39401063189398,-2.36936655431358,2.63266126151976,-1.18856301052009,
                0.016,-2.36726704714492,-2.37890499542831,2.63638052130087,-1.18728465954110,
                0.017,-2.34204452956494,-2.38344810016013,2.62436031648801,-1.17480242369033,
                0.018,-2.31938793246755,-2.37859055392552,2.59473536421628,-1.15187937866662,
                0.019,-2.29706183804186,-2.38728728443721,2.60840554638736,-1.16087975588664,
                0.02,-2.27764398637985,-2.38496242140969,2.60263295661073,-1.15798860090946,
                0.021,-2.25731480870636,-2.38952552936716,2.6039693164398,-1.15836596610297,
                0.022,-2.23886441365299,-2.38753369437995,2.59191022998421,-1.14990752147427,
                0.023,-2.21939247191389,-2.39772850308898,2.60586086082025,-1.15721620800697,
                0.024,-2.20118270837848,-2.39801671912039,2.59102268920164,-1.14404291230146,
                0.025,-2.18529465304526,-2.38513941071564,2.55535332188154,-1.12210545896903,
                0.026,-2.16533458425789,-2.41531930221518,2.61173680866713,-1.15470256083287,
                0.027,-2.14815987071213,-2.42617195300031,2.62675973081734,-1.16182602607780,
                0.028,-2.13239795255753,-2.42585002211641,2.61881522735665,-1.15632385018225,
                0.029,-2.11769466523668,-2.42938315088141,2.62544376300692,-1.16207164863147,
                0.03,-2.10340121816706,-2.42760423922763,2.62197699174453,-1.16305651620467,
                0.031,-2.08993843741671,-2.42245310463776,2.60540804649467,-1.15208943434388,
                0.032,-2.07638674368986,-2.42361889466573,2.60333304049139,-1.14982766131999,
                0.033,-2.06439332891622,-2.40813149132782,2.56388839292225,-1.12579262829593,
                0.034,-2.05063585586695,-2.41652775842681,2.57666070951899,-1.13321875435451,
                0.035,-2.03746058155027,-2.4191298169161,2.57878815208427,-1.13574277559314,
                0.036,-2.02509285766269,-2.41621542473671,2.56421235170347,-1.12515351131618,
                0.037,-2.01340179156469,-2.4108901827954,2.54883452209941,-1.11582413062460,
                0.038,-2.00086551598069,-2.41964374569072,2.56678032147908,-1.12806006841167,
                0.039,-1.98762132953332,-2.42972331472399,2.58058934199592,-1.13508982870303,
                0.04,-1.97642860165498,-2.42655932406001,2.5719602336133,-1.13149870698843,
                0.041,-1.96557984676107,-2.42160268215654,2.5539667379969,-1.11957819529926,
                0.042,-1.95541779418785,-2.41618366641018,2.53758068262177,-1.10934961023925,
                0.043,-1.94400863386027,-2.41874286111591,2.5375793308322,-1.11013526415558,
                0.044,-1.93312098562475,-2.42383198733122,2.54634810665942,-1.11578594321805,
                0.045,-1.92321849675495,-2.41730743897955,2.52523159718519,-1.10173796621839,
                0.046,-1.91201881345706,-2.41978679963198,2.52007627261888,-1.09618467649355,
                0.047,-1.90157151429257,-2.42277849393006,2.51940009133478,-1.09427242111548,
                0.048,-1.89190914916267,-2.42152298484338,2.51102075061844,-1.08799794504074,
                0.049,-1.88165472078119,-2.42721322049435,2.51831810313561,-1.09144572223122,
                0.05,-1.87151794587288,-2.43137947922814,2.52395819219422,-1.09539885975878,
                0.051,-1.86161695056313,-2.43450967092247,2.52394269787394,-1.09374053389441,
                0.052,-1.85124475075101,-2.44529601331748,2.54272198580973,-1.10427527968945,
                0.053,-1.84208422206266,-2.44569547737316,2.54184198259401,-1.10494728763283,
                0.054,-1.83242270619313,-2.44831379775977,2.54264098016688,-1.10538782725348,
                0.055,-1.82336022334665,-2.45424413581822,2.55488670249189,-1.11328398785363,
                0.056,-1.81388093532656,-2.46096782757533,2.56479640253207,-1.11815132315222,
                0.057,-1.80525504881031,-2.4616432838059,2.56288121491312,-1.11692122290004,
                0.058,-1.79593861159882,-2.47252870137806,2.58873102084339,-1.13508554747860,
                0.059,-1.78671878692541,-2.47717808445772,2.59316145546616,-1.13692964394896,
                0.06,-1.77824534999305,-2.47502533191129,2.58054431696215,-1.12784669757857,
                0.061,-1.76958835562833,-2.47902225414795,2.58684455018197,-1.13288385755656,
                0.062,-1.76103051117149,-2.47947710183047,2.58127090968323,-1.12879299582100,
                0.063,-1.75291562581232,-2.48177576736696,2.58392150379575,-1.13104466753869,
                0.064,-1.74452417829825,-2.48308386517021,2.58246072914838,-1.13029688198401,
                0.065,-1.73746409747968,-2.47788131296983,2.56869637346868,-1.12227935279612,
                0.066,-1.73002433346151,-2.47573125079006,2.55854581484834,-1.11457876401016,
                0.067,-1.72223516258574,-2.47343417659745,2.54855812291518,-1.10839662748195,
                0.068,-1.71484487389716,-2.47185969072889,2.54194806224357,-1.10421538208005,
                0.069,-1.70764488844602,-2.47100943169688,2.53883819518559,-1.10303423910608,
                0.07,-1.70052760323763,-2.47135888235805,2.53913863228752,-1.10447213943362,
                0.071,-1.69351676212348,-2.47068654573010,2.53540504308594,-1.10221834095285,
                0.072,-1.68522224985952,-2.47866534920188,2.54474784786349,-1.10563807124742,
                0.073,-1.67863338845964,-2.47685021003558,2.53881753584336,-1.10230587249518,
                0.074,-1.67138330901158,-2.47801506293709,2.53691654104177,-1.10034585048778,
                0.075,-1.66465792222179,-2.47759852594497,2.53488066015215,-1.10013964233569,
                0.076,-1.65753553126818,-2.47872617233225,2.53366151076492,-1.09932543082563,
                0.077,-1.65050604378306,-2.48058839795098,2.53481922114223,-1.10019972894036,
                0.078,-1.6431745407283,-2.48701952616875,2.54717856342835,-1.10843084139761,
                0.079,-1.6363616358543,-2.48726925347294,2.54310808002500,-1.10528635440646,
                0.08,-1.62932916227204,-2.49345282421644,2.55746109676766,-1.11554549865645,
                0.081,-1.62322059992446,-2.49072261238926,2.54875524017415,-1.11012764344556,
                0.082,-1.61619842430551,-2.49517298945588,2.5550367330954,-1.11361702665286,
                0.083,-1.60984821272572,-2.49762882031112,2.55797273264424,-1.11496829317919,
                0.084,-1.60274431748399,-2.49873543814820,2.55284943715939,-1.11026212693058,
                0.085,-1.59694632580172,-2.49647974688762,2.54680409265683,-1.10667381334171,
                0.086,-1.590704616839,-2.49645356073606,2.54325324688718,-1.10392062696200,
                0.087,-1.58470883134023,-2.49383446173757,2.53578596830306,-1.09981888676510,
                0.088,-1.57885742685756,-2.48931996085528,2.52004231752483,-1.08883659917810,
                0.089,-1.57287074399491,-2.48925409428421,2.51962736721107,-1.08935491041108,
                0.09,-1.56652978711411,-2.49116844449831,2.52011800030181,-1.08974168744320,
                0.091,-1.55955901771667,-2.49637648217312,2.52625808265060,-1.09254412950788,
                0.092,-1.55404989978564,-2.49140686307936,2.5116634152448,-1.08322485922822,
                0.093,-1.54830456836354,-2.48982441110634,2.50705582534134,-1.08118723135196,
                0.094,-1.54190039445092,-2.49134228540008,2.50579106773507,-1.08031490740202,
                0.095,-1.53620838853564,-2.4910883392419,2.50226271559009,-1.07797772939296,
                0.096,-1.53029437488118,-2.48971414887359,2.49418626860899,-1.07250784421556,
                0.097,-1.52415237547121,-2.49191112130462,2.4934885201133,-1.07099971753483,
                0.098,-1.51868602485317,-2.48910243365482,2.48421370060395,-1.06508824214963,
                0.099,-1.51277752872704,-2.49068829630145,2.48483622513667,-1.06546685535329,
                0.1,-1.50710541311944,-2.4922317511466,2.48669005515346,-1.06696123115733,
                0.101,-1.50089164617979,-2.49679421932172,2.49353690042508,-1.07133284508024,
                0.102,-1.49562820669950,-2.49363613605502,2.48360969808193,-1.06494216485080,
                0.103,-1.48993199221946,-2.49352010550511,2.47881316546139,-1.06139688495137,
                0.104,-1.48446135071027,-2.49374031509580,2.47798430294097,-1.06158834169745,
                0.105,-1.47934528717016,-2.49029284536994,2.46802166639816,-1.05563627110182,
                0.106,-1.47350190181267,-2.49299766014829,2.47063151613378,-1.05686072684313,
                0.107,-1.46788603873532,-2.49888729033310,2.48439206964738,-1.06636197254138,
                0.108,-1.46309493312709,-2.49602500074529,2.4768048326451,-1.06190636098320,
                0.109,-1.45714940013243,-2.49984727260935,2.48095161250501,-1.06366966295412,
                0.11,-1.45166784002111,-2.50233068238979,2.48521553808757,-1.06695627158772,
                0.111,-1.44656683566197,-2.5017178687904,2.48248759812525,-1.06562431829296,
                0.112,-1.44135190325831,-2.50379324933968,2.48642893408765,-1.06871696275825,
                0.113,-1.4367417409709,-2.49921507056688,2.47578941168370,-1.06322943641694,
                0.114,-1.43171236491325,-2.49809461823305,2.46998771109946,-1.05922803860434,
                0.115,-1.42628981304150,-2.50156856415320,2.47561831222554,-1.06305278381602,
                0.116,-1.42151158218527,-2.50083339968521,2.47298558384876,-1.06189087727050,
                0.117,-1.41657157417761,-2.50239474073716,2.47528565533495,-1.06347368314714,
                0.118,-1.41096272298348,-2.50614875197293,2.47904670996669,-1.06453704896392,
                0.119,-1.40523291195263,-2.51132271603368,2.48575618362833,-1.06731204907204,
                0.12,-1.40049619038096,-2.50923126209007,2.47798583018274,-1.06205309493067,
                0.121,-1.39604200820284,-2.50783176881354,2.47522066305205,-1.06141852522816,
                0.122,-1.39114333883634,-2.50732028879359,2.47312404238122,-1.06098770878650,
                0.123,-1.38585586809921,-2.51138244501478,2.47834317052331,-1.06347786836644,
                0.124,-1.38095576447165,-2.51525897440189,2.48780268705666,-1.07025901543784,
                0.125,-1.37560553353075,-2.52127095116972,2.49800033876310,-1.07623456621927,
                0.126,-1.37099943534054,-2.52205753036308,2.50190573175688,-1.08058920854430,
                0.127,-1.36642870343154,-2.52105951573998,2.49764206107519,-1.07784727222662,
                0.128,-1.36150020086588,-2.52346685293205,2.50224872165905,-1.08149385379677,
                0.129,-1.35680042283391,-2.52462414525931,2.50281125202989,-1.08163577154438,
                0.13,-1.35222275060189,-2.52512126810483,2.50320332779011,-1.08246793610545,
                0.131,-1.34739581977242,-2.52694736287886,2.50461835879804,-1.08315651439216,
                0.132,-1.34360491089581,-2.52122746869212,2.4916080634618,-1.07580682268747,
                0.133,-1.33844150628403,-2.5256968702794,2.49822138446574,-1.07927914635250,
                0.134,-1.33379022724340,-2.5279675356339,2.50243003715344,-1.08240262054433,
                0.135,-1.32867779465740,-2.53157613310738,2.50674410766247,-1.08469383340298,
                0.136,-1.32391538456421,-2.53296011053141,2.50838438491677,-1.08622533782842,
                0.137,-1.31957479526543,-2.53226422140494,2.50464692109593,-1.08376426625056,
                0.138,-1.3148108757073,-2.5354935441632,2.50969554728187,-1.08690732041655,
                0.139,-1.30983304213226,-2.54102281035468,2.51983616877106,-1.09316779483079,
                0.14,-1.30533406275742,-2.54172245580977,2.51947389783894,-1.09305869340938,
                0.141,-1.30093217505459,-2.54263371345778,2.51914398290282,-1.09277698083968,
                0.142,-1.29588009511326,-2.54672486539948,2.52402664667327,-1.09486698284361,
                0.143,-1.29124987329826,-2.54744947850564,2.52155647039059,-1.09243760379038,
                0.144,-1.2869100750384,-2.54598829756240,2.51581264338106,-1.08876440913437,
                0.145,-1.28235613815087,-2.54818271672369,2.51883562297984,-1.09088591659208,
                0.146,-1.27764257530790,-2.55133416343155,2.52333695834206,-1.09352676085378,
                0.147,-1.27292321583126,-2.55506246028538,2.52994879802928,-1.09768259467676,
                0.148,-1.26886689360739,-2.55175511682815,2.52120173086432,-1.09264507677687,
                0.149,-1.26521119547141,-2.54945919561706,2.51649426068872,-1.09046816366356,
                0.15,-1.26092273283093,-2.54965719679668,2.51465614525407,-1.08924039979709,
                0.151,-1.25689701799875,-2.54720313965976,2.50680918187206,-1.08435863090177,
                0.152,-1.25296336856921,-2.54567554326116,2.50203137484411,-1.08148692417617,
                0.153,-1.24906537869824,-2.54524633555526,2.50111251193891,-1.08173887333187,
                0.154,-1.24509681840111,-2.54331963711334,2.49538131956797,-1.07852204633069,
                0.155,-1.24072068670524,-2.54665126874857,2.50174954662259,-1.08282002169478,
                0.156,-1.23668221866474,-2.54720090666984,2.50367531574962,-1.08516882571806,
                0.157,-1.23231345187262,-2.55029944008501,2.50915604953068,-1.08894487568625,
                0.158,-1.22826528094943,-2.54836782512306,2.50311650815362,-1.08549798417575,
                0.159,-1.22434274942280,-2.5463237682037,2.49621001276139,-1.08139849802562,
                0.16,-1.22086786240041,-2.54161562177953,2.48527772481682,-1.07518209710468,
                0.161,-1.21725900410800,-2.53806746165366,2.47562223734101,-1.06920209295904,
                0.162,-1.21331678827522,-2.53857368166659,2.47620013557243,-1.07007162963923,
                0.163,-1.20935477974559,-2.53872907802415,2.4749179639601,-1.06937466931200,
                0.164,-1.20534599654020,-2.53897212233947,2.47445238850599,-1.06959009208988,
                0.165,-1.20151121590084,-2.53917755044743,2.47458889341592,-1.07027739442013,
                0.166,-1.19760237350805,-2.54070316398578,2.47708277844140,-1.07207049096015,
                0.167,-1.19354438196294,-2.54205833750773,2.47804300028504,-1.07253873714743,
                0.168,-1.18982979430848,-2.54140230286847,2.47547382412110,-1.07123613188178,
                0.169,-1.18585750326198,-2.54199403845838,2.47523659535549,-1.07125716075986,
                0.17,-1.18185107175604,-2.54334369503495,2.47574239533949,-1.07123380917741,
                0.171,-1.17781188267060,-2.54459996931643,2.47629522634078,-1.07152800953122,
                0.172,-1.17406165068325,-2.54138678990845,2.46553699942648,-1.06448515028066,
                0.173,-1.16985761495812,-2.54377284067927,2.46797506254501,-1.06559796240937,
                0.174,-1.16625372531431,-2.54143321301408,2.46159265065708,-1.06195581785661,
                0.175,-1.16259404186739,-2.53989834361485,2.45671349358076,-1.05897345661411,
                0.176,-1.15861641123779,-2.5408066008863,2.45839099871944,-1.06101820960464,
                0.177,-1.15493538278895,-2.54029718566255,2.45737369710627,-1.06130543358060,
                0.178,-1.15096490993149,-2.54233021705445,2.45959174538851,-1.06238804328674,
                0.179,-1.14688209493050,-2.54593465370718,2.46412682588338,-1.06412837473544,
                0.18,-1.14321667820062,-2.54580918660756,2.46226229916289,-1.06294268714312,
                0.181,-1.13931117895658,-2.54590648458455,2.45984290216719,-1.06142328206864,
                0.182,-1.13576181723033,-2.54457095062008,2.45648094766092,-1.05983430625095,
                0.183,-1.13170562277338,-2.54708857811344,2.45909821100622,-1.06099922385683,
                0.184,-1.12803750075253,-2.54736188762615,2.45915052394431,-1.06169674555386,
                0.185,-1.12413521564710,-2.54848166436928,2.46023633449344,-1.06269796566917,
                0.186,-1.12058206390306,-2.54768473829052,2.45783564905226,-1.06178409673061,
                0.187,-1.11660408013163,-2.54924969707767,2.46007999056391,-1.06372912406771,
                0.188,-1.11284385955724,-2.55035724508617,2.46156852554202,-1.06488757160227,
                0.189,-1.10919081957288,-2.55068881276158,2.46183720644786,-1.06573306630778,
                0.19,-1.10569686470057,-2.54902707004125,2.45772275311469,-1.06408973356134,
                0.191,-1.10200567057848,-2.54955052848417,2.45753934411047,-1.06414847854726,
                0.192,-1.09804739816377,-2.55146227013352,2.45913707921287,-1.06494408075493,
                0.193,-1.09400903183790,-2.55489351899405,2.46388686499054,-1.06743646773092,
                0.194,-1.09033896216530,-2.55520075407084,2.46325665287885,-1.06706606254585,
                0.195,-1.08657326010642,-2.55712638396381,2.46655208971429,-1.06943594579594,
                0.196,-1.08277939212597,-2.55663275703096,2.46200100649157,-1.06625787001521,
                0.197,-1.07935547910890,-2.55643395242424,2.45965946407878,-1.06466512104396,
                0.198,-1.07537084166835,-2.55954312594853,2.46246999107158,-1.06556348741191,
                0.199,-1.07133386246020,-2.56326568469932,2.46919419885514,-1.06999545765055,
                0.2,-1.06764141874307,-2.56490575477690,2.47160944795012,-1.07165848138023,
                0.201,-1.06403230890513,-2.56506991343563,2.46980522903617,-1.07054657667591,
                0.202,-1.06047472712969,-2.56429365792483,2.46542912897154,-1.06745537969795,
                0.203,-1.05709324881729,-2.56292337801527,2.45975809394849,-1.06355060936444,
                0.204,-1.05323638104255,-2.56538441894082,2.46363190650959,-1.06594715161675,
                0.205,-1.04952265967588,-2.56693640651797,2.46610076817285,-1.06803711821309,
                0.206,-1.04607905344907,-2.56747004422427,2.46640182747854,-1.06841644936693,
                0.207,-1.04274536949995,-2.56526008128228,2.45960479330709,-1.06430545587755,
                0.208,-1.03910570431392,-2.56705409443068,2.46109503835400,-1.06493081219624,
                0.209,-1.03566643774716,-2.56665245930521,2.45881748793487,-1.06370049522205,
                0.21,-1.03228942913344,-2.56690496498008,2.45863127847265,-1.06411989987586,
                0.211,-1.02899548101667,-2.56836324260364,2.46185595255756,-1.06642270632883,
                0.212,-1.02570856219978,-2.56928347591984,2.46418706307361,-1.06846725074148,
                0.213,-1.02258671820548,-2.56609471515861,2.45610039561445,-1.06394785551854,
                0.214,-1.01869328729315,-2.56989343166574,2.46151325425316,-1.06677337384494,
                0.215,-1.01542830161826,-2.56959142794645,2.45969199935103,-1.06554304274474,
                0.216,-1.01216307925309,-2.56866459269236,2.45542132998171,-1.06266576296606,
                0.217,-1.00873715697363,-2.56880651259929,2.45472724311569,-1.06276297183021,
                0.218,-1.00487192745365,-2.57330332551182,2.46219840219109,-1.06710949789491,
                0.219,-1.00117675905705,-2.57663056127595,2.46880965089613,-1.07188059618530,
                0.22,-0.997704443039281,-2.57712138354718,2.46844343785913,-1.07193231514963,
                0.221,-0.994402906286783,-2.57846086278953,2.47097656663175,-1.07386887212590,
                0.222,-0.991349273683185,-2.57777044075504,2.46923964389121,-1.07327008171755,
                0.223,-0.988044120675254,-2.57672266591894,2.46495176897871,-1.07071770497472,
                0.224,-0.98464609200476,-2.57942306242429,2.47076295087394,-1.07481907136905,
                0.225,-0.98107440854911,-2.58062219271603,2.47065058951039,-1.07436048166486,
                0.226,-0.977766247600196,-2.58164691629653,2.47164928547861,-1.07480000436551,
                0.227,-0.97459021436329,-2.58118675413289,2.46908417121626,-1.07318277313501,
                0.228,-0.971508915871325,-2.58095532652967,2.46911432613796,-1.07417372092067,
                0.229,-0.967809579640195,-2.58357716651779,2.47360202959016,-1.07733697538855,
                0.23,-0.96444654836274,-2.58382453821454,2.47128608898961,-1.07520858107881,
                0.231,-0.961170378481067,-2.58349584132557,2.46890660892467,-1.07371150313879,
                0.232,-0.958071854789933,-2.58266838632636,2.46630263970776,-1.07246820476932,
                0.233,-0.954423482267651,-2.58670427715575,2.47434245386153,-1.07766700785408,
                0.234,-0.951354847204175,-2.58712176188392,2.47474043874966,-1.07808190548305,
                0.235,-0.948006061153074,-2.58929659555376,2.47933892257936,-1.08145492327989,
                0.236,-0.94440490990488,-2.59195535659157,2.48288028620659,-1.08344368917493,
                0.237,-0.941252616104867,-2.59054616259636,2.47781014704481,-1.08046598097198,
                0.238,-0.938077204200976,-2.59075837782178,2.47712598137701,-1.08015143157725,
                0.239,-0.934941942218437,-2.59126822980547,2.47848255551858,-1.08179034078554,
                0.24,-0.931719453619479,-2.59203322951764,2.47951419485041,-1.08306860415209,
                0.241,-0.928559951509603,-2.59140313508288,2.47597078592069,-1.08067965280362,
                0.242,-0.925453286161111,-2.59181473761919,2.47586909975846,-1.08079679906005,
                0.243,-0.922014456117252,-2.59399748031679,2.47872081335888,-1.08259370573872,
                0.244,-0.919092866462346,-2.59431346887190,2.48028615704598,-1.08443897901433,
                0.245,-0.916126886325632,-2.59206488061928,2.47342349289976,-1.08019180859326,
                0.246,-0.912785423597168,-2.59320435800187,2.47437307828793,-1.08086114620129,
                0.247,-0.909476032269681,-2.59434314136582,2.47474190793428,-1.08085689479317,
                0.248,-0.906274408506641,-2.59569561934262,2.47789306796080,-1.08360691578552,
                0.249,-0.903205724914606,-2.59532265338786,2.47579570757124,-1.08245358030955,
                0.25,-0.89973858161064,-2.5980791097756,2.47954205237021,-1.08445902208103,
                0.251,-0.896468590946054,-2.59962888065638,2.48163236511018,-1.08593625392526,
                0.252,-0.893037927666712,-2.60240003069901,2.48557363227518,-1.08804725314457,
                0.253,-0.890192709621909,-2.60163165552345,2.48384744591585,-1.08754064609991,
                0.254,-0.887370089787262,-2.59820489638567,2.47347985892206,-1.08065306994348,
                0.255,-0.88395865499574,-2.60130059229519,2.47876889782647,-1.0839000278988,
                0.256,-0.88090676327227,-2.60167411073264,2.47810564448744,-1.08348479958983,
                0.257,-0.877745080459445,-2.60210802996151,2.47723524361159,-1.08284719173028,
                0.258,-0.87439584194267,-2.60400356180238,2.47875692054252,-1.08350384173533,
                0.259,-0.871124911412474,-2.60468358373656,2.47672152203606,-1.08149862807343,
                0.26,-0.868120554090604,-2.60530655827665,2.47707236186959,-1.08161181675191,
                0.261,-0.864831573214885,-2.60728366199698,2.47984365687389,-1.08317041320457,
                0.262,-0.86168205368246,-2.60710871633864,2.47694221271054,-1.08102867563653,
                0.263,-0.85849148122687,-2.60809557586476,2.47777505290297,-1.08181712493426,
                0.264,-0.85527946833077,-2.60926458204431,2.47790243688865,-1.08157950863633,
                0.265,-0.852242650701707,-2.61084997717610,2.48092992574522,-1.08365637447529,
                0.266,-0.849200872146515,-2.61076679180590,2.47960516070085,-1.08316908522928,
                0.267,-0.845916474774775,-2.61362863306868,2.48508984526521,-1.08690187347359,
                0.268,-0.84297444598941,-2.61456343482791,2.48575475154333,-1.08733504052466,
                0.269,-0.84010988466514,-2.61329371542481,2.48159807096887,-1.08481207827589,
                0.27,-0.83711233498348,-2.61422095071845,2.48293412967185,-1.08593982423012,
                0.271,-0.83425253912818,-2.61316234317478,2.47968943406721,-1.08411725288068,
                0.272,-0.831084782439247,-2.61543901502136,2.48303793529819,-1.08628418100088,
                0.273,-0.828091442246882,-2.61661203184626,2.4860194162827,-1.08895731104707,
                0.274,-0.825119611710746,-2.61717664392882,2.48705710812519,-1.09018634821905,
                0.275,-0.822296928484908,-2.61669035595824,2.48534742493449,-1.08941119001954,
                0.276,-0.819296089441687,-2.61595638927585,2.48119732289527,-1.08644233632322,
                0.277,-0.815730814923332,-2.61982406770971,2.48696294037814,-1.08969403809045,
                0.278,-0.812797888191678,-2.61999195952603,2.48611585299747,-1.08928249419495,
                0.279,-0.809742263377369,-2.62155471663658,2.48865774941539,-1.09099382770389,
                0.28,-0.806745852360094,-2.62216414641469,2.48996222799099,-1.09274087849374,
                0.281,-0.803659264183458,-2.62438080998000,2.49330071657276,-1.09465805500638,
                0.282,-0.80100078276777,-2.62185433937901,2.48669306776204,-1.09084758530615,
                0.283,-0.798286672411488,-2.61987287589549,2.48230823501452,-1.08892037038786,
                0.284,-0.795382347546153,-2.62104938595902,2.48486479818327,-1.09110312206475,
                0.285,-0.792557254261983,-2.61914520731520,2.47798075888929,-1.08641275566857,
                0.286,-0.789721854572513,-2.61753432844969,2.47215454369467,-1.08252581387346,
                0.287,-0.786540959156464,-2.61905599663784,2.47390486404022,-1.08370294242766,
                0.288,-0.783326536696043,-2.62190918292313,2.47879836642608,-1.08676818351509,
                0.289,-0.780237138517774,-2.62334548739404,2.48104779719320,-1.08862883053159,
                0.29,-0.777360410965905,-2.62395871377904,2.48191115408504,-1.08949401323210,
                0.291,-0.774445072866584,-2.62509877424808,2.48312066822541,-1.09025707561296,
                0.292,-0.771617048223042,-2.62504296187904,2.48092579090455,-1.08861705513669,
                0.293,-0.768907274358587,-2.62390681636097,2.47699640230307,-1.08628188379432,
                0.294,-0.766219449494306,-2.62300990719263,2.47469308638648,-1.08529695783441,
                0.295,-0.763340382877523,-2.62389628229613,2.47530632087487,-1.08561487213773,
                0.296,-0.760716449450687,-2.62218349294687,2.47080698798737,-1.08314717965353,
                0.297,-0.757737735421896,-2.62455047478152,2.47603977160967,-1.08709935259642,
                0.298,-0.754625660910075,-2.62718879027141,2.48033571904754,-1.08950844355938,
                0.299,-0.751830163409163,-2.62582939595168,2.47524097666194,-1.08630775203758,
                0.3,-0.749061941543694,-2.62551505508244,2.47267543418054,-1.08458190683634,
                0.301,-0.746689625349151,-2.62279212019791,2.46730009363611,-1.08200108830237,
                0.302,-0.743610010166014,-2.62377703838086,2.46752509603636,-1.08223524264107,
                0.303,-0.740707135305134,-2.62393091267389,2.46606152135829,-1.08126964136730,
                0.304,-0.737958689602958,-2.62462337402099,2.46703853577949,-1.08195916912286,
                0.305,-0.735141075942782,-2.62528267054780,2.46763315611318,-1.08241396380838,
                0.306,-0.732174778823066,-2.62650823979603,2.46924016985867,-1.08358640600494,
                0.307,-0.72923332979925,-2.62722736696549,2.46846931368552,-1.08263284319591,
                0.308,-0.72651062620982,-2.62668565855817,2.46739347140341,-1.08276893180699,
                0.309,-0.723827721155083,-2.62660608914825,2.46701074017531,-1.08311565621284,
                0.31,-0.720891073128432,-2.62749707941475,2.46674706840395,-1.08243293846016,
                0.311,-0.718350175269822,-2.6247455225224,2.45969514856325,-1.0784261119034,
                0.312,-0.715721129825801,-2.62382912988517,2.45752225137514,-1.07754408050404,
                0.313,-0.713016686729023,-2.62291511757046,2.45438341262538,-1.07584382307995,
                0.314,-0.710309075918455,-2.62227609581743,2.45142280016243,-1.07398956560732,
                0.315,-0.707489815452494,-2.62243449690016,2.44936510519897,-1.07226061207260,
                0.316,-0.704643819476715,-2.62211636520205,2.44704078592611,-1.07085712673957,
                0.317,-0.701686570517212,-2.62379542764822,2.45042146618324,-1.07343167700899,
                0.318,-0.698652662248735,-2.626251018469,2.45432961550838,-1.07561499482841,
                0.319,-0.695730340225671,-2.62831466623367,2.45815327236737,-1.078319196297,
                0.32,-0.69298371299772,-2.62839955561784,2.45713971479032,-1.07771868451924,
                0.321,-0.690554743447856,-2.62695951781625,2.45473841699349,-1.07711349376126,
                0.322,-0.687916547275922,-2.62641449708313,2.45328878996427,-1.07666371577731,
                0.323,-0.684941751852916,-2.62978676390652,2.46049499628338,-1.08141619888260,
                0.324,-0.682240661200336,-2.62975526734401,2.45927845104940,-1.08077351374297,
                0.325,-0.679304092017971,-2.63222852998844,2.46359586679340,-1.08353153144063,
                0.326,-0.676613852993152,-2.63191067760145,2.46144901061474,-1.08196083512763,
                0.327,-0.67394076661903,-2.6303690151686,2.4556521058334,-1.07807332763806,
                0.328,-0.671231219307221,-2.63007962294334,2.45387869461948,-1.07711087004655,
                0.329,-0.668637909431575,-2.62954095068421,2.45256462373791,-1.07685325892516,
                0.33,-0.66572495059176,-2.63057491187028,2.45336567626152,-1.07735763875091,
                0.331,-0.663118608182186,-2.63003709900280,2.45171456608174,-1.07674154466916,
                0.332,-0.66029149478163,-2.63049725039501,2.45083866586089,-1.07603293885806,
                0.333,-0.657257714810918,-2.63157638748444,2.44954413577389,-1.07443575058763,
                0.334,-0.654076705740137,-2.63550168511532,2.45649173548154,-1.07851347651784,
                0.335,-0.651130557135842,-2.63641289759271,2.45540805424895,-1.07705099850206,
                0.336,-0.648377286978894,-2.63735122780722,2.45757032530311,-1.07911051671430,
                0.337,-0.645801324511891,-2.63688142797728,2.45504383877408,-1.07747555227493,
                0.338,-0.643197421755487,-2.63559349587242,2.45174713840119,-1.07607280217072,
                0.339,-0.640437005852852,-2.63495050066817,2.44934448744525,-1.07502342837487,
                0.34,-0.637534928225699,-2.63625495132966,2.45116213090332,-1.07643912173738,
                0.341,-0.635096327468757,-2.63447678773781,2.44644455653725,-1.07389638597759,
                0.342,-0.632395624834647,-2.63629921186391,2.45049582456124,-1.07689212873443,
                0.343,-0.62980250604787,-2.63705978453465,2.4515407684149,-1.07756914385942,
                0.344,-0.6266940508497,-2.64081326787916,2.45889655810003,-1.08225879157155,
                0.345,-0.624040532284327,-2.64105531643802,2.45943646432620,-1.08297857763465,
                0.346,-0.621490129981975,-2.64004352038847,2.45712470131160,-1.08222005372225,
                0.347,-0.618509552681368,-2.64146683410482,2.45787109293249,-1.08242099842089,
                0.348,-0.615621634784286,-2.64195365248683,2.45641045069971,-1.08130669129294,
                0.349,-0.613151374301991,-2.64008353524134,2.45174540533234,-1.07879595505132,
                0.35,-0.610446215426815,-2.64092756745431,2.45304751691879,-1.08012600600875,
                0.351,-0.607759299860710,-2.64263977634807,2.45688772675611,-1.08308671402205,
                0.352,-0.605016599755566,-2.64417776258737,2.45978154955669,-1.08529426227871,
                0.353,-0.602239345481062,-2.64567286544510,2.46234878345978,-1.08717368287153,
                0.354,-0.599543654548769,-2.64633926436651,2.46291934604529,-1.08777528951219,
                0.355,-0.597175831946563,-2.64404790234563,2.45720173449061,-1.08456256869044,
                0.356,-0.594272359751585,-2.64670038019272,2.46207648800495,-1.08752656515463,
                0.357,-0.59148526465888,-2.6479448412251,2.46292405924497,-1.08784005091796,
                0.358,-0.588893479680474,-2.64815132317829,2.46371702729215,-1.08917481206653,
                0.359,-0.586103348086667,-2.64954943658078,2.46534654529837,-1.09016056118076,
                0.36,-0.583587101998333,-2.64961688104829,2.46538486963777,-1.09058467765232,
                0.361,-0.581079056144950,-2.64816813206801,2.46166568602977,-1.08875137859643,
                0.362,-0.578410097534947,-2.64900159865335,2.46212388796845,-1.08903144824543,
                0.363,-0.57569027130527,-2.64983365540441,2.46231860250383,-1.08897508733695,
                0.364,-0.573359598205703,-2.64737593409414,2.45573112210209,-1.08491758220479,
                0.365,-0.570718022953327,-2.64686090084684,2.45282309598218,-1.08317538518419,
                0.366,-0.567819967196613,-2.64990716285028,2.45948912714246,-1.08783287301658,
                0.367,-0.565211836122731,-2.64981882009929,2.45798634594156,-1.08688590297790,
                0.368,-0.562764125036393,-2.64835117570014,2.45370842701858,-1.08440698512452,
                0.369,-0.560065726745915,-2.6492578195328,2.4542092499528,-1.08462014706666,
                0.37,-0.557596473684223,-2.64859838047867,2.45267963299608,-1.08420869040421,
                0.371,-0.554817615043474,-2.64874487833851,2.45196134432952,-1.08412868337055,
                0.372,-0.552262899904794,-2.6481912420785,2.44876214727303,-1.08168509916771,
                0.373,-0.549397733950468,-2.65005216183312,2.45065543417996,-1.08255984031342,
                0.374,-0.546815997653072,-2.64943025713132,2.44777728825342,-1.08074032145195,
                0.375,-0.544204538672355,-2.64804323335005,2.44345495707354,-1.07850273860513,
                0.376,-0.54162385136363,-2.64757772548065,2.44152168949031,-1.07748846740107,
                0.377,-0.539019618906884,-2.64944847975482,2.44592234362928,-1.08059551103346,
                0.378,-0.53650960727735,-2.64913161196252,2.44459532853491,-1.08002621298413,
                0.379,-0.534023139555204,-2.64903902491692,2.44371233904158,-1.07970411337079,
                0.38,-0.531080017531732,-2.65028075229816,2.44362770843460,-1.07916375025484,
                0.381,-0.528153497511712,-2.65306415854062,2.44858587085536,-1.08234967379586,
                0.382,-0.525361651170218,-2.6555218862757,2.45380431372702,-1.08588306123264,
                0.383,-0.522425051224934,-2.65892762382190,2.46062695455337,-1.09032592385983,
                0.384,-0.5201162039161,-2.65705689147414,2.45632508516266,-1.08823338058090,
                0.385,-0.517532374537989,-2.65683416383531,2.45507647185575,-1.08757799643568,
                0.386,-0.515229027250853,-2.65487201647416,2.45050428162259,-1.08523727369921,
                0.387,-0.512399891544507,-2.65713147904229,2.45438534447141,-1.08788193325235,
                0.388,-0.510044284346907,-2.65576012730486,2.45073183187716,-1.08577586002146,
                0.389,-0.507476946645688,-2.65431407293494,2.4450358773516,-1.08181889505362,
                0.39,-0.504916151485885,-2.65402704624015,2.44306676379196,-1.08057263967917,
                0.391,-0.502392890852352,-2.65349044601111,2.43989778874452,-1.07829170763555,
                0.392,-0.500162933110197,-2.65155862874848,2.43541554398680,-1.07596735149587,
                0.393,-0.497433404101001,-2.65304664313588,2.43830854229826,-1.07828271424606,
                0.394,-0.494700333558419,-2.65464729703797,2.44127817030425,-1.08044721174773,
                0.395,-0.491862208239985,-2.65868157659538,2.44993644292674,-1.08604394443225,
                0.396,-0.489383001769422,-2.65747354073891,2.44591778974563,-1.08353005538839,
                0.397,-0.486944030642174,-2.65694899908117,2.44310255788847,-1.08145943661076,
                0.398,-0.484062965099619,-2.65986017824419,2.44865000376740,-1.08504015374547,
                0.399,-0.481602759416851,-2.65851895263463,2.44485703954742,-1.08306134756820,
                0.4,-0.479231596471517,-2.65809260198572,2.44420464209654,-1.08314262500892,
                0.401,-0.476768620653201,-2.65764762213850,2.44321224927221,-1.08320445921127,
                0.402,-0.473957581624609,-2.65894740616233,2.44462971892469,-1.08417287807138,
                0.403,-0.47133478423702,-2.65988479197871,2.44588222410549,-1.08509724403843,
                0.404,-0.468673334812516,-2.66161057842253,2.44926652518129,-1.08754179714139,
                0.405,-0.466175665407872,-2.66127261735137,2.44896117619145,-1.08820416874354,
                0.406,-0.463446478989953,-2.66352288140156,2.45268908680425,-1.09045670807280,
                0.407,-0.461103539364716,-2.66280407127991,2.45084732122204,-1.08959584733829,
                0.408,-0.458664916404809,-2.66187790490563,2.44704189407258,-1.08701693835370,
                0.409,-0.456128232652568,-2.66303600046799,2.44950437594696,-1.08896249412486,
                0.41,-0.453506185762123,-2.66332548661698,2.44915126133716,-1.08895568779417,
                0.411,-0.450944795555559,-2.66296960518284,2.44650960135384,-1.08702043640164,
                0.412,-0.448460930269988,-2.66326562611105,2.44714613609099,-1.08785747701027,
                0.413,-0.446106107292652,-2.66239981018817,2.44548827847914,-1.08746855513719,
                0.414,-0.443646757138144,-2.66236582460188,2.44497957620972,-1.08749128718884,
                0.415,-0.441168940280108,-2.66193975736988,2.44309659184311,-1.08649041068902,
                0.416,-0.438506134345527,-2.66280000635314,2.44400355226405,-1.08735383130019,
                0.417,-0.436010309789563,-2.66232629805247,2.44220914673774,-1.08671361134458,
                0.418,-0.433716161939756,-2.66096989645291,2.43979870611176,-1.08614008671093,
                0.419,-0.43074693877674,-2.66409798500224,2.44510007546974,-1.08938006107722,
                0.42,-0.427900686278445,-2.66610632028949,2.44780451958276,-1.09107436332447,
                0.421,-0.425238137577921,-2.66596792795799,2.44571622426180,-1.08966887958183,
                0.422,-0.422736517811152,-2.66596094336854,2.44515785808637,-1.08968209064522,
                0.423,-0.420469391866557,-2.66579093295558,2.44493587505643,-1.09014116850877,
                0.424,-0.417841712815004,-2.66698183080504,2.44648196502553,-1.09112946456437,
                0.425,-0.415337669435176,-2.66734462273669,2.4459546967023,-1.09070821683706,
                0.426,-0.412641859343474,-2.66842769065779,2.44706030104041,-1.09160036899644,
                0.427,-0.409966344788112,-2.67001581607994,2.44868285096881,-1.09239620743679,
                0.428,-0.407570565292416,-2.66817809607694,2.44397011940991,-1.09000668219787,
                0.429,-0.405369540621459,-2.66624220248174,2.43971153778465,-1.08785536245164,
                0.43,-0.40294402964694,-2.66535008266872,2.43711690776204,-1.08661886385140,
                0.431,-0.400376127182307,-2.66580783544059,2.4363777950631,-1.08586825251427,
                0.432,-0.397727301743222,-2.66682971241738,2.4373343934725,-1.08656683980415,
                0.433,-0.394887489263161,-2.66882330372955,2.44002248851775,-1.08819857367475,
                0.434,-0.392654906519158,-2.66822882386704,2.43933171327860,-1.08842500587017,
                0.435,-0.390019308212698,-2.66985228570107,2.44212555393713,-1.09025104652087,
                0.436,-0.387558830118159,-2.67029485541725,2.4422819683256,-1.09041654962169,
                0.437,-0.384924513114607,-2.67073833855506,2.44166224075830,-1.09012776649099,
                0.438,-0.382411891408983,-2.67079110471058,2.44110232408617,-1.09021225503699,
                0.439,-0.379810309018975,-2.6712113779358,2.44009480193243,-1.08937430183466,
                0.44,-0.377447150124576,-2.66963894499868,2.43575704426063,-1.08683007409284,
                0.441,-0.375007711434591,-2.67002296242891,2.43680414618565,-1.08812766621985,
                0.442,-0.372599976806523,-2.66977647505789,2.43571630367232,-1.08770155773501,
                0.443,-0.370141250197584,-2.67105664799816,2.43894809907669,-1.09029978314794,
                0.444,-0.367646100205895,-2.67054866363524,2.43552377240467,-1.08768855260019,
                0.445,-0.365116344447185,-2.67146022129124,2.43693341566228,-1.08870231794958,
                0.446,-0.36268207709644,-2.67173535762792,2.43780533180104,-1.08984585429153,
                0.447,-0.360219841659194,-2.67188284861277,2.43726798465011,-1.08959612399254,
                0.448,-0.357433200839255,-2.67437425208771,2.44146739031495,-1.09210221395190,
                0.449,-0.354874675410044,-2.67451705269174,2.43979860114453,-1.09083449983501,
                0.45,-0.35234928094732,-2.67401587527295,2.437188910077,-1.0893451107246,
                0.451,-0.349612856236052,-2.67640432881841,2.44157716603714,-1.09226985982132,
                0.452,-0.347217336070901,-2.67652854606679,2.44182452871297,-1.09296386422116,
                0.453,-0.344477755878968,-2.67838105917673,2.44496419572415,-1.09508191321660,
                0.454,-0.341650336060348,-2.68094611456589,2.44907051014531,-1.09746513321842,
                0.455,-0.33903313926586,-2.68154734827973,2.44941592228067,-1.09804206311674,
                0.456,-0.336359442493522,-2.68260997848003,2.45017669372095,-1.09836302589239,
                0.457,-0.333482517709964,-2.68556085258608,2.45515039348829,-1.10127489141568,
                0.458,-0.330880707419441,-2.68705379863595,2.45820241422184,-1.10362429852397,
                0.459,-0.328410818533661,-2.68604555860029,2.45460177099540,-1.10140311411015,
                0.46,-0.325648174608239,-2.68797595214925,2.45757797955606,-1.10326098095858,
                0.461,-0.323181126183218,-2.68845271730716,2.45768669101059,-1.10321268957709,
                0.462,-0.320595923942676,-2.69007051419643,2.46035346110343,-1.10486462670104,
                0.463,-0.317885324216368,-2.69203078533137,2.46415287517671,-1.10735240847767,
                0.464,-0.315469046241322,-2.69086294705987,2.46021751519664,-1.10480622208347,
                0.465,-0.312860664415297,-2.69095847440390,2.45953141440271,-1.10462459879504,
                0.466,-0.310252209596963,-2.69202170100723,2.46151761583069,-1.10643718281090,
                0.467,-0.307461215670127,-2.69370347905946,2.46389772482089,-1.10803999547229,
                0.468,-0.305243321402881,-2.69123023095460,2.45869920411664,-1.10564121266656,
                0.469,-0.302644602902148,-2.69199277984443,2.45917142093009,-1.10586549385278,
                0.47,-0.300159048837821,-2.69301209301905,2.4607713597532,-1.10710936109215,
                0.471,-0.297502003760150,-2.69455246936398,2.46303715079075,-1.10838199513360,
                0.472,-0.294930230131898,-2.69554142056458,2.46398771171871,-1.10880005923726,
                0.473,-0.29261343926355,-2.69424554202116,2.45947818596285,-1.10563184152247,
                0.474,-0.289897272515123,-2.69719537400556,2.46562117524969,-1.10973857379130,
                0.475,-0.287394822584988,-2.69686971436257,2.46357422833517,-1.10841086127951,
                0.476,-0.284746722450084,-2.69841289149126,2.46602302948573,-1.11024043707194,
                0.477,-0.282162599272609,-2.69863663973304,2.46495196925232,-1.10941604461007,
                0.478,-0.279525790541198,-2.7007388796762,2.46891482243769,-1.11205071465502,
                0.479,-0.277075582981435,-2.70152909135204,2.47105361578243,-1.11384842428853,
                0.48,-0.274635170205252,-2.70160418296349,2.47087257236129,-1.11400083758001,
                0.481,-0.272013745799979,-2.70332163832598,2.47384769154097,-1.1160075930265,
                0.482,-0.269144018570845,-2.70545906531961,2.47587405132716,-1.11661882476205,
                0.483,-0.266719933562799,-2.70654648234488,2.47883008376576,-1.11922731439465,
                0.484,-0.264364701628365,-2.70480064384296,2.47363049183868,-1.11611391622243,
                0.485,-0.261513700062572,-2.70828132252825,2.48032041177106,-1.12034274518777,
                0.486,-0.259018792516302,-2.70906296182177,2.48222896051903,-1.12213588908166,
                0.487,-0.256748543293259,-2.70749116227061,2.47837342711307,-1.12017958620077,
                0.488,-0.254433088717242,-2.70532766538512,2.47251662409922,-1.11669664032064,
                0.489,-0.252222668598031,-2.70342485823600,2.4691063629923,-1.11549824158601,
                0.49,-0.249347012234258,-2.70637988505035,2.47429385256642,-1.11875523280302,
                0.491,-0.246695387309599,-2.70795888266757,2.47772017727516,-1.12152584549241,
                0.492,-0.244002789506954,-2.70827779758374,2.47619752636147,-1.12036175816713,
                0.493,-0.241710265168298,-2.70598600267172,2.46943635031454,-1.11605779121422,
                0.494,-0.238891563418268,-2.70903394428499,2.47450679101561,-1.11908235331040,
                0.495,-0.235982082741926,-2.71275689497416,2.48141069221172,-1.12331517042374,
                0.496,-0.233534179170022,-2.71308464679391,2.48195886033179,-1.12395885026306,
                0.497,-0.230889600307306,-2.71473147183465,2.4851473345634,-1.12626017443883,
                0.498,-0.22847850602128,-2.71393735968300,2.48245002660602,-1.12477818398684,
                0.499,-0.226192800753372,-2.71358500106642,2.48214401354683,-1.12525334973793,
                0.5,-0.223740983782388,-2.71331618027158,2.4809943241049,-1.12496079638355,
                0.501,-0.221340213510131,-2.71228768444762,2.47817927571855,-1.12357922300673,
                0.502,-0.218807832344622,-2.71215678279248,2.47632278509586,-1.12238256275777,
                0.503,-0.216307670413337,-2.71265718156360,2.47672648857956,-1.12293788665837,
                0.504,-0.213851267138710,-2.71315272325606,2.47688912219399,-1.12307061071168,
                0.505,-0.211346792653229,-2.71306795724204,2.47569924761828,-1.12249033300124,
                0.506,-0.208703256470156,-2.7140163488036,2.47559901756683,-1.12201451003274,
                0.507,-0.206290916602908,-2.71354779485758,2.47352264893177,-1.12071281257084,
                0.508,-0.203718892506273,-2.71424779890237,2.47390639910967,-1.12103789416535,
                0.509,-0.201146787168683,-2.71610906575091,2.47760339263322,-1.12356028868802,
                0.51,-0.198701689530111,-2.71486773573761,2.47306989026009,-1.12063243091508,
                0.511,-0.196206329141197,-2.71522549121987,2.47334146149466,-1.12109912356698,
                0.512,-0.193878334788121,-2.71366949717453,2.46879805665312,-1.11834615166262,
                0.513,-0.19129019608982,-2.71450818524687,2.47014024518867,-1.11964334759605,
                0.514,-0.188712847737229,-2.71479853426423,2.46888902540412,-1.11859862390389,
                0.515,-0.186435176503328,-2.71270967963474,2.46365681037307,-1.11564368033749,
                0.516,-0.183625836584897,-2.71561737851832,2.46917393017266,-1.11920180887193,
                0.517,-0.181337854837736,-2.71373906954402,2.46331336774643,-1.11539399949975,
                0.518,-0.178891756107897,-2.71311734021494,2.46104167719840,-1.11410312545671,
                0.519,-0.175854488211160,-2.71718636336089,2.46790602983300,-1.11793661242230,
                0.52,-0.173292902512753,-2.71648465577152,2.46456770133107,-1.11591406312033,
                0.521,-0.170577266637102,-2.71856992103028,2.46859186186722,-1.11857164504707,
                0.522,-0.168171782786629,-2.71869674426621,2.46885736108086,-1.11905202471765,
                0.523,-0.165633954327194,-2.71872850025603,2.46783517476988,-1.11862888457587,
                0.524,-0.163217640605699,-2.71774158887722,2.4652604481427,-1.11751894726760,
                0.525,-0.160786368664659,-2.71787785336245,2.46529147325315,-1.11797192881160,
                0.526,-0.158290639268267,-2.71788940224863,2.46329244340098,-1.11642517414755,
                0.527,-0.15565776798348,-2.71935924360231,2.4654855159677,-1.11771998642081,
                0.528,-0.153041473518842,-2.72088836129525,2.46770316124343,-1.11904790084272,
                0.529,-0.150733701733996,-2.72085628564181,2.46778051988725,-1.11943162106691,
                0.53,-0.148473058689892,-2.71970520054687,2.46430556521899,-1.11735050361437,
                0.531,-0.146118753698419,-2.71838632090695,2.46006322676358,-1.11463673333058,
                0.532,-0.143394019981787,-2.72061267027045,2.46376392101886,-1.11687648886180,
                0.533,-0.140594564231605,-2.72204004129012,2.46504913546736,-1.11752044428262,
                0.534,-0.138186927589252,-2.72234476525612,2.46526136903137,-1.11787493727411,
                0.535,-0.135342886645291,-2.72419743177116,2.46742479681736,-1.11909393844234,
                0.536,-0.132473719208936,-2.72693470647228,2.47245321462388,-1.12236730358833,
                0.537,-0.130083050795729,-2.72526919979470,2.46776148087869,-1.11970188884073,
                0.538,-0.127666290438964,-2.72472036963821,2.46608042884181,-1.11908646656704,
                0.539,-0.125166516856736,-2.72458657020757,2.46534906951900,-1.11902906492904,
                0.54,-0.122670686573123,-2.72373460828054,2.46281419073124,-1.11790817880437,
                0.541,-0.120032207618870,-2.72380177531855,2.46041083521858,-1.11605547907339,
                0.542,-0.117589273327319,-2.72516100452221,2.46349635668428,-1.11822570414511,
                0.543,-0.115348987192533,-2.72276019451309,2.45782402063808,-1.11514684598990,
                0.544,-0.113164575536025,-2.72061612799991,2.45258107636477,-1.11218454217572,
                0.545,-0.110759987879532,-2.71895306064791,2.44783746292528,-1.10941534934562,
                0.546,-0.108278752794710,-2.7197509603855,2.44899703311133,-1.11021867552121,
                0.547,-0.105633913327019,-2.71989303298161,2.44720925159874,-1.10890450128300,
                0.548,-0.102815365389304,-2.72277993943963,2.45232708229177,-1.11193489794855,
                0.549,-0.100297845653281,-2.72395120616035,2.45480366652796,-1.11392599945731,
                0.55,-0.0978427005286102,-2.72277118950742,2.45138390023351,-1.11227070063272,
                0.551,-0.0951214876972138,-2.72461344061900,2.45456064125125,-1.11443640667343,
                0.552,-0.092719991423233,-2.72411812274929,2.45282310706126,-1.11358339659466,
                0.553,-0.0901409355367337,-2.7249952141926,2.45443566654563,-1.11495323258309,
                0.554,-0.0879326156857452,-2.72285185317609,2.44965232233974,-1.11236200045725,
                0.555,-0.0852808484041758,-2.72329957354873,2.44970423358059,-1.11269024283895,
                0.556,-0.0826151036584976,-2.72516421615476,2.45301025366094,-1.11472006934184,
                0.557,-0.0800581038072863,-2.72522594869903,2.45216273761261,-1.1143563080109,
                0.558,-0.0776307349181486,-2.7250438095303,2.45049240759397,-1.11311417587302,
                0.559,-0.0750934825909895,-2.72624147192376,2.45309602933799,-1.11518744276050,
                0.56,-0.0725680995916848,-2.72650102558526,2.45297129501903,-1.11525616580951,
                0.561,-0.0698657803252747,-2.72795744951674,2.45536150939232,-1.11695654859480,
                0.562,-0.0672963330927616,-2.72846073676527,2.45514580193248,-1.11662674683811,
                0.563,-0.0648771338480002,-2.72811991259112,2.4532474090941,-1.11537586887523,
                0.564,-0.062242935971976,-2.73004386943499,2.45710815296462,-1.11802743560302,
                0.565,-0.0595728575890513,-2.73140970302252,2.45920612185756,-1.11941800844721,
                0.566,-0.0574254228129745,-2.72887927819955,2.4535488096285,-1.11635588736984,
                0.567,-0.0548150262626344,-2.73049198995766,2.4567011801629,-1.11866221259624,
                0.568,-0.0523550057877963,-2.73070572936399,2.45612847861554,-1.11832306288094,
                0.569,-0.0501162770442714,-2.72848737094774,2.45128624397851,-1.11603757785505,
                0.57,-0.0476421537930501,-2.72712925411576,2.44579395497178,-1.11209555258263,
                0.571,-0.0451063275157679,-2.72762571059683,2.44537589554408,-1.11179482488923,
                0.572,-0.042776231277947,-2.72673161440458,2.44357046840942,-1.11118971983155,
                0.573,-0.0402617209557344,-2.72616860878987,2.44176535243161,-1.11040365442240,
                0.574,-0.0377340800142126,-2.72664249245385,2.44214554068832,-1.11084408326031,
                0.575,-0.0351595650797977,-2.72666673980771,2.44038989447216,-1.10948788187888,
                0.576,-0.032848051771991,-2.72580879705123,2.43809011129781,-1.10825045683212,
                0.577,-0.0300469862531265,-2.72774772378299,2.44088474852574,-1.10993177546052,
                0.578,-0.0276511318875863,-2.72801881448339,2.44157544763244,-1.11074332100964,
                0.579,-0.0250534604794135,-2.72840558720747,2.44097605424293,-1.110213481534,
                0.58,-0.0222613489713188,-2.72998452539237,2.44277811587503,-1.11112133811838,
                0.581,-0.0192791531982803,-2.73327184646719,2.4478238460302,-1.11368656967783,
                0.582,-0.0167403321224715,-2.73394500667134,2.44978179505573,-1.11555544839670,
                0.583,-0.0142414174098362,-2.73477052683245,2.45134401727845,-1.11679311037785,
                0.584,-0.0118426074075551,-2.73470585354165,2.45222192759435,-1.11834023610989,
                0.585,-0.00922793104766218,-2.73618400429463,2.45576651466502,-1.12117784527269,
                0.586,-0.006356984997731,-2.73942675087037,2.46158250445218,-1.12458989532748,
                0.587,-0.0040384171244093,-2.73824665109081,2.45933454522210,-1.12374308836128,
                0.588,-0.00153430609363440,-2.73823436233039,2.45924051442875,-1.12401500477104,
                0.589,0.000881528614895053,-2.73724067661511,2.45621983461815,-1.12226519704907,
                0.59,0.00329480748456369,-2.73663875354363,2.45416475632875,-1.12115620194777,
                0.591,0.00587418562274528,-2.73759439480578,2.45492646195374,-1.12145043383836,
                0.592,0.00813271301396099,-2.7350840111602,2.44855183010896,-1.11772066298343,
                0.593,0.0106789758593600,-2.73422715269004,2.44528966624066,-1.11581371546355,
                0.594,0.013416553975494,-2.73775488396270,2.45379306048913,-1.12172682896226,
                0.595,0.0159902906556457,-2.73889687418736,2.45538229193430,-1.12277245881316,
                0.596,0.0183009924366342,-2.73717227225239,2.45145129563128,-1.12088185055426,
                0.597,0.0207694961336595,-2.73607997938963,2.44754468614485,-1.11836091705682,
                0.598,0.0232641182948438,-2.73412090903396,2.44103068635702,-1.11399553407375,
                0.599,0.025689737911152,-2.73236918332897,2.43537679828529,-1.11000839633809,
                0.6,0.0280892678464655,-2.73237050046873,2.43565392001000,-1.11051011445519,
                0.601,0.0308758008823124,-2.73368344527323,2.43821425068057,-1.11277624450707,
                0.602,0.0334959121361322,-2.73516036674277,2.4401754506692,-1.11380545869213,
                0.603,0.0363580718339939,-2.73692742895864,2.44200488745861,-1.11451300477628,
                0.604,0.0391376191380312,-2.73832050983361,2.44350589578204,-1.11527973511452,
                0.605,0.0415523827770207,-2.73647594984508,2.43868170225359,-1.11259601925522,
                0.606,0.0440842057228736,-2.73680775011252,2.43854886280551,-1.11248761574364,
                0.607,0.047064036438151,-2.74021556462988,2.4452070835565,-1.11694434549720,
                0.608,0.0495016017161288,-2.73858455315328,2.44111365390044,-1.11468637093831,
                0.609,0.0522579969668312,-2.74083449405948,2.44546597555648,-1.11748881383547,
                0.61,0.0547301248210907,-2.73912925500567,2.44029295404182,-1.11426341722905,
                0.611,0.0575151632668935,-2.74075079118578,2.44216484435592,-1.11503916652625,
                0.612,0.0602335890784179,-2.7422606930373,2.44542651839739,-1.11758328041787,
                0.613,0.0628317088695219,-2.74260308096743,2.44583590663199,-1.11820899569263,
                0.614,0.0658258371174115,-2.74551432747221,2.45032970991637,-1.12066322348883,
                0.615,0.068657847509485,-2.74789406623562,2.45449105953334,-1.12344601422617,
                0.616,0.0716290779513504,-2.75126785259979,2.46044452793380,-1.12701421639333,
                0.617,0.0742689699765213,-2.75312632313201,2.46406150213283,-1.12933501419388,
                0.618,0.0768859146165078,-2.75158865979982,2.45851047724197,-1.12580236680498,
                0.619,0.0794143655486147,-2.75158819687682,2.45850622750876,-1.12616747622275,
                0.62,0.08179827674452,-2.74912369344833,2.45187179838338,-1.12226696187502,
                0.621,0.0842648040430879,-2.74668416695477,2.44489927201805,-1.11795417220802,
                0.622,0.086981274201796,-2.74849496158281,2.44851013403107,-1.12034986865356,
                0.623,0.0898791355001362,-2.74935370150025,2.44822109384364,-1.11999843211614,
                0.624,0.0925397324654154,-2.75034780347391,2.44948478198271,-1.12076592710038,
                0.625,0.0953952711733633,-2.75224800852028,2.45222297757645,-1.12252329039200,
                0.626,0.0981575196918747,-2.7527643918354,2.45220954209624,-1.12275668076594,
                0.627,0.100989522811278,-2.75389813155027,2.45269109256845,-1.12265642832238,
                0.628,0.103779581005206,-2.75666833270478,2.45926968831497,-1.12727000604463,
                0.629,0.106464600477449,-2.75556628846364,2.45517059435219,-1.12464037634657,
                0.63,0.108725009904428,-2.75210333255960,2.44631301058142,-1.11917492892490,
                0.631,0.111423691684525,-2.75297406188067,2.44745535131551,-1.12006580764482,
                0.632,0.113887441110570,-2.75208818878620,2.44534867152537,-1.11906242804858,
                0.633,0.116557564140155,-2.75307471877069,2.44673809000732,-1.12009923453588,
                0.634,0.119185477377185,-2.75351491290040,2.44683182809106,-1.12009083535317,
                0.635,0.121859663374169,-2.75432042000417,2.44787404007892,-1.12094499641375,
                0.636,0.124357076230228,-2.75238051647350,2.44317366976310,-1.11837354128611,
                0.637,0.127271963284186,-2.75441207625721,2.44692034366221,-1.12086635045984,
                0.638,0.130010242150887,-2.75409852920546,2.44544885902653,-1.12040171810153,
                0.639,0.132506559181897,-2.75380776258557,2.44465792921799,-1.12024780478219,
                0.64,0.135084226544354,-2.75351826638696,2.44341605882082,-1.11972713827363,
                0.641,0.137770962665463,-2.7552247818185,2.44689627499645,-1.12189419970853,
                0.642,0.140551906345176,-2.75707980404494,2.45014269813062,-1.12386941918484,
                0.643,0.143533774132002,-2.75963999945024,2.45510913690724,-1.12722902013212,
                0.644,0.146306308510753,-2.76088082658716,2.45691951096331,-1.12837047967506,
                0.645,0.148898402744231,-2.76025819623828,2.45533794070513,-1.12783802032338,
                0.646,0.151458799848583,-2.76013576222194,2.45458201693523,-1.12757871822874,
                0.647,0.154296307283625,-2.76112451862419,2.45486938847781,-1.12735157427101,
                0.648,0.156944747308350,-2.76184677383084,2.45606481819071,-1.12835516541320,
                0.649,0.159737140600052,-2.76357910927194,2.45975706297864,-1.13087678739104,
                0.65,0.162352936882786,-2.76439654436773,2.46067002821719,-1.13128914469252,
                0.651,0.165087953011025,-2.76525660162118,2.46264549417299,-1.13307302238191,
                0.652,0.167974421933035,-2.76701914417646,2.46550383458692,-1.13492892309167,
                0.653,0.170732517314724,-2.76750161602565,2.46552571894641,-1.13510618183204,
                0.654,0.173453580301900,-2.76805455126587,2.46539695346156,-1.13477702253671,
                0.655,0.176584377612525,-2.77095073517776,2.46896886503568,-1.13639190606768,
                0.656,0.179397426364083,-2.77139668004005,2.46910325145767,-1.13678189519316,
                0.657,0.182208492872870,-2.77317285400824,2.47192790063292,-1.13833674720521,
                0.658,0.184751295984455,-2.77209100548484,2.46952104265762,-1.13749282683526,
                0.659,0.187229903027282,-2.77075271982083,2.46617405568304,-1.13575276895896,
                0.66,0.189909283291119,-2.77194704865321,2.46840186775718,-1.13730661518684,
                0.661,0.192885343272502,-2.77372769880346,2.47098563218115,-1.13882868943876,
                0.662,0.195633016237745,-2.77535775931535,2.47523124631510,-1.14217742028743,
                0.663,0.198553384553973,-2.77671764454405,2.47656696082944,-1.14282678024546,
                0.664,0.201186158337469,-2.77700885157696,2.47698386111476,-1.14333359995408,
                0.665,0.203839659058872,-2.77590706232553,2.47216061547869,-1.13970000778036,
                0.666,0.207051476680778,-2.77933296993012,2.47834887832711,-1.14342902281194,
                0.667,0.209747914431445,-2.78011345416072,2.47968892035239,-1.14445579169911,
                0.668,0.212235657814924,-2.77862608978333,2.47594303919171,-1.14216098322333,
                0.669,0.214845206435260,-2.77941357365173,2.47823885153919,-1.14424264082638,
                0.67,0.218256922790963,-2.78464989403437,2.48779283208292,-1.14961560871823,
                0.671,0.220702412600895,-2.78340794587557,2.48536080044544,-1.14852112840699,
                0.672,0.223576047857699,-2.7847515628841,2.48688741882777,-1.14914415788632,
                0.673,0.226110080956426,-2.78284772909107,2.48170176791834,-1.14618854315361,
                0.674,0.228802824924277,-2.78284135317984,2.48134054299644,-1.14615672933775,
                0.675,0.231562506547450,-2.78400824235250,2.48427782168454,-1.14837122711267,
                0.676,0.234301732916654,-2.78391561406465,2.4825482046992,-1.14697348161651,
                0.677,0.237069228936193,-2.78406451599359,2.48179572570737,-1.14651434355814,
                0.678,0.239595743352975,-2.78293713973210,2.47906503516479,-1.14515565668959,
                0.679,0.242784171553322,-2.78616510882863,2.48448907218128,-1.14807080895551,
                0.68,0.245734641252993,-2.78750853274393,2.48683670503017,-1.14972568977165,
                0.681,0.248404423807332,-2.78734539160371,2.48612400875918,-1.14959535418839,
                0.682,0.251017284374976,-2.7855284686002,2.48113308049409,-1.14660876082778,
                0.683,0.253839472838733,-2.78465839135819,2.47861733788489,-1.14545919667488,
                0.684,0.256817143810152,-2.78684803123555,2.48306937961501,-1.14853524943694,
                0.685,0.259826696921175,-2.78810365161303,2.48335608515523,-1.14820628807517,
                0.686,0.262412256904589,-2.78616650886855,2.47810526692543,-1.14495441861472,
                0.687,0.265243472223915,-2.78685025857567,2.47924492014382,-1.14605496426605,
                0.688,0.268102732783272,-2.78644401192781,2.47776529991612,-1.14561906078266,
                0.689,0.270924176489093,-2.78671407858056,2.47773860373141,-1.14599963805892,
                0.69,0.273858426681038,-2.78853649217338,2.48070549943166,-1.14778983502535,
                0.691,0.276583675233398,-2.78813127853969,2.47879426206757,-1.14640252823666,
                0.692,0.279799984644209,-2.79285825991398,2.48903965854608,-1.15304239449680,
                0.693,0.282710761718538,-2.79254518816208,2.48653496919426,-1.1513562189356,
                0.694,0.285352667785025,-2.79065679435574,2.480709685713,-1.14748119606357,
                0.695,0.288041374074541,-2.79082823234981,2.48149406858949,-1.14846986087282,
                0.696,0.290987063281057,-2.79167530197262,2.48081633621844,-1.14719995231767,
                0.697,0.293877184524065,-2.79278519748225,2.48159967540504,-1.14713448725536,
                0.698,0.296473333859471,-2.79109701730062,2.4773241107254,-1.14456291722045,
                0.699,0.299122869602911,-2.78974572818102,2.47289580581691,-1.14146286711486,
                0.7,0.301843115428905,-2.79016941456017,2.47405190457852,-1.14239276455255,
                0.701,0.304619553052003,-2.79080991082778,2.47604414613308,-1.14417722736059,
                0.702,0.307614039539394,-2.79187972037794,2.47728020551135,-1.14488682028476,
                0.703,0.310258738557102,-2.79019058248892,2.47289184385755,-1.14233964825325,
                0.704,0.313536103988656,-2.79406190034132,2.47966356837197,-1.14610947115011,
                0.705,0.316402118514786,-2.79474728838208,2.48094667230803,-1.14699058273361,
                0.706,0.319293592768345,-2.79332377055075,2.4758629241167,-1.14374278087742,
                0.707,0.322153620325003,-2.79381496894122,2.47569001563215,-1.14336413656422,
                0.708,0.325058881656917,-2.79364180758901,2.47328090014544,-1.14141772504303,
                0.709,0.327887873604308,-2.79345702633551,2.47271557711910,-1.14131010502801,
                0.71,0.330911532028339,-2.79530284097862,2.47672335628973,-1.14416602359518,
                0.711,0.333925016298262,-2.79742250616146,2.48101062731839,-1.14701996017156,
                0.712,0.336956822285383,-2.79740717601428,2.47986305295236,-1.14652488932672,
                0.713,0.339812682413374,-2.79776787997287,2.48030928131405,-1.14682382846089,
                0.714,0.342880193853163,-2.79972402229537,2.48433343518359,-1.1495185020355,
                0.715,0.345487402556173,-2.79769124159614,2.47934170374898,-1.14652464505436,
                0.716,0.348505479486966,-2.79854896058911,2.48036114432497,-1.14710829917481,
                0.717,0.351492878336187,-2.79913321683758,2.4810128123437,-1.14766636448028,
                0.718,0.354737912588971,-2.80066378406123,2.48209579783876,-1.14789310891922,
                0.719,0.357854824200008,-2.80232370005464,2.48455999743875,-1.14914362440709,
                0.72,0.360823931143181,-2.80211251580585,2.48292265694749,-1.14812631057468,
                0.721,0.363946458762244,-2.80288121318791,2.48285831156353,-1.14778709465633,
                0.722,0.367178753668815,-2.80586842830855,2.48865772426228,-1.15138612162222,
                0.723,0.370173083967037,-2.80603033003256,2.48782410596588,-1.15086117952783,
                0.724,0.373316369620959,-2.80764910750932,2.48981993209875,-1.15174985681499,
                0.725,0.375976922753323,-2.80476701297043,2.48360022962746,-1.14854035006308,
                0.726,0.379057466675954,-2.80603244881648,2.48565761052259,-1.14975249423707,
                0.727,0.381971287333192,-2.80636475919527,2.48590294633756,-1.1501332815456,
                0.728,0.38534378333123,-2.80851316512184,2.48777221007845,-1.15079288593664,
                0.729,0.388466787620727,-2.81039980968753,2.49166717481861,-1.15344456327789,
                0.73,0.391596273580067,-2.8120022197826,2.49354145097096,-1.15408020888721,
                0.731,0.394593386752793,-2.81271669858255,2.49480426116018,-1.15526703180418,
                0.732,0.397381557208798,-2.8114835517704,2.49092594926275,-1.15275414344143,
                0.733,0.400477713291631,-2.81243277442093,2.49305253446335,-1.15446905959512,
                0.734,0.403612815677279,-2.81338288054170,2.49414894892628,-1.15497177093266,
                0.735,0.406647387803249,-2.81445086523317,2.49655246549049,-1.15694536102251,
                0.736,0.409784288405904,-2.81500722644986,2.49518348414561,-1.1552552630456,
                0.737,0.413173533857945,-2.8175859085998,2.49878157905449,-1.15687084189473,
                0.738,0.416745725237244,-2.82180105346676,2.50667508160413,-1.16150325915351,
                0.739,0.419594434057121,-2.82046259987255,2.50172970376611,-1.15795200283521,
                0.74,0.422611422087270,-2.82007903561032,2.49910369766357,-1.15593889072404,
                0.741,0.425904986036784,-2.82173213108134,2.50063379784105,-1.15641372191734,
                0.742,0.428570806857251,-2.81948912284363,2.49624516038819,-1.15413031061349,
                0.743,0.431601988428972,-2.81914146699299,2.49392620644158,-1.15241326282287,
                0.744,0.434999279700136,-2.81940520185867,2.49048153330015,-1.14914994331372,
                0.745,0.438187032886671,-2.81984239943333,2.49025880820604,-1.14894987218934,
                0.746,0.441259516154035,-2.82077477941365,2.49165998177518,-1.14956290226594,
                0.747,0.443948595664330,-2.81788129230861,2.48435429469201,-1.14479389871681,
                0.748,0.446858596436588,-2.81681109709114,2.48045066263456,-1.14212046410771,
                0.749,0.44985978486172,-2.81672977490542,2.480271996818,-1.14215551415259,
                0.75,0.453254608455155,-2.81943306896372,2.4858179995071,-1.14595823393009,
                0.751,0.456260834975306,-2.82030225749399,2.48768926493076,-1.14719476025128,
                0.752,0.459375938665539,-2.82127128335201,2.48951156103552,-1.14854545436831,
                0.753,0.462530037703551,-2.82263859167744,2.4921163531631,-1.15017558507413,
                0.754,0.465375343141303,-2.82155681336967,2.48927691363179,-1.14833811303466,
                0.755,0.468231636601106,-2.81840204692388,2.48064037061667,-1.14257243070662,
                0.756,0.471521330376382,-2.81858634912312,2.47798425650985,-1.13981922002817,
                0.757,0.474775708630185,-2.81910673158603,2.47719065890755,-1.13893781787119,
                0.758,0.477767488529691,-2.81772488694220,2.4733805135259,-1.13657579025196,
                0.759,0.480822306256542,-2.81774637162809,2.47291495465853,-1.13638825627585,
                0.76,0.484073355020016,-2.81663837022439,2.4682171389762,-1.13310886332407,
                0.761,0.487170225412601,-2.81572161599973,2.46567035664467,-1.13180470515233,
                0.762,0.490558675317509,-2.81698279208156,2.46652149109812,-1.13187443712987,
                0.763,0.493949580778455,-2.81928295693729,2.47000682872564,-1.13347927788389,
                0.764,0.496805945600273,-2.81664029475431,2.46362024278464,-1.12967619324490,
                0.765,0.499877378890191,-2.81484035276039,2.45808835403102,-1.12603929103182,
                0.766,0.503211255931565,-2.81581040638365,2.45908576564085,-1.12660821274106,
                0.767,0.506561561239164,-2.81618829923842,2.45872064535358,-1.12618436632157,
                0.768,0.509842619167242,-2.81823571752485,2.46305386659991,-1.12883546066144,
                0.769,0.512906775383512,-2.81589932167639,2.45628662066386,-1.12419238810487,
                0.77,0.516056354951638,-2.81558528525843,2.45530222645135,-1.12381005888915,
                0.771,0.519125998726695,-2.81458077600941,2.45170325571490,-1.12124689626594,
                0.772,0.522565345254525,-2.81555850008741,2.45223905628972,-1.12118071901627,
                0.773,0.525670541138887,-2.81470786115387,2.44966814602075,-1.11938379846344,
                0.774,0.528736160942278,-2.81457224430144,2.4502164481187,-1.12026716883642,
                0.775,0.532064405777437,-2.81507384794333,2.44950779438377,-1.11921355560811,
                0.776,0.535404699720111,-2.81525872177684,2.44949453332295,-1.11937417747541,
                0.777,0.538978212229794,-2.81799693581463,2.45537759835760,-1.12334388814779,
                0.778,0.542467961879572,-2.81904355267165,2.4560765124039,-1.12356849142233,
                0.779,0.546185556577924,-2.82147373734338,2.45798866318718,-1.12347061437456,
                0.78,0.549649880480155,-2.82236173750380,2.457698890612,-1.12247886353799,
                0.781,0.552842912680939,-2.82122031102548,2.45278557281227,-1.11848511644650,
                0.782,0.556199694648921,-2.82167768708697,2.45228707661811,-1.11777633782722,
                0.783,0.559690382221369,-2.82327942751733,2.45546069649143,-1.11993610836307,
                0.784,0.56292886510269,-2.82286276769918,2.4548158000871,-1.11998370539039,
                0.785,0.566440507209887,-2.82486421943908,2.45898686133166,-1.12293048230322,
                0.786,0.570159994850938,-2.82913794101215,2.4686638670761,-1.12904219860161,
                0.787,0.573605208088593,-2.8296448037459,2.46891773622742,-1.12936153640522,
                0.788,0.576927494192152,-2.82834085317976,2.46286998324227,-1.12464448827765,
                0.789,0.580775833284138,-2.82994084222609,2.46359675313709,-1.12451588485272,
                0.79,0.584512147060156,-2.83356339403013,2.47025980313444,-1.12822168452241,
                0.791,0.587925375726282,-2.83310090915997,2.46766646381009,-1.12604944497705,
                0.792,0.591492041884525,-2.83491690979928,2.46996208299518,-1.12677296135679,
                0.793,0.594474345985269,-2.83153166434774,2.46199259789664,-1.12190346992554,
                0.794,0.597672133008014,-2.82890897611664,2.45438571412026,-1.11672870371909,
                0.795,0.601330609151649,-2.83013958761151,2.45625112521298,-1.11785812385015,
                0.796,0.60506922846492,-2.83202232320758,2.45757423309758,-1.11773618347700,
                0.797,0.609181830832246,-2.83742803042998,2.46808881692320,-1.12408452316971,
                0.798,0.612623050475054,-2.83778281753224,2.46805506539344,-1.12419339431053,
                0.799,0.616070225289946,-2.83607953814113,2.46229295877197,-1.12008725850443,
                0.8,0.619569707646688,-2.83674715003202,2.46381952572131,-1.12126114675608,
                0.801,0.623306334820896,-2.83817394826755,2.46518180795615,-1.12160180953852,
                0.802,0.626563871457423,-2.83633353769387,2.46091024049172,-1.11914318282625,
                0.803,0.630303059247186,-2.83691676599376,2.46024532171516,-1.11825397346610,
                0.804,0.634071596998181,-2.83928293751461,2.46516455651089,-1.12133990840906,
                0.805,0.637656210022878,-2.83946342820702,2.46423477370078,-1.12048227256228,
                0.806,0.641091379067608,-2.83819299101488,2.46031690316782,-1.11790634387982,
                0.807,0.644948716465467,-2.83922715577191,2.46063316837950,-1.11790942351562,
                0.808,0.648123883845807,-2.83460009631977,2.44933106698074,-1.11104641647210,
                0.809,0.651716672733796,-2.8344040090833,2.44867133008877,-1.11050186737547,
                0.81,0.65555949019414,-2.83587890335542,2.45147149842548,-1.11234856595254,
                0.811,0.659643966313838,-2.83962846155166,2.457669145474,-1.11564478526215,
                0.812,0.663727385031908,-2.84295894284437,2.46388708212363,-1.11908132151494,
                0.813,0.66753003577696,-2.84400338545090,2.46483065845692,-1.11931393014481,
                0.814,0.671566339384641,-2.84737843802013,2.47179441321866,-1.12391898629754,
                0.815,0.675509328621146,-2.84878258364347,2.47292673356285,-1.12426057726654,
                0.816,0.679413053546511,-2.84918053642013,2.47138590990816,-1.12271651424411,
                0.817,0.683096219056864,-2.8489879497504,2.46941061257220,-1.12095190440761,
                0.818,0.687072415660923,-2.84939135741587,2.46727073194015,-1.11879660817919,
                0.819,0.690794784394538,-2.84839231233974,2.46396941313880,-1.11686453148991,
                0.82,0.694539631639765,-2.84934499227554,2.46558276857356,-1.11789283006028,
                0.821,0.698141808902104,-2.84864719460945,2.46357759455195,-1.11667291589231,
                0.822,0.702403752982144,-2.85061811443874,2.46387329317635,-1.11559765693212,
                0.823,0.706335259063444,-2.85202996948726,2.46612903031012,-1.11696692001743,
                0.824,0.709879414361802,-2.85114615994309,2.46381056189406,-1.11545006903994,
                0.825,0.713248912751122,-2.84733708539171,2.45511189034110,-1.1100403994421,
                0.826,0.716953755354932,-2.84461620318493,2.44672798537789,-1.10425024083967,
                0.827,0.720915685596677,-2.84706315284339,2.45190556955457,-1.10746836529896,
                0.828,0.725131517853033,-2.84877038685128,2.45390164813947,-1.10893884009652,
                0.829,0.729444950809428,-2.85191349530030,2.45831351915597,-1.11073547401903,
                0.83,0.733525077606594,-2.85112371219328,2.45274088401518,-1.10580981914733,
                0.831,0.737784914942571,-2.85540830962905,2.46139919977402,-1.11071797600827,
                0.832,0.741788262679329,-2.85926428840512,2.47126039798757,-1.11725388532523,
                0.833,0.746015404234608,-2.86246003116414,2.47804449035349,-1.12123841539198,
                0.834,0.750253058313142,-2.86416271698882,2.4795014670416,-1.12135439708676,
                0.835,0.754151469705117,-2.86373720151893,2.47701852491411,-1.11925408740204,
                0.836,0.758072694401155,-2.86217995563524,2.47069962085358,-1.11406280666820,
                0.837,0.762112733469373,-2.86200478877317,2.46954972737183,-1.11343713456396,
                0.838,0.766073750017463,-2.86136547203832,2.46759233873228,-1.11240644771047,
                0.839,0.77000680120953,-2.86153535998078,2.46762009573708,-1.11231271767173,
                0.84,0.77399853189667,-2.86098001679356,2.46398175348131,-1.10921671895199,
                0.841,0.778353440378512,-2.86523033928327,2.47435289384891,-1.11587597594906,
                0.842,0.782376502554442,-2.86358935692307,2.46972189707789,-1.11315690282655,
                0.843,0.786618321879187,-2.86338485639007,2.46579187644498,-1.10963420485211,
                0.844,0.79091103763815,-2.86569644900773,2.46920523583349,-1.11101896110020,
                0.845,0.794761438016169,-2.86289836865102,2.46154769621922,-1.10573710919153,
                0.846,0.798692074545781,-2.86128744025424,2.45539458515489,-1.10079892156588,
                0.847,0.802609569583003,-2.85916361081215,2.44764844470370,-1.09444924447139,
                0.848,0.806347636923768,-2.85387539420484,2.43490044017149,-1.08650322463850,
                0.849,0.810375211188032,-2.85397678736981,2.43615385727265,-1.08768425946063,
                0.85,0.814535259938465,-2.85325713747873,2.43341772493065,-1.08581232787526,
                0.851,0.819023616528834,-2.8553041402292,2.43615437987596,-1.08710749655586,
                0.852,0.82281144688883,-2.85045578018185,2.42370662972603,-1.07871381660583,
                0.853,0.82682429678064,-2.84761942197134,2.41563043213536,-1.07307315956286,
                0.854,0.831395142601552,-2.84973203200772,2.41942483810978,-1.07543996652860,
                0.855,0.835993635394766,-2.85306562488223,2.42695832045166,-1.08050594178439,
                0.856,0.840353885343651,-2.85189838972587,2.42167505117761,-1.07627746106382,
                0.857,0.844889029874608,-2.85456994033790,2.42704066177530,-1.07951582894084,
                0.858,0.849763488455597,-2.85907224470331,2.43563091216723,-1.08456091234077,
                0.859,0.854627154322323,-2.86274667950991,2.44231829041755,-1.08841608458342,
                0.86,0.859263714575497,-2.86446909411989,2.44433862164473,-1.08906912153132,
                0.861,0.863851809591842,-2.86373713307321,2.44041320727822,-1.08598962160555,
                0.862,0.868064315290845,-2.8611813124199,2.43407590642651,-1.08210908189963,
                0.863,0.87199928643945,-2.85820390394517,2.42875238307098,-1.07915950082773,
                0.864,0.875839056977387,-2.85191071143855,2.41269769081843,-1.06850176937487,
                0.865,0.88033512447832,-2.85116971719282,2.40835183277648,-1.06471664303183,
                0.866,0.884980565725004,-2.85203754071946,2.40848624398087,-1.06415252123953,
                0.867,0.889890035299967,-2.85604457244682,2.41606173535357,-1.06799584838857,
                0.868,0.894605495733318,-2.85513757609871,2.41023382193604,-1.06306890725567,
                0.869,0.899300763979352,-2.85419747275584,2.40657679906184,-1.06017115613492,
                0.87,0.9040351964303,-2.85550861009346,2.40902173408519,-1.06164633566352,
                0.871,0.908622614663947,-2.85576888943626,2.40842670878386,-1.06067658586659,
                0.872,0.913158038569156,-2.85641133379922,2.41049806300254,-1.06225053241592,
                0.873,0.918241704058523,-2.86011396070361,2.41776451158354,-1.06632670207853,
                0.874,0.923431260823916,-2.86374318233264,2.42454649141690,-1.07021643088630,
                0.875,0.927876481325605,-2.86092365510856,2.41711313338745,-1.06516285107367,
                0.876,0.933140479898142,-2.86514249435683,2.42456554393995,-1.06872505135434,
                0.877,0.93755915792726,-2.86320402890223,2.42115757652702,-1.06695430374847,
                0.878,0.94223440553139,-2.86154459987766,2.41703478535717,-1.06465062486308,
                0.879,0.946886997849345,-2.85999826244617,2.41304499445954,-1.06196958104605,
                0.88,0.95133032979424,-2.85632630275971,2.40461312141719,-1.05709683309091,
                0.881,0.95563913164566,-2.85233889130042,2.39416362479301,-1.04979638470205,
                0.882,0.960890625754388,-2.85162521319182,2.38685398907072,-1.04296429482350,
                0.883,0.965728474825057,-2.85022886514726,2.38277939153826,-1.03965007120464,
                0.884,0.971058622154406,-2.85371657589717,2.39071689398257,-1.04456715995921,
                0.885,0.976352797846036,-2.85457937731714,2.38992829139616,-1.04303603320079,
                0.886,0.98129063328854,-2.8527725666489,2.38440778897631,-1.03923494203277,
                0.887,0.986755919407307,-2.85158249496874,2.37670852257337,-1.03268444020596,
                0.888,0.991909248283105,-2.85194763811106,2.37538995398192,-1.03133207838319,
                0.889,0.997446342222495,-2.85513898622119,2.37929545713178,-1.03242344305993,
                0.89,1.00325152576399,-2.86126015084447,2.39170412196928,-1.03967377831849,
                0.891,1.00887547702212,-2.86362440636524,2.39375219054148,-1.03933416788527,
                0.892,1.01455007606907,-2.86598630894444,2.39591044406987,-1.03966723927841,
                0.893,1.01957901020427,-2.8651907927929,2.39491831442211,-1.03925993563570,
                0.894,1.02467577168643,-2.86314232158998,2.39052668398684,-1.03676604848610,
                0.895,1.02988390267645,-2.86158700793669,2.38562947948869,-1.03339684461634,
                0.896,1.03547573962249,-2.86137065585329,2.38242276778082,-1.03070509833450,
                0.897,1.04061940288120,-2.858922630234,2.37771777163826,-1.02815328644141,
                0.898,1.04594028121540,-2.8582408789757,2.37360355553745,-1.02393088516902,
                0.899,1.05181610216367,-2.86149553225908,2.37945326650607,-1.02687072673599,
                0.9,1.05678474278886,-2.85666806668554,2.36470498499653,-1.01550172192016,
                0.901,1.06261401277333,-2.85644897436651,2.36217090938621,-1.01335997335328,
                0.902,1.06874277318613,-2.86095739748169,2.37012112309515,-1.01734536445847,
                0.903,1.07442379395386,-2.85912398335051,2.36440347298652,-1.01346441155307,
                0.904,1.07977093790628,-2.85706353217651,2.35834335395948,-1.00861856391308,
                0.905,1.08544613782356,-2.85867420157605,2.36276808185321,-1.01182593631286,
                0.906,1.09173011308832,-2.86083179847529,2.36579811912039,-1.01310456310889,
                0.907,1.09748907954301,-2.86118472011445,2.36579740995810,-1.01274971384206,
                0.908,1.10383419907286,-2.86475350407358,2.37237595830434,-1.01604639596007,
                0.909,1.1101936291942,-2.87060833382631,2.38761170727324,-1.02636924647323,
                0.91,1.11589367481865,-2.86969468363178,2.3844411164382,-1.02363520686307,
                0.911,1.12222710819810,-2.87197842654179,2.38761810863832,-1.02510330083028,
                0.912,1.12898454170535,-2.87749058503629,2.39661154106668,-1.02928603867827,
                0.913,1.13523091019796,-2.8773973708662,2.39493398437906,-1.02733077465781,
                0.914,1.14193149595039,-2.88246879858656,2.40694638062849,-1.03539273029046,
                0.915,1.14826604121550,-2.88221899999589,2.4050590329259,-1.03363994269640,
                0.916,1.15459892523254,-2.88350271063234,2.40824244789258,-1.03521670393182,
                0.917,1.16074296176871,-2.88234410504354,2.40420803888132,-1.03173955185713,
                0.918,1.16756127909974,-2.88391996456519,2.406009446551,-1.03226679604494,
                0.919,1.17387357587380,-2.88541185059695,2.40976732100799,-1.03434823859159,
                0.92,1.18041482452308,-2.88211688274417,2.40034182445696,-1.02821646360116,
                0.921,1.18708404780342,-2.88195809409918,2.39984404464323,-1.02803506340318,
                0.922,1.19325006546670,-2.87892460609429,2.39469042986234,-1.02537443892633,
                0.923,1.20044003765180,-2.88148295061966,2.39897056257127,-1.02787605195380,
                0.924,1.20767420524777,-2.88662044106639,2.41223315340184,-1.03711910892870,
                0.925,1.21560268424279,-2.89451293207186,2.42862547434656,-1.0475597700928,
                0.926,1.22256092026760,-2.89569234362714,2.42977874334591,-1.04691421999426,
                0.927,1.23013261318168,-2.89623214636132,2.42646127104237,-1.04328016019110,
                0.928,1.23730034091828,-2.89863425513565,2.43099260284055,-1.04552540147816,
                0.929,1.24446001074506,-2.90121064046754,2.436502386134,-1.04859942264764,
                0.93,1.25200995638628,-2.90699744528368,2.44928951611335,-1.05569354639373,
                0.931,1.2598638253479,-2.90937304773548,2.45080117577635,-1.05516808485720,
                0.932,1.26668222258568,-2.90508282486411,2.44389440250457,-1.05241725424914,
                0.933,1.27390900016296,-2.90205251941600,2.43449490143413,-1.04503119881846,
                0.934,1.28061503232856,-2.89608249662187,2.42468007078186,-1.04022695894599,
                0.935,1.28799226161296,-2.89571252464487,2.42305467051677,-1.03826496571213,
                0.936,1.29543897928278,-2.89477557997749,2.42347782673872,-1.03958686981853,
                0.937,1.30371373418388,-2.89533634165018,2.42148384824872,-1.03679769932862,
                0.938,1.31214200487819,-2.90196993890519,2.43620824351372,-1.04524010628259,
                0.939,1.31978720602181,-2.89961378782043,2.43084582869877,-1.04172070906108,
                0.94,1.32887187126713,-2.90187899074044,2.43089773669995,-1.03940286139523,
                0.941,1.33814741030866,-2.91220886769694,2.45361220299123,-1.05307292116501,
                0.942,1.34746971928788,-2.91538897902511,2.45474972609273,-1.05136814031152,
                0.943,1.35651740673205,-2.92000518830698,2.46480621682283,-1.05773728075497,
                0.944,1.36577030812561,-2.9269437060758,2.48103111566168,-1.06789798402493,
                0.945,1.37655615273811,-2.94199711828281,2.50766113387833,-1.08178779121520,
                0.946,1.38567030855097,-2.94621057126384,2.51617022943696,-1.08624359482008,
                0.947,1.39469304791164,-2.94762524539497,2.52252042284488,-1.09175843847283,
                0.948,1.40441092554970,-2.95286198902941,2.53159355900149,-1.09641292361180,
                0.949,1.41494726936603,-2.96293758837419,2.55244833253366,-1.10862568915841,
                0.95,1.42436285158074,-2.96084724953592,2.54877759000967,-1.10652872434293,
                0.951,1.4337211316085,-2.95916169497063,2.54227476472034,-1.10041945886076,
                0.952,1.44447551188983,-2.96518329335348,2.55200033780875,-1.10464575621492,
                0.953,1.45487492497815,-2.972743782803,2.57253567948043,-1.11811736957678,
                0.954,1.46624547624773,-2.98266110245188,2.58790873574257,-1.12430059421006,
                0.955,1.47520369080540,-2.97026189518195,2.55721698370195,-1.10418691179238,
                0.956,1.48572991752776,-2.9701369111561,2.55562899512309,-1.10195226014480,
                0.957,1.49580208222591,-2.97076476933593,2.56414801618783,-1.10944697566194,
                0.958,1.5068796646923,-2.97352400748403,2.56741362093675,-1.10956315624131,
                0.959,1.51852113939532,-2.97899715738683,2.57344948431508,-1.11014989688292,
                0.96,1.53022167234643,-2.98011326068392,2.57194359193369,-1.10757575970965,
                0.961,1.54092043095263,-2.97502216640431,2.55869464228842,-1.09733677553120,
                0.962,1.55350612648668,-2.9827001535709,2.57513513877565,-1.10677959246409,
                0.963,1.56553790952560,-2.98258826427109,2.5717410370621,-1.10279498169707,
                0.964,1.57888148078768,-2.99516901287854,2.60028605061267,-1.12014068800057,
                0.965,1.59158540152548,-2.99402941537117,2.59587852087567,-1.11682300111251,
                0.966,1.60523675236956,-2.99969888752359,2.60309880578879,-1.11863621245681,
                0.967,1.61878528059494,-3.0054438122729,2.61403916182994,-1.12281000006730,
                0.968,1.63159182663982,-3.00442454496978,2.61684724058800,-1.12531813376720,
                0.969,1.64679901294935,-3.01345263938327,2.62972442480104,-1.12993803053873,
                0.97,1.66158082886222,-3.01471134153798,2.62715043865361,-1.12638445619767,
                0.971,1.67792740306183,-3.0291036332115,2.65438168889903,-1.14092287087381,
                0.972,1.69488469524672,-3.0402659850218,2.67160246619159,-1.14838368009454,
                0.973,1.70885610133724,-3.03011888844583,2.65188574967467,-1.13680012661267,
                0.974,1.72774091660669,-3.04835072416923,2.68837606380769,-1.15704800526889,
                0.975,1.74377976175596,-3.0444332251232,2.68161912022328,-1.15293877933525,
                0.976,1.76202648542675,-3.05268279170655,2.69455380855646,-1.15694548249028,
                0.977,1.78017629253165,-3.05059096660394,2.68672654658135,-1.15112014339909,
                0.978,1.79841914786173,-3.04581488524879,2.67161805247135,-1.13917591632243,
                0.979,1.81562975437973,-3.02769170476772,2.62411115539633,-1.10447099514999,
                0.98,1.83571332305234,-3.0245468474582,2.60988260492956,-1.09176538755244,
                0.981,1.85544846246025,-3.03278379519325,2.64825126434187,-1.12270846277526,
                0.982,1.87521566046331,-3.02255671352379,2.62639486146602,-1.107015406397,
                0.983,1.89648636415124,-3.00164536072361,2.58028622481280,-1.08056714501013,
                0.984,1.92146318151857,-3.01531269156013,2.60270938700317,-1.08918726736828,
                0.985,1.94660206919755,-3.00643280138591,2.57798997298795,-1.07257597401632,
                0.986,1.97441637467628,-3.02763052024518,2.64130701311848,-1.11711254878136,
                0.987,1.99878129520076,-2.98711419216223,2.56380457904741,-1.07560202026432,
                0.988,2.03256802027827,-3.01038110212046,2.6052312356989,-1.09330946195793,
                0.989,2.06349689790421,-3.00646183534944,2.5930851256399,-1.08161870629739,
                0.99,2.10083460397733,-3.01815317081042,2.61507247173373,-1.09280031776964,
                0.991,2.1422856856122,-3.02614344773863,2.60579843752675,-1.07390985863287,
                0.992,2.18713154545622,-3.04975356321037,2.66212619032211,-1.110724464012,
                0.993,2.24052476003954,-3.10081984993664,2.77025669906651,-1.17364172859743,
                0.994,2.29039668269601,-3.07243948479771,2.71932170840635,-1.14402209129542,
                0.995,2.34886646208379,-3.03044427338570,2.63365511255221,-1.09347277685986,
                0.996,2.42933936303139,-3.04139889702872,2.65529326614934,-1.11159643507674,
                0.997,2.52526809868838,-3.06378110890239,2.68303810020251,-1.11214753401839,
                0.998,2.64430122839061,-2.97282741938855,2.49225188627017,-0.997201697156246,
                0.999,2.84480326778455,-2.89877942323658,2.24377059741797,-0.779093386012649,
                0.99925,2.90935044527079,-2.68788161834596,1.77942463736579,-0.494298088956325,
                0.9995,3.02865191049732,-2.69035900081279,1.75726508465720,-0.467337686423802,
                0.99975,3.24362494692231,-2.81274991321387,1.86975827458549,-0.481710345384369)
 coeffs_nc <- c(0.00025,-3.54769045057301,-0.426595058633463,0.702434880326509,-0.403419444412317,
                5e-04,-3.36184750763457,-0.42229663576275,0.68123072401489,-0.399588320511102,
                0.00075,-3.24827158654778,-0.408254456988444,0.588065345756167,-0.316656205006621,
                0.001,-3.16555492079897,-0.416465684035444,0.598520618691402,-0.319131240741446,
                0.002,-2.92229790117722,-0.696910299539636,1.06103554865436,-0.546643140636167,
                0.003,-2.80208146980152,-0.659979117891795,0.965905805119292,-0.477972220219219,
                0.004,-2.70745169116045,-0.626946914198399,0.843579878621432,-0.386865018091449,
                0.005,-2.63833832093541,-0.608629006296792,0.807560910111462,-0.368069483762163,
                0.006,-2.57725541573185,-0.585964867385474,0.756338471375501,-0.340948828900006,
                0.007,-2.52649308367391,-0.569945387219101,0.7197003314222,-0.317815282179706,
                0.008,-2.48085826245442,-0.552886101363841,0.686609636264419,-0.304933381708511,
                0.009,-2.43243899189791,-0.609473705871165,0.810205583299187,-0.383216976210421,
                0.01,-2.39823691350270,-0.586685712617206,0.764246988140023,-0.359678815666942,
                0.011,-2.36055281954393,-0.612858621420206,0.822479414494401,-0.397398352264553,
                0.012,-2.33045326566649,-0.605493794973383,0.813502081840391,-0.397306038099498,
                0.013,-2.29897392439774,-0.624681885453038,0.860600935686291,-0.430332380218443,
                0.014,-2.26850130726153,-0.645645704314619,0.90863579363877,-0.459775302757583,
                0.015,-2.2395311336513,-0.650700180010348,0.895201725931872,-0.443651924979829,
                0.016,-2.21345471152244,-0.648021390848664,0.87655644359823,-0.430907776449462,
                0.017,-2.18618225236920,-0.656225119147759,0.870570709116836,-0.419507873295635,
                0.018,-2.16416176455691,-0.646289603525755,0.84053822757378,-0.39792073256334,
                0.019,-2.14261711671788,-0.634513952400539,0.799555307444569,-0.368285196948889,
                0.02,-2.12045064218626,-0.642453115155679,0.808073694410975,-0.372312775298245,
                0.021,-2.10215324536081,-0.63000722431516,0.774978582266686,-0.349717245179345,
                0.022,-2.08198336764514,-0.643800075612341,0.801507096855847,-0.364900422515371,
                0.023,-2.06429179175601,-0.634074227818445,0.766559367372083,-0.337870321572122,
                0.024,-2.04638172067048,-0.637309765511603,0.772271999015361,-0.341833394441960,
                0.025,-2.02920071224299,-0.636280677677828,0.77463485884847,-0.346728500974855,
                0.026,-2.01174387985953,-0.650356516891999,0.806103580260946,-0.367580395925461,
                0.027,-1.99457491466251,-0.66030402919072,0.825380194488618,-0.378923428936635,
                0.028,-1.97978569073370,-0.650423834593184,0.80105144963487,-0.364409992433957,
                0.029,-1.96446957041559,-0.649850448787679,0.793501215409204,-0.358397411979519,
                0.03,-1.95042110815901,-0.643573808214985,0.78039426984148,-0.352051425700820,
                0.031,-1.93483001008528,-0.649842353316446,0.785879269653073,-0.353193763483436,
                0.032,-1.92104356426488,-0.647879505057148,0.777933797817133,-0.347978328957488,
                0.033,-1.90564230855716,-0.668910710085467,0.821845118042865,-0.375088361189972,
                0.034,-1.89343678067591,-0.654636892858503,0.784472092484832,-0.350588384107762,
                0.035,-1.88099378968518,-0.650011776913202,0.770437069321409,-0.341431376862032,
                0.036,-1.86921214369210,-0.640860216836436,0.742690286251606,-0.32168173540129,
                0.037,-1.85658751654318,-0.646388498550073,0.751138468434131,-0.325376088152544,
                0.038,-1.84431134699012,-0.642721258869065,0.735433566998546,-0.313997868391098,
                0.039,-1.83224767201284,-0.651519524484572,0.758836486209615,-0.329643318662125,
                0.04,-1.81944067298549,-0.660317333456702,0.773364433005684,-0.338224668666643,
                0.041,-1.80837052520159,-0.663539565949229,0.782389109895739,-0.345635410847494,
                0.042,-1.79637320285893,-0.668922537921809,0.787468525111366,-0.346901760289389,
                0.043,-1.78537091879337,-0.667244781359096,0.780317320675811,-0.342144391823486,
                0.044,-1.77310828547484,-0.676205856639205,0.794956559918486,-0.350700265574394,
                0.045,-1.76143330145185,-0.684032795656158,0.808492092916648,-0.359100745819333,
                0.046,-1.75040101371970,-0.684205859275737,0.799709128648264,-0.351122587574812,
                0.047,-1.73939149708511,-0.689514318317428,0.806356186595635,-0.353884130375555,
                0.048,-1.72876666643480,-0.692524642844427,0.808679914592297,-0.354736952172377,
                0.049,-1.71847783139789,-0.694195517806187,0.810904205583408,-0.357013223410738,
                0.05,-1.70739601175374,-0.699578566684967,0.814897450374203,-0.358191036928158,
                0.051,-1.69697168471745,-0.705704517444966,0.821899243069315,-0.360548590424637,
                0.052,-1.68793203244067,-0.702290872589112,0.813756037420893,-0.35615337745579,
                0.053,-1.67808060393091,-0.706989295131817,0.82107936974966,-0.360270602145681,
                0.054,-1.66973882446319,-0.699895955083117,0.799563167112435,-0.344877962979795,
                0.055,-1.66035934986418,-0.704578504669982,0.807299079821003,-0.348944705878472,
                0.056,-1.65170758166222,-0.699912696761544,0.793177844008573,-0.33949260242416,
                0.057,-1.64310869046809,-0.697477708500035,0.785086096350434,-0.334335918275321,
                0.058,-1.63430594133281,-0.695576823298844,0.776518820626015,-0.32822053487444,
                0.059,-1.62513387862090,-0.698944944049765,0.778536793755292,-0.328365088126822,
                0.06,-1.61742340716292,-0.693983408551426,0.766152830340093,-0.320444016806954,
                0.061,-1.60865332870493,-0.69443060853404,0.761500323573635,-0.316493883028279,
                0.062,-1.60089862696621,-0.690805487401163,0.753218444105305,-0.312553145136325,
                0.063,-1.59295194617893,-0.693140364516032,0.759972458502561,-0.31769378915512,
                0.064,-1.58491985331364,-0.693158961450471,0.757323331083364,-0.31611181018428,
                0.065,-1.57743816566101,-0.692006642025895,0.755386410835174,-0.315748022827417,
                0.066,-1.56969740087744,-0.689542768304442,0.744976622883144,-0.307995151465740,
                0.067,-1.56240504873278,-0.686916507922895,0.739950054518318,-0.306408534512269,
                0.068,-1.55524207964514,-0.68176418911,0.724737911972444,-0.295977023217512,
                0.069,-1.54871921563106,-0.675112030426528,0.709290917373846,-0.286556109949386,
                0.07,-1.54085960287137,-0.680246075825678,0.718355388350773,-0.291585208331101,
                0.071,-1.53397643152042,-0.678860886444316,0.715474232396738,-0.290582858247215,
                0.072,-1.52698987607084,-0.676800701964851,0.711322146445534,-0.288901232631398,
                0.073,-1.51957384900924,-0.68053395404602,0.717934062154078,-0.293000390418617,
                0.074,-1.51240830274637,-0.682330412579875,0.719666978590843,-0.293520030033418,
                0.075,-1.50483971037852,-0.684795554781526,0.722062600656375,-0.295102230930127,
                0.076,-1.49701978048647,-0.690361561091872,0.729163708640682,-0.298304656277952,
                0.077,-1.49066351423411,-0.686769027698971,0.720416407348508,-0.29307406385134,
                0.078,-1.48387982636395,-0.685599693700695,0.715821094921026,-0.290208308408025,
                0.079,-1.47744115602595,-0.680443204151001,0.699782987826266,-0.279315587405544,
                0.08,-1.47070667976336,-0.68089100921469,0.698431321700671,-0.277896369223315,
                0.081,-1.46287329616116,-0.68929428670315,0.710865367239215,-0.283602936747906,
                0.082,-1.45579497452887,-0.693716538377075,0.717881564186065,-0.287583442835308,
                0.083,-1.44961350765082,-0.691733808336083,0.711213273211999,-0.283119752853766,
                0.084,-1.44293549860920,-0.69531245617472,0.718595305591192,-0.287656816498620,
                0.085,-1.43635238177626,-0.6985450484137,0.725497866742203,-0.291803330202068,
                0.086,-1.43057953845477,-0.693468624882186,0.714239481555036,-0.2859775972748,
                0.087,-1.42422772481892,-0.695544004530945,0.716984245339116,-0.286875648240198,
                0.088,-1.41833462477787,-0.692595099224244,0.707256413808913,-0.28009768512124,
                0.089,-1.41202515058362,-0.694672159008808,0.710828705166703,-0.282232329028163,
                0.09,-1.40604735840312,-0.693032047150005,0.707328600551041,-0.281135878523251,
                0.091,-1.40028768637027,-0.689484666780109,0.697645092031134,-0.275182016760405,
                0.092,-1.39360134572518,-0.694370831541402,0.70432033197873,-0.278342620101090,
                0.093,-1.38819206382686,-0.68812020360869,0.687845577932306,-0.267814942675458,
                0.094,-1.3820820284546,-0.690174049606473,0.690082230782328,-0.268743866791706,
                0.095,-1.37611591901149,-0.692255651473986,0.693493980706124,-0.270832222193293,
                0.096,-1.37024044229996,-0.691195468035462,0.687480822290475,-0.265833893554863,
                0.097,-1.36402086685534,-0.693192876716977,0.687303417950561,-0.264017200016192,
                0.098,-1.35874476387661,-0.686511718096692,0.670270038799258,-0.253907769969933,
                0.099,-1.35301787669186,-0.687487179644698,0.672149881975995,-0.255633912006844,
                0.1,-1.34726027677861,-0.687293254204855,0.669878434663121,-0.253727386960165,
                0.101,-1.34160048285340,-0.689168811108449,0.674326937044898,-0.257264234823528,
                0.102,-1.33590215291942,-0.6870408572375,0.666790154888081,-0.252503941211356,
                0.103,-1.32993403436946,-0.689561259441194,0.67002691917781,-0.254458870261865,
                0.104,-1.32403113427389,-0.691348132016108,0.671603081166358,-0.255443793200092,
                0.105,-1.31909244533226,-0.686452920609137,0.660423748690432,-0.24925086927287,
                0.106,-1.31430891044802,-0.679865684591851,0.643462948108803,-0.238447851262363,
                0.107,-1.30849461985805,-0.684004614245115,0.65144937586101,-0.243293849472116,
                0.108,-1.30281684642588,-0.687369320116084,0.657999041743303,-0.24768157472027,
                0.109,-1.29698547801551,-0.690128040922587,0.660941086802937,-0.249051086297898,
                0.11,-1.29126794802111,-0.692479073057719,0.664657394079222,-0.251634567085291,
                0.111,-1.28570056241620,-0.696116144725791,0.673078556320472,-0.257585208521533,
                0.112,-1.28032973681625,-0.696080252512672,0.670328924348186,-0.255429015837316,
                0.113,-1.27466463026944,-0.701455757579455,0.681689638416862,-0.262911411557842,
                0.114,-1.26944526843882,-0.701054411231674,0.679726381434696,-0.262261036856647,
                0.115,-1.26479683192183,-0.696463569757022,0.667039266054656,-0.253528032373283,
                0.116,-1.25943127739279,-0.69835096331343,0.668740252849784,-0.254205656375793,
                0.117,-1.25446970975003,-0.69615767726162,0.661425644969058,-0.249472770840704,
                0.118,-1.24936757440967,-0.694784078339602,0.656776897520655,-0.246779126357736,
                0.119,-1.24413165189716,-0.694879600115034,0.653707116979714,-0.244197980051309,
                0.12,-1.23950781247934,-0.691683928827855,0.646596451242221,-0.240528483319301,
                0.121,-1.23449837334731,-0.692743091737669,0.648050298359764,-0.241525389606114,
                0.122,-1.22948397786854,-0.693605125693385,0.648234149994129,-0.241440591853231,
                0.123,-1.22462467440040,-0.692374172763416,0.644541765433788,-0.239679322991846,
                0.124,-1.21966157824348,-0.692301593376448,0.642531674876215,-0.238503806046282,
                0.125,-1.21453949385149,-0.691462168850255,0.636537713214751,-0.234049602195841,
                0.126,-1.20964650388567,-0.690530581130233,0.629528256998224,-0.227977923889716,
                0.127,-1.20464360073715,-0.693535735881639,0.635463148487305,-0.231509467095815,
                0.128,-1.19932983432279,-0.697931204835848,0.644946788361881,-0.238080706796537,
                0.129,-1.19437912487441,-0.701194953549522,0.652089454903925,-0.242923130984861,
                0.13,-1.1899767469869,-0.698781305469847,0.646975401093742,-0.240636268905824,
                0.131,-1.18527892532019,-0.6987658716865,0.645812518549681,-0.240203306960861,
                0.132,-1.18051126460100,-0.700774143333634,0.650500627956511,-0.243760158515990,
                0.133,-1.17646377440135,-0.69690323818525,0.641963988803119,-0.238907370356907,
                0.134,-1.17152176877103,-0.698000693886714,0.644042956877378,-0.241026076764002,
                0.135,-1.16679298094116,-0.69958763867163,0.647693271278718,-0.244227672115202,
                0.136,-1.16233020400786,-0.697552094102286,0.641934997121361,-0.241076590041387,
                0.137,-1.15795545513724,-0.696625734046911,0.639006926943816,-0.2394807672935,
                0.138,-1.15397096416625,-0.693425094344926,0.633460848622169,-0.237154610602338,
                0.139,-1.14970656326767,-0.690756025872733,0.625441144625466,-0.231816604502032,
                0.14,-1.14475034651465,-0.69656369992186,0.637552579549171,-0.239591028315861,
                0.141,-1.14030734138645,-0.697143995434434,0.638881786133096,-0.241279381946218,
                0.142,-1.13550016441839,-0.700865467101936,0.645361467643489,-0.245259106620807,
                0.143,-1.13088403131165,-0.705648437640328,0.656761880021199,-0.252802197030264,
                0.144,-1.12623601119296,-0.707441631275026,0.659746052874152,-0.254895540500745,
                0.145,-1.12184236187616,-0.71011176943086,0.66668055001763,-0.260155664310981,
                0.146,-1.11721275112643,-0.714496209386036,0.677449855864181,-0.267743804937415,
                0.147,-1.1131252124693,-0.712408726415143,0.673014218912338,-0.265444799335211,
                0.148,-1.10839931672776,-0.714353602402442,0.674444916928192,-0.265734023743048,
                0.149,-1.10439350451013,-0.712462730977193,0.670713582465459,-0.263997437962084,
                0.15,-1.10032869095594,-0.708813764965392,0.661315948234294,-0.258387068738271,
                0.151,-1.09615266378492,-0.71040845016073,0.664960265354423,-0.260883757041414,
                0.152,-1.09168809664995,-0.71135863874918,0.664070136718552,-0.259422551022039,
                0.153,-1.08762544425951,-0.711428439574245,0.664322482530452,-0.259831689228086,
                0.154,-1.08382098754306,-0.710560004987618,0.663514667241561,-0.259977856698358,
                0.155,-1.07977955300825,-0.709583341698485,0.660707335823658,-0.258543101433030,
                0.156,-1.07574929998615,-0.708500467408483,0.657336154483675,-0.256474987793019,
                0.157,-1.07124388679481,-0.712409626585963,0.664697412377406,-0.260825228492293,
                0.158,-1.06674164733433,-0.715517731832902,0.670681297313143,-0.264811717058248,
                0.159,-1.06267343052180,-0.71546263979266,0.67099418696128,-0.266030810043008,
                0.16,-1.05911645134544,-0.71265255091706,0.666683674588171,-0.264649019183141,
                0.161,-1.05434116717185,-0.717954687646024,0.674944656405367,-0.269236980702785,
                0.162,-1.05018914632072,-0.717902373038124,0.673719783472799,-0.268502289926359,
                0.163,-1.04679446216761,-0.714987209315228,0.669409774228945,-0.266994084748312,
                0.164,-1.04292722769925,-0.712094417442297,0.661847419832494,-0.262614861473576,
                0.165,-1.03831800915318,-0.715510109840537,0.666056382088869,-0.264471135839247,
                0.166,-1.03431178844428,-0.715505433890848,0.665454236503492,-0.264273621970240,
                0.167,-1.03001466245340,-0.718081792336242,0.66960871785686,-0.266772606131655,
                0.168,-1.02633990290687,-0.715228087688734,0.661533495282581,-0.261779556069416,
                0.169,-1.02247805460348,-0.71401159539954,0.659253812997168,-0.261313754624165,
                0.17,-1.01891343669749,-0.712733688370868,0.657133347496958,-0.260751846185375,
                0.171,-1.01501972224951,-0.712165971137129,0.655558562421135,-0.260315520126396,
                0.172,-1.01099106808275,-0.712732752883769,0.655427285570756,-0.260347314925299,
                0.173,-1.00638820389686,-0.718513558439333,0.666447597621416,-0.266898136449826,
                0.174,-1.00268705593069,-0.716591225315012,0.661166960748748,-0.263782898560564,
                0.175,-0.9987484027045,-0.716443133356666,0.659234332731226,-0.262586601296916,
                0.176,-0.994804364948362,-0.717217221031695,0.661076978317722,-0.264595197065131,
                0.177,-0.991107811124103,-0.717443103407169,0.661561697901186,-0.265337267922027,
                0.178,-0.987037846754548,-0.719842255494553,0.66660643393846,-0.268903825699072,
                0.179,-0.983454059873574,-0.717906829138486,0.662320696591198,-0.266907953316457,
                0.18,-0.979279967883376,-0.719518550414032,0.663528052673768,-0.267464829956517,
                0.181,-0.975597669050323,-0.719814291974505,0.664230441415865,-0.268464936488295,
                0.182,-0.97195950214211,-0.718167894625875,0.660578571572603,-0.266859612260308,
                0.183,-0.968218457568137,-0.716483660186148,0.655632161820721,-0.264119036871162,
                0.184,-0.96405703899248,-0.719312406436343,0.660548587877775,-0.267170739979494,
                0.185,-0.96015783958641,-0.720426572457771,0.662391329750306,-0.268750356931834,
                0.186,-0.956347198655138,-0.721012960881514,0.662937870513323,-0.26937012449488,
                0.187,-0.952819095801296,-0.71971770420826,0.66017244244926,-0.268300838329034,
                0.188,-0.949553603873738,-0.7168640922491,0.654045655423963,-0.265059841308233,
                0.189,-0.94575222101992,-0.716440955885035,0.650320170274425,-0.262128986248885,
                0.19,-0.942204071129666,-0.715861178314323,0.648508030159051,-0.261259593678901,
                0.191,-0.938249445450578,-0.716029587134123,0.646270679355591,-0.259391592499819,
                0.192,-0.934713347220651,-0.716796286429376,0.648708468230844,-0.261695933455635,
                0.193,-0.931040211571744,-0.716218916884996,0.646447643244895,-0.260553887984131,
                0.194,-0.927351627149162,-0.717098868841454,0.649767260611698,-0.263922480898992,
                0.195,-0.92352364961888,-0.71828385394663,0.649509716089459,-0.262964086108105,
                0.196,-0.919878430484254,-0.719523934408437,0.651714282045127,-0.264495239511152,
                0.197,-0.916608415786822,-0.717874646630221,0.649289406883597,-0.264025481195710,
                0.198,-0.913031633669954,-0.716358233263663,0.644981679542842,-0.261860994800723,
                0.199,-0.90909650661664,-0.718792437689125,0.649116864411208,-0.264516576769846,
                0.2,-0.905465283370477,-0.719584651069423,0.649323935954166,-0.264162385679375,
                0.201,-0.902113994374575,-0.717691099988259,0.64353798654519,-0.260233863794771,
                0.202,-0.898513265832284,-0.719390585030536,0.648233122323046,-0.264117984283069,
                0.203,-0.89497191259134,-0.719189349927881,0.647029325465183,-0.263448501397463,
                0.204,-0.891771993402861,-0.716907763224764,0.642443637578803,-0.261347126934609,
                0.205,-0.887975795267613,-0.71818868624694,0.643869229476355,-0.262178552047458,
                0.206,-0.88446739129314,-0.719035144392404,0.645919132370541,-0.263868612418162,
                0.207,-0.880997554729206,-0.71852867610969,0.644602776604515,-0.263586905039829,
                0.208,-0.877377073496093,-0.720004263606213,0.646589483467288,-0.264525557729237,
                0.209,-0.874000253974485,-0.719262230193845,0.643566497677643,-0.262512647689033,
                0.21,-0.870354204778361,-0.721089239974878,0.645999177999923,-0.263904295146848,
                0.211,-0.866591318250154,-0.722573090408807,0.647915779586484,-0.265182686616421,
                0.212,-0.863325531784465,-0.721966950810548,0.64730255507822,-0.265521516050566,
                0.213,-0.85976441005025,-0.723499819085304,0.650656533587089,-0.267946230018137,
                0.214,-0.856232335767452,-0.724929312557185,0.65339285745604,-0.269769802598852,
                0.215,-0.852781366453646,-0.725070655064429,0.652821344326086,-0.269626855932121,
                0.216,-0.849160947652958,-0.727099395433399,0.657693328721515,-0.273371061905679,
                0.217,-0.845536168115848,-0.729686443243489,0.6642076215936,-0.278245410166296,
                0.218,-0.842170176697566,-0.728096047993167,0.65791410921668,-0.273826500001761,
                0.219,-0.83862430642241,-0.729344213851909,0.659581277587582,-0.274994206075022,
                0.22,-0.835309985211694,-0.728228954286084,0.65537654325368,-0.27206177221325,
                0.221,-0.831892289339456,-0.727089038179277,0.650857483456717,-0.268979306449814,
                0.222,-0.828747727265808,-0.72741199113545,0.65337616432334,-0.271658967742492,
                0.223,-0.825414871967963,-0.72820784751315,0.655392938176973,-0.273265122781694,
                0.224,-0.822387559842097,-0.726432421165615,0.652466574811228,-0.272380970835240,
                0.225,-0.818869212889315,-0.726525800366492,0.650253775473755,-0.270590284107628,
                0.226,-0.81535872107079,-0.726554120185105,0.647395562480517,-0.268258291316215,
                0.227,-0.811947252499385,-0.727555097340081,0.648236832776775,-0.268540826459212,
                0.228,-0.808291317159007,-0.730030921216723,0.651285673340323,-0.270027336920853,
                0.229,-0.804525827212081,-0.733366303875775,0.655801721214926,-0.27182928067321,
                0.23,-0.801210516680226,-0.732635493734568,0.65317044287307,-0.270414341530566,
                0.231,-0.797794237408256,-0.733568142399102,0.653512607983987,-0.270239499596602,
                0.232,-0.794085999119134,-0.73726180277926,0.661359125736732,-0.275535912725540,
                0.233,-0.791058282701733,-0.735998203802001,0.659310255026545,-0.275135938927077,
                0.234,-0.78815135598241,-0.733355382336396,0.653907005556664,-0.272428934893461,
                0.235,-0.78513874009887,-0.732509428606926,0.652416130550208,-0.272065568993883,
                0.236,-0.781579714907424,-0.73400046319334,0.65433321710321,-0.273383632680161,
                0.237,-0.778625583509909,-0.732717763597304,0.652360885020862,-0.273002844038815,
                0.238,-0.775635749877212,-0.731318114718912,0.6496478666666,-0.271909693280658,
                0.239,-0.772466827394193,-0.730718585267771,0.647826346301597,-0.271240229109714,
                0.24,-0.768947135705159,-0.73212740806215,0.64901465061326,-0.271628536478100,
                0.241,-0.76550438042495,-0.734258958180007,0.653070095553579,-0.274257146033597,
                0.242,-0.762202584324644,-0.735041522542238,0.65491356831654,-0.276146559007355,
                0.243,-0.759023725295879,-0.736020015295077,0.656822969645878,-0.277428932932089,
                0.244,-0.756040197901657,-0.734414605760507,0.652375453496328,-0.274542219539501,
                0.245,-0.75270500119361,-0.736517907377727,0.657164093393839,-0.278083195670134,
                0.246,-0.749437016618836,-0.737572089337525,0.659004609179549,-0.279409060097226,
                0.247,-0.746269573252262,-0.737365087683065,0.657029582160631,-0.278106591248283,
                0.248,-0.743383560451207,-0.736311066736958,0.653962614911553,-0.275959251934991,
                0.249,-0.739942226135677,-0.737625770768496,0.655558170942549,-0.276842326309819,
                0.25,-0.73662774808901,-0.7392292048368,0.658400619379209,-0.278772832562891,
                0.251,-0.733665879677878,-0.736615978835184,0.651592889289883,-0.274805115557458,
                0.252,-0.730616579635594,-0.736730138272452,0.65211630120337,-0.275555048133204,
                0.253,-0.727268401676132,-0.739730634872995,0.658551905790497,-0.279808224143169,
                0.254,-0.724219937708071,-0.738630501424826,0.65652937278191,-0.279573503800958,
                0.255,-0.721055112972527,-0.740173229589328,0.66064131285433,-0.282886342345363,
                0.256,-0.717686688037854,-0.74235360318683,0.664691030465811,-0.285361623599184,
                0.257,-0.714334374331371,-0.744054318999748,0.668821037808881,-0.288685671735251,
                0.258,-0.711553541049443,-0.741306942846244,0.662279007107962,-0.28496182374158,
                0.259,-0.708429989089992,-0.741508377099955,0.662399721480958,-0.285379567812154,
                0.26,-0.705148233980115,-0.743100279060811,0.664372753648538,-0.286423452459823,
                0.261,-0.702167300208514,-0.742965807696065,0.664388193379175,-0.287030914966768,
                0.262,-0.699130894098574,-0.743225871785416,0.66555470642903,-0.28849070408577,
                0.263,-0.696086029770722,-0.743504453580237,0.665283544047494,-0.288381231433062,
                0.264,-0.692868918922173,-0.74395066057769,0.664259537001817,-0.287379662115671,
                0.265,-0.689657815047259,-0.744500671799126,0.664003469916042,-0.287114903682925,
                0.266,-0.686785364882314,-0.744854024855485,0.666486040443095,-0.289715555856997,
                0.267,-0.684100455741809,-0.743646984700069,0.664970251353896,-0.289444105891193,
                0.268,-0.68089634735714,-0.743267543022592,0.661715160688289,-0.286969248499862,
                0.269,-0.677638235633283,-0.744465530161207,0.663225870619316,-0.287820139339598,
                0.27,-0.674342463653276,-0.745405865503773,0.663126994515734,-0.287544225409857,
                0.271,-0.671419873163092,-0.74427976139916,0.660523222654812,-0.286289419460184,
                0.272,-0.668186728518284,-0.747029228823251,0.667135334715692,-0.29077666781779,
                0.273,-0.665090538233586,-0.748708594075932,0.671351628284673,-0.293988514903152,
                0.274,-0.662069947051248,-0.748618249879034,0.670709272904697,-0.293744510869478,
                0.275,-0.659192582547434,-0.748163890142265,0.669264042469462,-0.292840374966193,
                0.276,-0.656128559685475,-0.747129900516919,0.66665761001686,-0.291875722985466,
                0.277,-0.653092374377309,-0.747219707438786,0.666346983725457,-0.292091847970068,
                0.278,-0.64979234413374,-0.750025884807616,0.671459143063848,-0.295285901359861,
                0.279,-0.647313706641674,-0.746802336046683,0.664800514620604,-0.29157014860588,
                0.28,-0.644781288335062,-0.744352328328856,0.660724489357457,-0.290071416183665,
                0.281,-0.641452693314923,-0.74536028427145,0.660859270567485,-0.289886467352363,
                0.282,-0.638572958168043,-0.744848470080732,0.659417089393874,-0.289407208718059,
                0.283,-0.635745413805563,-0.74500056308554,0.660791168599189,-0.290890415613264,
                0.284,-0.632841734124677,-0.745075519587258,0.661608274830192,-0.292075068885866,
                0.285,-0.629800770956724,-0.746695165610035,0.665152431004934,-0.294654818917849,
                0.286,-0.626966524946696,-0.747007252808583,0.666207602566569,-0.295706940245498,
                0.287,-0.62416002113435,-0.745326269065282,0.663582192107476,-0.295146914543339,
                0.288,-0.62138786966217,-0.745735210003115,0.66454465186637,-0.295783183176051,
                0.289,-0.618440324536977,-0.745457454181994,0.662965504270745,-0.295015960035062,
                0.29,-0.615657759125751,-0.744377464205327,0.66188075585437,-0.295667261958772,
                0.291,-0.612634439920778,-0.745734972510095,0.665278232715648,-0.298447929283119,
                0.292,-0.610038325858797,-0.743311418261686,0.660943720327737,-0.296487542910185,
                0.293,-0.607279296306642,-0.74155205623833,0.656954236448242,-0.294597340114891,
                0.294,-0.604798928429961,-0.738000606228658,0.649480463140362,-0.290560483225774,
                0.295,-0.602003140530401,-0.736976564409279,0.64577856595468,-0.287820135820158,
                0.296,-0.59949844921562,-0.733387918333577,0.638455435303882,-0.284196822852286,
                0.297,-0.596601503672992,-0.733702110943427,0.639023089386596,-0.284971105178472,
                0.298,-0.593788265744095,-0.732814122272035,0.636560481979746,-0.283563007073619,
                0.299,-0.590915101590105,-0.73142411675257,0.631834555746077,-0.280593585018652,
                0.3,-0.587874250925849,-0.733695886471182,0.637595352889314,-0.284820628419446,
                0.301,-0.584800263413732,-0.735396294715564,0.640868530079784,-0.287086003725590,
                0.302,-0.58227866245172,-0.732684821422664,0.635995911458636,-0.285203277815352,
                0.303,-0.579655937238973,-0.731031040183345,0.631686014241716,-0.282395344282410,
                0.304,-0.577020398542159,-0.729183643620246,0.627192867629944,-0.279941478097976,
                0.305,-0.574053629475804,-0.73089377674916,0.631123238437316,-0.282750772955420,
                0.306,-0.5712486691499,-0.729570731122547,0.627392798119748,-0.280567985082518,
                0.307,-0.568365050532405,-0.729822307661047,0.62764509231902,-0.281006324914291,
                0.308,-0.565471133365036,-0.728410826922015,0.622499701274386,-0.277664165874124,
                0.309,-0.562400704775021,-0.730201475520337,0.625525227728,-0.279623042112498,
                0.31,-0.559451711762342,-0.731574724072757,0.628891396538349,-0.282313401358734,
                0.311,-0.55658983155885,-0.730368118742526,0.624847283374377,-0.27975310012379,
                0.312,-0.553826452803687,-0.73054309855375,0.625647642840064,-0.280714124704486,
                0.313,-0.551038025291522,-0.730828585945021,0.62719641123408,-0.282374283371808,
                0.314,-0.548034263591530,-0.733594219752167,0.633725319970022,-0.286870565743043,
                0.315,-0.545371802047242,-0.733214950118335,0.633380623064326,-0.28715996383006,
                0.316,-0.542564422176324,-0.733793941894321,0.635401296780696,-0.289225035023732,
                0.317,-0.539719545803308,-0.734751724098951,0.637718773839483,-0.290944957096658,
                0.318,-0.537167807933137,-0.73345997462352,0.635657815942027,-0.290398748119976,
                0.319,-0.534507213223638,-0.731957333268727,0.632489134561379,-0.288957385914626,
                0.32,-0.53164910617811,-0.732137690941215,0.631560189681394,-0.287934557123181,
                0.321,-0.528306210507756,-0.735708256634032,0.637918725462325,-0.291651789841560,
                0.322,-0.52543532628976,-0.73635648562229,0.639098452900849,-0.292777971335153,
                0.323,-0.522792983594556,-0.734627592228087,0.635264812053426,-0.290931379536664,
                0.324,-0.520072412168177,-0.73486868815803,0.63687669368637,-0.292788673089882,
                0.325,-0.517238064074140,-0.734631812053078,0.635885086825694,-0.292535509158517,
                0.326,-0.514266025783882,-0.737073882566566,0.641184631919892,-0.296065707932928,
                0.327,-0.511667515528486,-0.735631773178825,0.638624640388066,-0.295308871180294,
                0.328,-0.508939683197206,-0.735210476379278,0.63588449359046,-0.293067582042798,
                0.329,-0.505881728614632,-0.738092997557805,0.64273689315965,-0.297961466399564,
                0.33,-0.503465098827192,-0.73492635871822,0.63546659615291,-0.293783229955588,
                0.331,-0.500771130178997,-0.734737259545373,0.635366397630412,-0.294289288955727,
                0.332,-0.498156035295792,-0.733523874042811,0.631312032252962,-0.291565104665561,
                0.333,-0.49568312600932,-0.731784172629986,0.627954002474498,-0.290022418137518,
                0.334,-0.492730456553159,-0.733541403768609,0.632188416410475,-0.293395438497652,
                0.335,-0.490240718130781,-0.730365168994832,0.626078585200606,-0.290544209379043,
                0.336,-0.487372794215637,-0.731417582992375,0.628897866473468,-0.292963871891438,
                0.337,-0.484545311710864,-0.733000009074073,0.633777557095325,-0.296999988596012,
                0.338,-0.482014033424195,-0.731964179861161,0.632193507745115,-0.296716543209402,
                0.339,-0.479007444295279,-0.733832421942922,0.636278021008658,-0.299630924080472,
                0.34,-0.47601799254294,-0.736239550189875,0.640981876605394,-0.302460117692982,
                0.341,-0.473114147525341,-0.7380590757681,0.64478638167324,-0.305241071306307,
                0.342,-0.47039596741623,-0.738647654244625,0.646024379334366,-0.306282811298526,
                0.343,-0.467782688641449,-0.737297512093902,0.641800690082073,-0.303556178094393,
                0.344,-0.465043934216697,-0.736857580519552,0.640185494205022,-0.302785755632104,
                0.345,-0.462161017077727,-0.737502601602104,0.640816733413503,-0.303220205121155,
                0.346,-0.459440216183111,-0.73788450493845,0.642052579895075,-0.304564602857753,
                0.347,-0.45646525734955,-0.739746141131468,0.64527191049913,-0.306577960474082,
                0.348,-0.453734176841092,-0.740064194759861,0.645744728996217,-0.307131282756114,
                0.349,-0.451093829672605,-0.739639417758782,0.645081953530923,-0.307180992565547,
                0.35,-0.448398165029764,-0.738733247830245,0.642115489556725,-0.305481280936529,
                0.351,-0.445718987100642,-0.738372353697938,0.641168782750778,-0.30532315810533,
                0.352,-0.443243572946114,-0.737491918331444,0.639915104343164,-0.305065817561735,
                0.353,-0.440691159583265,-0.736693839731719,0.638489772392785,-0.304568246535805,
                0.354,-0.438219783346190,-0.735171993590354,0.63532463675714,-0.303116652873234,
                0.355,-0.435411484688226,-0.735804757778866,0.635962986516906,-0.303707831000524,
                0.356,-0.432851031671776,-0.734889065052927,0.634623602808736,-0.303480482643063,
                0.357,-0.430084240517356,-0.736929380940914,0.640859454814518,-0.308401178542768,
                0.358,-0.427308965787407,-0.737515319414605,0.641319320645071,-0.308747706311776,
                0.359,-0.424459605995515,-0.738394060125526,0.643282946705024,-0.310359508947615,
                0.36,-0.42180659560896,-0.738791838363535,0.644687580403218,-0.311694073449657,
                0.361,-0.419153234434065,-0.738948214518276,0.645655767348654,-0.312843475576786,
                0.362,-0.416621872616715,-0.738425197724953,0.644269097859004,-0.312288088328339,
                0.363,-0.413963673595602,-0.737907266562384,0.642706574371659,-0.311611729516931,
                0.364,-0.410954621299191,-0.740437456579201,0.646851912915294,-0.313940280627062,
                0.365,-0.408114190198838,-0.740182185099101,0.644725629585392,-0.312583857043545,
                0.366,-0.405737577344018,-0.73848900812003,0.64146368251597,-0.310929324802964,
                0.367,-0.403242171958243,-0.737224795492137,0.638453407548018,-0.309215695143002,
                0.368,-0.40055007081059,-0.737082859722917,0.636819069546577,-0.307963266347703,
                0.369,-0.397831538907735,-0.73758717545797,0.6369277244188,-0.308040055565687,
                0.37,-0.394691798911111,-0.74262235652839,0.648795559789745,-0.315987430209485,
                0.371,-0.392352179620518,-0.739820679081949,0.643013114106021,-0.312936955985439,
                0.372,-0.389618700327532,-0.740107717753532,0.643661752793783,-0.313589729405634,
                0.373,-0.386823935043248,-0.740386111116494,0.643249139604098,-0.313285921948629,
                0.374,-0.384198552933438,-0.74005752752065,0.643085556065461,-0.313918490404107,
                0.375,-0.381396705409120,-0.740959328882658,0.64464140048559,-0.315089842455199,
                0.376,-0.378861352442054,-0.739897247042664,0.642581672695456,-0.314275455344896,
                0.377,-0.376069430982065,-0.73950280238298,0.640446824134838,-0.312944830734099,
                0.378,-0.373472569577234,-0.739413129583264,0.639882653715747,-0.312677278164688,
                0.379,-0.370617720675447,-0.741379798223079,0.644496996448618,-0.316030449347335,
                0.38,-0.368066997267278,-0.74044321299529,0.64228290353388,-0.314795201802829,
                0.381,-0.36528371777273,-0.742112166250635,0.646189900586699,-0.317629003003592,
                0.382,-0.36282520493318,-0.741014839712214,0.644023882430029,-0.316829415591662,
                0.383,-0.360000800471211,-0.742316044874669,0.645373899303962,-0.317415058991351,
                0.384,-0.357496329404132,-0.742070012577656,0.645423513742372,-0.317821900110975,
                0.385,-0.35474572851012,-0.741592203267024,0.64305057223293,-0.316154406999253,
                0.386,-0.351915270598098,-0.743001011902321,0.64529640416012,-0.317488326240853,
                0.387,-0.349347411460549,-0.743136299073841,0.645258730376952,-0.317505794497942,
                0.388,-0.346807345809555,-0.743134867134622,0.645726893204579,-0.318093554074266,
                0.389,-0.344037671159504,-0.744149394535167,0.647710603576464,-0.319567743905746,
                0.39,-0.341016054466528,-0.747912527767963,0.655276184683517,-0.324095944209243,
                0.391,-0.338202332694907,-0.74889790578618,0.657019895324098,-0.325403930608604,
                0.392,-0.335606033732142,-0.750006129814578,0.660338897411824,-0.327911316783873,
                0.393,-0.333091826344582,-0.74853725603903,0.656873413755504,-0.326154302380285,
                0.394,-0.330383381119508,-0.75048142350283,0.662104174409031,-0.330060889625024,
                0.395,-0.32749221848301,-0.751465079802963,0.662434642025601,-0.330039123161368,
                0.396,-0.325252744445319,-0.749849878058328,0.659515276365664,-0.328605407437462,
                0.397,-0.322488884366654,-0.750663388700953,0.661453859652892,-0.330254945191601,
                0.398,-0.319950149833782,-0.74938099952773,0.658867964392428,-0.329164462448290,
                0.399,-0.317444556404172,-0.748159904764032,0.655690196483882,-0.327373022202489,
                0.4,-0.314780987526045,-0.748900111972405,0.656635263324612,-0.328009502155245,
                0.401,-0.312285768495929,-0.74845868892958,0.656287753054015,-0.328433408723547,
                0.402,-0.309761143340052,-0.748288904584013,0.655861442383043,-0.328189383252827,
                0.403,-0.307228783066637,-0.748101083345653,0.654868887965727,-0.327439371047289,
                0.404,-0.304744341146399,-0.747793865083532,0.653959656424331,-0.326753038426869,
                0.405,-0.302271759931848,-0.748705456629922,0.657574380391322,-0.32962058743137,
                0.406,-0.299504559164679,-0.750111723426512,0.660180706256911,-0.331257273825719,
                0.407,-0.297072277503429,-0.749254360048238,0.658678428423462,-0.330734880875021,
                0.408,-0.294627623487431,-0.747245028194713,0.653517189811673,-0.327705708211411,
                0.409,-0.292089332670925,-0.74786159595233,0.656463606037032,-0.330472708934634,
                0.41,-0.289309237044986,-0.749185055089724,0.65905358228672,-0.332211076897248,
                0.411,-0.286853802030015,-0.747043416585263,0.653248011308811,-0.328758093010871,
                0.412,-0.284446164751519,-0.747061017272243,0.653383429450113,-0.328943446924752,
                0.413,-0.281921172735991,-0.746621838308271,0.652434140432207,-0.328541808201812,
                0.414,-0.279350545471513,-0.747481005064466,0.654889948881929,-0.330318568401147,
                0.415,-0.276896568760389,-0.747087281345584,0.654356186946466,-0.330279777307068,
                0.416,-0.274032623594511,-0.74962514455818,0.660004179178867,-0.334118374270983,
                0.417,-0.271499659228796,-0.749186796062386,0.65879385986565,-0.333528529831259,
                0.418,-0.269290682090761,-0.746815923811641,0.65434154819465,-0.331215978175444,
                0.419,-0.266506208309506,-0.748116159304454,0.656160800446487,-0.332390546796508,
                0.42,-0.263814202246164,-0.75021505833676,0.66171789051294,-0.336394939714533,
                0.421,-0.261198398978524,-0.751150551535518,0.664076747122621,-0.33821061914239,
                0.422,-0.258796848093907,-0.750240271224696,0.663040941095869,-0.338156455667335,
                0.423,-0.256103060193388,-0.75162008874766,0.666178831592955,-0.340348981555861,
                0.424,-0.253753860918096,-0.750103218214792,0.66306779542911,-0.338863990094373,
                0.425,-0.250882288780575,-0.752457807168658,0.667085168576177,-0.341105794594323,
                0.426,-0.248324963974510,-0.752985567356266,0.667903408353478,-0.341607778733850,
                0.427,-0.245449504998314,-0.755315658887692,0.672980510154096,-0.345084282096789,
                0.428,-0.243009107835381,-0.755284677713257,0.674768803175015,-0.347324241165257,
                0.429,-0.240514427114058,-0.753909625705364,0.670974501072376,-0.344955979739818,
                0.43,-0.237948744685361,-0.754069477881186,0.671199581480749,-0.345315587959255,
                0.431,-0.235295988076049,-0.755478976266173,0.67402588667169,-0.346930252509965,
                0.432,-0.23260961778482,-0.757095533579957,0.67772430423078,-0.349396571895714,
                0.433,-0.230258307535088,-0.755237692208075,0.673600601788007,-0.347017639621939,
                0.434,-0.227329962545314,-0.757516296049586,0.67832985527115,-0.350237306928297,
                0.435,-0.224733507413370,-0.758111700950603,0.680367068829188,-0.352014718156142,
                0.436,-0.222131354985862,-0.758298623275434,0.679500188716082,-0.351179824331296,
                0.437,-0.219915627342166,-0.755776014533938,0.674111454666427,-0.348257035082297,
                0.438,-0.217260384752825,-0.757969649296555,0.679267633004551,-0.351833713728116,
                0.439,-0.214664577105500,-0.75959580753124,0.68273840222076,-0.353841415343484,
                0.44,-0.212053180505112,-0.761095579488268,0.687402497005162,-0.357658937965862,
                0.441,-0.209152227455473,-0.76342856012113,0.69145674921059,-0.360100347896554,
                0.442,-0.206798181504287,-0.761762456795877,0.68765197146767,-0.357924414974614,
                0.443,-0.20401047958756,-0.763238111072335,0.689677505858469,-0.358801035006073,
                0.444,-0.201419787249387,-0.762221047747412,0.686229715617583,-0.356584707435972,
                0.445,-0.198953006874878,-0.762583413215683,0.687750833151475,-0.357881673642186,
                0.446,-0.196729864768299,-0.761209771616515,0.685997233351787,-0.357518554106568,
                0.447,-0.194159651336476,-0.762099318458377,0.686984363874198,-0.357630534954336,
                0.448,-0.191752869487833,-0.761949120224885,0.687384864041358,-0.358256934938571,
                0.449,-0.189434940698338,-0.760971504706397,0.686105181126927,-0.358118988841455,
                0.45,-0.186913704766938,-0.760956001962027,0.686315610244032,-0.358750455647214,
                0.451,-0.184245353650881,-0.761546878833526,0.68660477854561,-0.358706008035927,
                0.452,-0.18180451815578,-0.761199547402679,0.686242837586035,-0.358782987284634,
                0.453,-0.179180527050990,-0.762375758781822,0.688750584370187,-0.360438536785494,
                0.454,-0.176649794270360,-0.762617797592195,0.68877986081167,-0.36024922095028,
                0.455,-0.174156167945543,-0.762293759409134,0.687687541716986,-0.359385090971687,
                0.456,-0.171346561289028,-0.763802554732766,0.689711444012935,-0.360323492984304,
                0.457,-0.168896605382159,-0.763764357468654,0.689975681572114,-0.360794125189103,
                0.458,-0.166525514291145,-0.762388878722444,0.686546863584798,-0.358669340974469,
                0.459,-0.163961855267516,-0.763218326875517,0.688766080423851,-0.360461856433012,
                0.46,-0.161292154424875,-0.764301149535849,0.690104223927963,-0.360839840470046,
                0.461,-0.158677607714936,-0.764571808042914,0.690275841172113,-0.361057021756215,
                0.462,-0.156268637275785,-0.764072688051971,0.689475987043609,-0.360843325938911,
                0.463,-0.153841372081657,-0.762829737665531,0.686391909634769,-0.358851558378223,
                0.464,-0.151128559851283,-0.764476099302167,0.690245642602147,-0.361402760597904,
                0.465,-0.148662757649783,-0.765275619509193,0.692802568695626,-0.363377143176658,
                0.466,-0.146189535005576,-0.765909787286659,0.693940403543909,-0.36382732874054,
                0.467,-0.143863051277392,-0.764041719072184,0.689690821804193,-0.361277927549161,
                0.468,-0.141044379630541,-0.766992606743428,0.696618043491467,-0.365977955665969,
                0.469,-0.138716322168681,-0.76498859433689,0.69119287094488,-0.362368215658613,
                0.47,-0.136259371772125,-0.764269889909766,0.688322247465588,-0.360129083958951,
                0.471,-0.133714592736247,-0.765395060565679,0.691131725336845,-0.361992851357745,
                0.472,-0.131186319245906,-0.765374586468957,0.691347554045654,-0.362620280120901,
                0.473,-0.128849258954083,-0.763683920890398,0.688282628938593,-0.361238045875226,
                0.474,-0.126085605690072,-0.765171338822082,0.691064420776711,-0.36297903905876,
                0.475,-0.123296009598837,-0.767796027740437,0.69689722211898,-0.366633845603809,
                0.476,-0.120842580583883,-0.767037379545644,0.695582415713501,-0.366289896625505,
                0.477,-0.118592969520313,-0.765982226584845,0.69416643083236,-0.365664830990727,
                0.478,-0.116091312121572,-0.765693218561297,0.692999532961766,-0.364835638710961,
                0.479,-0.113751127745156,-0.765053102084496,0.692084524208922,-0.364611028089502,
                0.48,-0.111348815519485,-0.76418514991494,0.689206083624069,-0.362344344282667,
                0.481,-0.108951150300805,-0.76381986015357,0.687760359071612,-0.361110216926798,
                0.482,-0.106446125518881,-0.76311959218672,0.6853480128368,-0.359429750331579,
                0.483,-0.104028887866516,-0.762984650995223,0.685028276830423,-0.359371967120799,
                0.484,-0.101638452709361,-0.761804532266534,0.682239074680678,-0.357616431341042,
                0.485,-0.09919363711973,-0.761596913380115,0.680986335073157,-0.356516479442116,
                0.486,-0.096668261878896,-0.761871590239202,0.681935354005853,-0.357423624717799,
                0.487,-0.0941305659118464,-0.762755624509271,0.683913900572415,-0.358853897194053,
                0.488,-0.09169383892148,-0.762242344744556,0.682652261645829,-0.358139948633554,
                0.489,-0.0894766096402948,-0.760597011116938,0.680356817852367,-0.357380335341472,
                0.49,-0.0867954641139026,-0.762422050305635,0.685126205017351,-0.360758300620371,
                0.491,-0.084252681592998,-0.762461703889424,0.684864688897041,-0.360615496450175,
                0.492,-0.0819449812180741,-0.760818740316685,0.681426795979655,-0.358663053724184,
                0.493,-0.0792269992295438,-0.76273394538676,0.685081200407892,-0.360724514458271,
                0.494,-0.0767705863491897,-0.763259029025366,0.687112044672046,-0.362286249844161,
                0.495,-0.0740106073231419,-0.764567791518297,0.689322584536675,-0.36352415129123,
                0.496,-0.0711196247914386,-0.767339867850007,0.694079763112904,-0.366071486777498,
                0.497,-0.0685592662063852,-0.767359210061809,0.693769036423166,-0.365793040190799,
                0.498,-0.0660282359636956,-0.76684556887519,0.692595180489131,-0.365143012330243,
                0.499,-0.0632457814255899,-0.768867435690126,0.696200637555867,-0.367086086618601,
                0.5,-0.06071462404235,-0.769567285344051,0.697984282046033,-0.368266747785921,
                0.501,-0.0584625245886047,-0.768167175686005,0.695730029190095,-0.367295570114802,
                0.502,-0.0560195597293309,-0.767347032515376,0.693758833835521,-0.366351328384224,
                0.503,-0.0535198661944481,-0.768746220896076,0.698359384452157,-0.369797163506520,
                0.504,-0.0509233297334401,-0.768251023629614,0.696558830483387,-0.368751377134992,
                0.505,-0.0484849754263263,-0.767346519844718,0.694048868330273,-0.367087899931803,
                0.506,-0.0460515403597572,-0.766401719202206,0.691609262638567,-0.365555642415334,
                0.507,-0.0435980429222659,-0.76680446555712,0.69277192091794,-0.366481121911645,
                0.508,-0.0411716548227468,-0.766294181402916,0.691989938699306,-0.366169426138649,
                0.509,-0.0388949622116227,-0.76312655024586,0.684921913546442,-0.362202794020428,
                0.51,-0.0364566327039202,-0.763157454771665,0.685151296877282,-0.362231459188558,
                0.511,-0.0337660528464135,-0.765502111793083,0.689982556020862,-0.364917294926163,
                0.512,-0.0312967566683865,-0.76626437154922,0.693301874270321,-0.367642310561265,
                0.513,-0.0286816302090407,-0.767114820972031,0.69460100633993,-0.368171887954779,
                0.514,-0.0260830107926428,-0.768379561335787,0.697214911260871,-0.369548992132372,
                0.515,-0.023600168027996,-0.76719485981106,0.693086805496291,-0.366302336045539,
                0.516,-0.0212574755435514,-0.765168094558045,0.688054887080569,-0.363239793753013,
                0.517,-0.0187890805040064,-0.766036638701815,0.690989754181344,-0.365466484080133,
                0.518,-0.0163303296592894,-0.765789489463168,0.690487508985215,-0.365161209014893,
                0.519,-0.0138082709030448,-0.766156144562262,0.69155924690015,-0.365973762788838,
                0.52,-0.0111379326222218,-0.767526335264066,0.694235062473088,-0.367388257716386,
                0.521,-0.00848529081219019,-0.768106607387082,0.695247764430997,-0.368257747442249,
                0.522,-0.00592906778000479,-0.769162528987686,0.69881115955166,-0.371106847339586,
                0.523,-0.00351724457214119,-0.769212560401962,0.699122482346123,-0.371393658006803,
                0.524,-0.00107668110177876,-0.768233764687834,0.696772137980332,-0.370046465841696,
                0.525,0.00140262386818418,-0.768893355438264,0.698162814142823,-0.370964798057129,
                0.526,0.00392489227421638,-0.768294503121763,0.695286954223135,-0.368517494154785,
                0.527,0.0063244407009775,-0.767262619932799,0.692708670519611,-0.366625061860476,
                0.528,0.00902090225116772,-0.768748605545084,0.696101649958563,-0.368679144659899,
                0.529,0.0116878751491284,-0.769284703712738,0.696395990796499,-0.368714362297256,
                0.53,0.0141703946830059,-0.76943285841304,0.696737961780129,-0.36881799118413,
                0.531,0.0165527900374312,-0.767873559777706,0.69282942058307,-0.366082940685132,
                0.532,0.0190714511690079,-0.768715361013688,0.69531351326846,-0.36775586354935,
                0.533,0.0215880133959013,-0.768046405589406,0.69354969133233,-0.366802879215564,
                0.534,0.0241224718571697,-0.768010883811677,0.693441273314706,-0.3666697088463,
                0.535,0.0267838273901896,-0.769246290637208,0.695645947007237,-0.367832727464351,
                0.536,0.0292873816620324,-0.768695528788256,0.693542420390794,-0.366140997998473,
                0.537,0.0316469313509129,-0.767967288325907,0.692220112119151,-0.365372429616979,
                0.538,0.0340288844583962,-0.766787186593798,0.688828223223651,-0.36303027142866,
                0.539,0.0365591171020113,-0.766819139858526,0.688434941574507,-0.362636557206196,
                0.54,0.0393713327455103,-0.76945182700357,0.693512953750576,-0.365444158503002,
                0.541,0.0421048670170626,-0.770684747532458,0.694555005768632,-0.365527089548349,
                0.542,0.0444472250897437,-0.76912741448776,0.691552854217957,-0.363960286681645,
                0.543,0.0470478534809317,-0.769412781143954,0.69159993141671,-0.363869641744485,
                0.544,0.0496337431946362,-0.769417467503338,0.690168182079962,-0.362339216203141,
                0.545,0.0520457801332479,-0.768640678496513,0.688856561408803,-0.361649593960231,
                0.546,0.0545351257803197,-0.766831419600285,0.683115964794713,-0.357384529950726,
                0.547,0.0567656812100118,-0.765280900233792,0.680809318441847,-0.356435600163117,
                0.548,0.0593942336559912,-0.765676965929242,0.681185222994659,-0.356533265414933,
                0.549,0.0617511157241123,-0.763819385102844,0.676902412946472,-0.353736952106453,
                0.55,0.0643225930559735,-0.765860427967449,0.681915419018632,-0.356895224899760,
                0.551,0.0666418871777732,-0.763897932952901,0.67737425186682,-0.353999237963421,
                0.552,0.0691270781382937,-0.763975018901908,0.677332773074836,-0.353873741937256,
                0.553,0.0716287931926239,-0.763634385037723,0.676451144634405,-0.353211532344714,
                0.554,0.0739734145550498,-0.761900445077342,0.672669552115925,-0.350952150212386,
                0.555,0.0766348536486985,-0.763518093643077,0.675995363235648,-0.352778603502251,
                0.556,0.07910921790861,-0.762512967682731,0.673250155687136,-0.350815867180513,
                0.557,0.0815512003881004,-0.761852731451918,0.672324841091584,-0.350661776241750,
                0.558,0.083961924995573,-0.760856642881904,0.670274753245347,-0.349424430140953,
                0.559,0.086488144303997,-0.760703654068581,0.670381310433598,-0.349860636406649,
                0.56,0.0888421295492273,-0.75941812715078,0.668056787225726,-0.348775764854007,
                0.561,0.0914324840526452,-0.758969907510592,0.665754571917762,-0.346716809051549,
                0.562,0.093904849994417,-0.759163508627233,0.665519372858183,-0.346174586812531,
                0.563,0.0962424333466322,-0.756523997233005,0.658526286143322,-0.341279765903430,
                0.564,0.0987326041141071,-0.755075450240038,0.6546670305426,-0.338697767933556,
                0.565,0.101511535489263,-0.75691532844541,0.657464781955566,-0.339645745283478,
                0.566,0.104098255435920,-0.755971004762194,0.654039433635486,-0.336924769848571,
                0.567,0.106439785307634,-0.754638765080225,0.651350780487915,-0.335234815171728,
                0.568,0.108749173680564,-0.752711541225478,0.64682902036805,-0.332262122594154,
                0.569,0.111439708259227,-0.754157087627203,0.649725790587084,-0.33380437561589,
                0.57,0.113991530595357,-0.753457049777695,0.646659717169759,-0.331106416219609,
                0.571,0.116645778210766,-0.754148845507822,0.647022887162429,-0.330619028583638,
                0.572,0.119171317007292,-0.753329195849692,0.645416521663323,-0.329715366833499,
                0.573,0.121697384985237,-0.75286427565382,0.643134086732773,-0.327557911952123,
                0.574,0.123837203544267,-0.750260710897219,0.638608351333165,-0.325038023752417,
                0.575,0.126196695996478,-0.748187267013305,0.633181520388511,-0.321163590191458,
                0.576,0.128636532235100,-0.747472006783669,0.631829035292612,-0.320125718134304,
                0.577,0.131240600517758,-0.748270809689766,0.63397246627252,-0.321527511656196,
                0.578,0.133608078345330,-0.747271273309048,0.632288048009909,-0.320622641316717,
                0.579,0.135923560610494,-0.745971187689681,0.630217764127554,-0.319391191889029,
                0.58,0.138199378223501,-0.743925916014445,0.62631532151824,-0.317244281711100,
                0.581,0.140848769839277,-0.744416081550612,0.6259488390019,-0.316346765112566,
                0.582,0.143446696169502,-0.743461804402585,0.623354776149099,-0.314737205615061,
                0.583,0.145955867169884,-0.74250905947657,0.620806427514744,-0.312884550681860,
                0.584,0.148269194971202,-0.739928954002294,0.615075314309098,-0.309202208363751,
                0.585,0.151066706326547,-0.74125010079584,0.617778262678953,-0.311017475703607,
                0.586,0.153590151983949,-0.741422927628576,0.619128341863627,-0.312296525778371,
                0.587,0.156301399350918,-0.742058109295608,0.620299354211538,-0.313113867659674,
                0.588,0.158938653068759,-0.743576002684045,0.623198243799972,-0.314200211554826,
                0.589,0.161644242535534,-0.744901454822968,0.626434908474523,-0.316179377619682,
                0.59,0.164386858909450,-0.746546023383062,0.629807323445594,-0.318063921195757,
                0.591,0.167016664459371,-0.746689249199541,0.62884823891345,-0.316846281834449,
                0.592,0.169767807752805,-0.748896943530054,0.632585962726157,-0.318366048617941,
                0.593,0.172097618452423,-0.746943879854629,0.628119491170562,-0.315436799789059,
                0.594,0.174931337684459,-0.749463996075587,0.632892271495802,-0.317869876993939,
                0.595,0.177744360519661,-0.750499687580977,0.634210830621478,-0.318103158365416,
                0.596,0.180653568253776,-0.75202590276605,0.636357730702287,-0.31877650503297,
                0.597,0.183137979799029,-0.752487167364274,0.639154180631055,-0.32100247839832,
                0.598,0.185717703060857,-0.751493568942043,0.636304710683082,-0.319012679262846,
                0.599,0.188137970353734,-0.748761790360854,0.629082382106914,-0.314120327918803,
                0.6,0.190843009300664,-0.748555193400588,0.627396950360796,-0.312753618329442,
                0.601,0.193216237192744,-0.746741672521015,0.623872293530936,-0.310600637778025,
                0.602,0.19588866790143,-0.747506409887377,0.624480447868187,-0.310242330123372,
                0.603,0.198356612963165,-0.746126997117547,0.62112949468509,-0.308032960647682,
                0.604,0.200840081002803,-0.743671586185764,0.613356105672199,-0.302124300829082,
                0.605,0.203237139611787,-0.740906248669389,0.60626001577748,-0.29746150274383,
                0.606,0.205577161275033,-0.738365320265172,0.59995357647224,-0.292973989093436,
                0.607,0.208344313630531,-0.739790001323717,0.603085615838922,-0.294877723054357,
                0.608,0.210724452921159,-0.738172528669365,0.598747866509667,-0.291615135825574,
                0.609,0.213117551025320,-0.735480939628745,0.593630358259811,-0.288807528730966,
                0.61,0.215863819607422,-0.737939011846814,0.598978464941144,-0.291737362430611,
                0.611,0.218502967077456,-0.738974816666117,0.602003119964678,-0.293664739704758,
                0.612,0.221184501211349,-0.740692642455427,0.606604527566247,-0.296458472916964,
                0.613,0.223998161698201,-0.742197456573854,0.609951825603285,-0.298639424463026,
                0.614,0.226314191186545,-0.741201038153853,0.608599350170202,-0.297604553633425,
                0.615,0.229114166871928,-0.742037702963772,0.610035619327002,-0.298232307800156,
                0.616,0.231945549765132,-0.74388702842405,0.61334348614384,-0.299509421922888,
                0.617,0.23462715574836,-0.74445456167517,0.613825167892169,-0.299111348571562,
                0.618,0.237188237478,-0.743707104764028,0.611156237871221,-0.296835547090665,
                0.619,0.240057314000897,-0.744843427481591,0.611150882306822,-0.295647546300505,
                0.62,0.243011724093744,-0.745106095568561,0.609297467558784,-0.293395857389041,
                0.621,0.245573779710282,-0.743558976934985,0.605160016309267,-0.290533010770667,
                0.622,0.248288700874011,-0.744188824988807,0.606294740237106,-0.290980334765535,
                0.623,0.250939104819019,-0.74473210424231,0.60806458251347,-0.292337018391800,
                0.624,0.253460275522757,-0.744598762013294,0.608193838182611,-0.292374230085973,
                0.625,0.255938892002684,-0.742920478355007,0.604240931455812,-0.28962176974861,
                0.626,0.258555233938579,-0.742256388548632,0.60200293712792,-0.287856345500941,
                0.627,0.26094210679949,-0.739902851526041,0.59648976686768,-0.283930750474012,
                0.628,0.263335868612004,-0.737222023380829,0.589321058893361,-0.278832929284216,
                0.629,0.266057932047351,-0.735357083748906,0.583150564994193,-0.274277994890994,
                0.63,0.268582193845611,-0.7331861906612,0.576306073487868,-0.268967756315886,
                0.631,0.271260082531864,-0.73370573627298,0.577242256666104,-0.269355267371671,
                0.632,0.273941668304149,-0.734005678039394,0.577600469411139,-0.269319210498352,
                0.633,0.276471395700315,-0.731361370305826,0.570055633916544,-0.263766794354732,
                0.634,0.279108823269807,-0.73179120129775,0.570729588558931,-0.263898147830985,
                0.635,0.281482821418974,-0.731042829150233,0.570327533267436,-0.263902199933652,
                0.636,0.283945167410452,-0.72995014865945,0.56878268817302,-0.263106626292982,
                0.637,0.286844595997410,-0.732466726590015,0.573801282444397,-0.265535843468081,
                0.638,0.289482365935127,-0.732682262551285,0.574620436830774,-0.26592351052089,
                0.639,0.292007034129207,-0.732443860058087,0.574472547736102,-0.265734800145263,
                0.64,0.294608989340855,-0.73226742259203,0.574571501996164,-0.265983540351400,
                0.641,0.297307289632709,-0.732191809745385,0.574747182215221,-0.266047830013092,
                0.642,0.300019589722881,-0.732415125101435,0.575022867584638,-0.266027646205654,
                0.643,0.30253582340945,-0.73160878851731,0.573532566025431,-0.264874263116955,
                0.644,0.305028922155737,-0.729107433307071,0.566882254375478,-0.260171695055895,
                0.645,0.307753194474043,-0.727801337043295,0.562136242363444,-0.256668666889645,
                0.646,0.310217141053366,-0.726794293552458,0.560367508125966,-0.255332458576590,
                0.647,0.313099060799653,-0.726686704982568,0.559176662577533,-0.254233823377298,
                0.648,0.315935658765793,-0.727256283084613,0.56061376816815,-0.255313339215943,
                0.649,0.318627238933197,-0.726839928129403,0.558765555523934,-0.253670712666070,
                0.65,0.321558689734686,-0.728141479688519,0.560824397014638,-0.254491129854848,
                0.651,0.323924356433468,-0.727493030373572,0.561382195588753,-0.255154432892025,
                0.652,0.326619516761739,-0.727615548397027,0.561423163673738,-0.254864534283997,
                0.653,0.329306954998534,-0.728259613886502,0.564104023523529,-0.257062003298996,
                0.654,0.332171562081995,-0.729255061123962,0.566443310677798,-0.258532267287138,
                0.655,0.334988850304582,-0.730959731245365,0.570007322537327,-0.26045121109876,
                0.656,0.337979199089218,-0.732679333359699,0.573033853087863,-0.261945681492266,
                0.657,0.340482987939252,-0.731854878521394,0.572107031831446,-0.261422062506338,
                0.658,0.343164500585406,-0.732078486736132,0.571966054673650,-0.260690988951745,
                0.659,0.346031529363402,-0.733571152707426,0.57578531557935,-0.263162932980873,
                0.66,0.348564770404941,-0.732731860209138,0.574594766743092,-0.262448381776051,
                0.661,0.351265673067858,-0.732743340394891,0.574455242197749,-0.262129415822109,
                0.662,0.354379025442224,-0.734898887096008,0.578824748438069,-0.264658991776188,
                0.663,0.357420147211666,-0.738244608244833,0.586161545889349,-0.269018170385232,
                0.664,0.360309297006228,-0.738278639015615,0.584894059983026,-0.267658238944232,
                0.665,0.363294086619387,-0.74074598143444,0.590428560313839,-0.270823550610427,
                0.666,0.365928899225005,-0.740937167559477,0.590982580109212,-0.270869576561480,
                0.667,0.368577686062853,-0.738640145619669,0.584758819950967,-0.266436987129309,
                0.668,0.371394961373130,-0.738148899964385,0.581086090333411,-0.262846862927690,
                0.669,0.37359970106882,-0.734191488530287,0.573240085420622,-0.258188957042093,
                0.67,0.376183668025923,-0.731315652304346,0.566043142512479,-0.253242642402693,
                0.671,0.379024318521736,-0.730021313309703,0.561067860603632,-0.249239982257023,
                0.672,0.381558957349434,-0.728677430959734,0.559000668944393,-0.248198836722857,
                0.673,0.384610407278653,-0.730895738360053,0.563713975885182,-0.25077018567191,
                0.674,0.387237652791426,-0.730328201433371,0.563302410595627,-0.250490198295644,
                0.675,0.389994871800612,-0.729059175714503,0.559898144912629,-0.247944348172129,
                0.676,0.392886535135056,-0.729441054922536,0.559319221616054,-0.246516557509306,
                0.677,0.395597371575382,-0.728708243356625,0.557378122471667,-0.244849316486492,
                0.678,0.398115237783421,-0.727419315770162,0.555599697858561,-0.243863204932089,
                0.679,0.40071537818241,-0.727060214905095,0.55531782473577,-0.243577531024138,
                0.68,0.403112567324128,-0.722669859795574,0.544724272767085,-0.236474683206776,
                0.681,0.406212629323279,-0.726299804974085,0.55374127318303,-0.242223099066465,
                0.682,0.408948029475487,-0.726831930017441,0.556294732580312,-0.244112495950577,
                0.683,0.411768946670843,-0.728994722981472,0.562387719821759,-0.247901820088316,
                0.684,0.414170532752839,-0.72581797826651,0.555268231236315,-0.243148122429684,
                0.685,0.417120655012693,-0.727925671624206,0.559362657866601,-0.244761368119636,
                0.686,0.420176003387037,-0.729966565120843,0.563281974272152,-0.246645617096407,
                0.687,0.423221991846637,-0.73114600210997,0.564742724956495,-0.246936184736727,
                0.688,0.426120949264943,-0.731483453056867,0.565022914552836,-0.246488984311664,
                0.689,0.428640043105329,-0.728665371896887,0.558686989722988,-0.242246687596638,
                0.69,0.431444210279406,-0.730349523181447,0.563991163352938,-0.245953997392502,
                0.691,0.434146053948108,-0.729194951603821,0.561785049894985,-0.244466513383683,
                0.692,0.437025058727524,-0.728187178435198,0.558137269081808,-0.241568242271633,
                0.693,0.439871446371976,-0.726759838690878,0.553319630529704,-0.237730756580901,
                0.694,0.442519538112373,-0.72611450983171,0.552747906482381,-0.23744654062907,
                0.695,0.445518028572154,-0.726822549447369,0.554424105725742,-0.238410571309901,
                0.696,0.448245738334999,-0.726005062656077,0.553378205738312,-0.237650394956884,
                0.697,0.45095204153492,-0.7253752711623,0.552242047758819,-0.236818500731784,
                0.698,0.454251433127409,-0.728317260672021,0.558113736063117,-0.239970478886507,
                0.699,0.457204960787633,-0.72844993306872,0.556836164452299,-0.23819401551906,
                0.7,0.460214092730119,-0.727738315163293,0.553978925006576,-0.235735590938992,
                0.701,0.462937852420018,-0.726191038831625,0.550890320585403,-0.233871481189212,
                0.702,0.465759861816334,-0.725226131718852,0.547826480697923,-0.2313679554956,
                0.703,0.468651041188810,-0.725512402960507,0.548868268315475,-0.231895464471356,
                0.704,0.471521283719275,-0.725348855572121,0.548393339500242,-0.231298480280433,
                0.705,0.474646438584222,-0.725817282221365,0.548575641644544,-0.231053502021089,
                0.706,0.477594757586279,-0.725707356871241,0.547459801113671,-0.229636381826769,
                0.707,0.480283232762909,-0.724348807143844,0.545657624974244,-0.228614716527258,
                0.708,0.483196773756642,-0.724475812165317,0.546442424465716,-0.228987492430434,
                0.709,0.485867839157565,-0.722930596775239,0.543180208657945,-0.226575285321513,
                0.71,0.488966814305445,-0.724600350319037,0.546984422808932,-0.228577261209097,
                0.711,0.491516921715826,-0.723329768961494,0.545506359135498,-0.22777583240391,
                0.712,0.494192376410099,-0.72080123558798,0.538914613021134,-0.222837713115858,
                0.713,0.497128523755802,-0.72178830584105,0.541989295900547,-0.224798687707680,
                0.714,0.500083671572555,-0.721384950483269,0.539721992575293,-0.222622255881594,
                0.715,0.503446883199402,-0.724949643987612,0.547160270870432,-0.226876022044724,
                0.716,0.506352034800429,-0.72291673680868,0.541234475415732,-0.222602285824956,
                0.717,0.509562577511903,-0.725592268557819,0.547755477346723,-0.226757731736923,
                0.718,0.512875312665817,-0.72765016145291,0.551811775199301,-0.228870517453322,
                0.719,0.516010837938426,-0.72828144676719,0.552899147321733,-0.229244929626312,
                0.72,0.519330266059716,-0.730833534567846,0.557972943256783,-0.231571699227370,
                0.721,0.522481736309124,-0.730307670403736,0.554277859735198,-0.228313471767863,
                0.722,0.525491822420222,-0.730235792219832,0.554537207295759,-0.22843693736519,
                0.723,0.528674203017227,-0.731084453780925,0.555250392365608,-0.228200204316813,
                0.724,0.531729456430093,-0.732306104023217,0.559551992590072,-0.231411993130249,
                0.725,0.534646740689368,-0.732173973502458,0.559894238618638,-0.231586752803565,
                0.726,0.537652456110918,-0.731573441870824,0.558312351551413,-0.230398690503195,
                0.727,0.540648951477498,-0.732439466293036,0.561389187430063,-0.232498536752042,
                0.728,0.543429344721277,-0.730996339100301,0.558856725433981,-0.2308020240271,
                0.729,0.546431235985332,-0.729483630108964,0.554580360397992,-0.227173718849774,
                0.73,0.54947876909691,-0.727333639165555,0.547492006110339,-0.221949069459199,
                0.731,0.551970713165691,-0.720785869495834,0.531947681988875,-0.211833557022726,
                0.732,0.555018335003048,-0.720778590461432,0.532491802859568,-0.212228764127997,
                0.733,0.557915308194267,-0.720513781802793,0.531811595509192,-0.211369242555972,
                0.734,0.560957047300058,-0.71998070780719,0.529833816051685,-0.209490423284770,
                0.735,0.564039476343384,-0.719562054812816,0.529134409188054,-0.208965324865935,
                0.736,0.567088345812445,-0.718723026927736,0.5270633306794,-0.207355503714150,
                0.737,0.570141374761102,-0.718960347829768,0.527544943261483,-0.207148024595529,
                0.738,0.573273290068634,-0.718828746186553,0.527365106331598,-0.206953819812521,
                0.739,0.576217832218822,-0.717889433959773,0.526007091010810,-0.20631519551438,
                0.74,0.579335929919994,-0.717842448145595,0.524913440703439,-0.204593609016788,
                0.741,0.582620949355711,-0.719242297577715,0.528438311396168,-0.206673570237292,
                0.742,0.586075420112958,-0.72182795030521,0.533310121589597,-0.208869856023456,
                0.743,0.589418244495409,-0.723203689491489,0.536258591732558,-0.210465476169959,
                0.744,0.592287702627637,-0.720173951219671,0.529558593962048,-0.206248162757452,
                0.745,0.595359628070863,-0.719469578479603,0.527099176521802,-0.203999817327320,
                0.746,0.598535300592977,-0.720870561975198,0.530831282194956,-0.206223461750354,
                0.747,0.601550721483664,-0.71986356349971,0.528815938025303,-0.204507102479117,
                0.748,0.604964697479399,-0.722157262861647,0.532825261514719,-0.206282710007456,
                0.749,0.608021747884689,-0.720920372322376,0.528273208706417,-0.202329366224181,
                0.75,0.611622311887387,-0.723834377196786,0.532495466052847,-0.203562233706432,
                0.751,0.615090397448997,-0.725289737914975,0.534754440356255,-0.204253320275696,
                0.752,0.618762719337763,-0.729360496742047,0.542165881394308,-0.207512389066178,
                0.753,0.62185720317297,-0.728842779204872,0.540865883609387,-0.206180399628026,
                0.754,0.624846364827131,-0.728147203588928,0.540451854231989,-0.205988261307729,
                0.755,0.628162895391837,-0.730238247892512,0.547371490705668,-0.211040301820007,
                0.756,0.63127811385873,-0.730738395307612,0.549735387523415,-0.212593652954147,
                0.757,0.634405208943917,-0.728486208223031,0.543518577404351,-0.208004184330027,
                0.758,0.637388098465332,-0.72876531563997,0.546378683687069,-0.210222874489996,
                0.759,0.640835394109242,-0.730649368119646,0.549633050832231,-0.211571653528791,
                0.76,0.6440383348699,-0.730469969426136,0.549727523013313,-0.211682366320615,
                0.761,0.647137735003774,-0.729335465919956,0.546997505039304,-0.209643835520923,
                0.762,0.650201493848456,-0.729725358137368,0.547924260615468,-0.209839718670148,
                0.763,0.653578291183846,-0.730133067212604,0.548933452493891,-0.210515923961934,
                0.764,0.656720969571572,-0.727733279820124,0.543104389463031,-0.206296379435795,
                0.765,0.65988499267124,-0.726766208292633,0.540856294562655,-0.204550682274685,
                0.766,0.662573464627113,-0.723040578329361,0.533880629517009,-0.200253783126899,
                0.767,0.666000715554778,-0.723498680023944,0.533893511988413,-0.199480883697860,
                0.768,0.669091688374572,-0.723200138252509,0.534116344775939,-0.199409762076057,
                0.769,0.67254677204057,-0.725656924214646,0.540055563273589,-0.202966797869030,
                0.77,0.675622006956919,-0.723652488184599,0.535975056723803,-0.200272186253551,
                0.771,0.678820224342179,-0.721107735490726,0.527884346780466,-0.193515187063276,
                0.772,0.682162976644391,-0.721397219794153,0.528648523368101,-0.193627934730562,
                0.773,0.685673589939419,-0.723303639176425,0.532917473429477,-0.195930387101196,
                0.774,0.688810586409915,-0.721805365714712,0.530055625488208,-0.194056540515546,
                0.775,0.692029055773442,-0.720413730027589,0.527025305527106,-0.191941417366597,
                0.776,0.69555102798164,-0.722616372135226,0.532413299709249,-0.195234422729690,
                0.777,0.698509312395288,-0.719549441504438,0.526020914548167,-0.191034801911392,
                0.778,0.701965917170919,-0.719541451341874,0.525490128849266,-0.190403547848096,
                0.779,0.705314461440549,-0.720149028874462,0.52684605713751,-0.190788165448205,
                0.78,0.708627834754783,-0.72028724463283,0.52682899898588,-0.190161801886398,
                0.781,0.712261859188248,-0.722116732533092,0.531213825424865,-0.192764193539632,
                0.782,0.715919424418304,-0.723462734082012,0.534330739287308,-0.194806600003382,
                0.783,0.71919134380367,-0.723179421091135,0.534922905282315,-0.195158774050301,
                0.784,0.722360262079652,-0.720720520182463,0.529916268770458,-0.192040007092229,
                0.785,0.725617871183467,-0.72154051691169,0.534846738501489,-0.196029534427737,
                0.786,0.729031525812632,-0.72065107996172,0.533188115747014,-0.194860875436891,
                0.787,0.732810366937321,-0.722831894766183,0.537896417449869,-0.197683004310142,
                0.788,0.735782234846165,-0.71999338189669,0.533787866945537,-0.195555865489373,
                0.789,0.739183889384157,-0.720898328792941,0.537117858437426,-0.197610120028684,
                0.79,0.742670946658377,-0.720970648863047,0.537665351489542,-0.197908125701517,
                0.791,0.745782003610855,-0.718190190608791,0.531166167559631,-0.193193818178678,
                0.792,0.749524961113403,-0.720873155056926,0.537689518305314,-0.197262717446449,
                0.793,0.752837413178081,-0.719246396845058,0.535063474859232,-0.195659900508622,
                0.794,0.756238831736152,-0.71742118291407,0.531975553731499,-0.193970405563942,
                0.795,0.759898123430146,-0.718250167965674,0.533368451972738,-0.194239312993165,
                0.796,0.763746503271524,-0.72021886114119,0.53705451522702,-0.196339523907183,
                0.797,0.766853959503311,-0.716533813984707,0.52926018926767,-0.191166633569574,
                0.798,0.770562026537894,-0.71635802857377,0.526939968475224,-0.188327431515299,
                0.799,0.774231443095104,-0.716864401042575,0.527933678107809,-0.188403938833847,
                0.8,0.777567841501373,-0.716454689872085,0.528085749495952,-0.188126579440896,
                0.801,0.781003629161188,-0.715183937936062,0.525766476700784,-0.186204601635467,
                0.802,0.784624540599124,-0.715147299527081,0.52508599077902,-0.185029450854414,
                0.803,0.788618395170773,-0.718219623720055,0.532828720876692,-0.189946511748509,
                0.804,0.792152514556637,-0.719151267661971,0.535627753401411,-0.191396359269284,
                0.805,0.795739743348271,-0.719258342399423,0.536377799259165,-0.191499475613800,
                0.806,0.79938042958447,-0.719072933439409,0.537195217785513,-0.192109739016135,
                0.807,0.802913969144315,-0.718897152872572,0.538164013258854,-0.193226596442784,
                0.808,0.806759443098814,-0.718926948923575,0.536647647116677,-0.191184922216795,
                0.809,0.810490848411853,-0.718384103879035,0.534372506552501,-0.189092655384660,
                0.81,0.814065869056315,-0.717807328240516,0.534785336963826,-0.189708336121143,
                0.811,0.817551751295823,-0.716994249419942,0.535106327844016,-0.190436141715074,
                0.812,0.821342382375167,-0.717053453299788,0.534705020039168,-0.189215843781411,
                0.813,0.825368280421926,-0.720525776996411,0.541372809066377,-0.192036752344885,
                0.814,0.829378011178355,-0.721632763125234,0.542830993765962,-0.192443003056336,
                0.815,0.832729435511344,-0.719424043199164,0.539627472366263,-0.190723790483242,
                0.816,0.836114422955175,-0.716362040386304,0.533880195967277,-0.187215521085149,
                0.817,0.839931979537198,-0.716076039347745,0.532358863837378,-0.185409950741595,
                0.818,0.843569767755515,-0.714513053393734,0.5288857397302,-0.182674265576945,
                0.819,0.847571848295783,-0.714135356841722,0.525113474723477,-0.178676965580161,
                0.82,0.851413029491236,-0.715071139073307,0.528000381934689,-0.180325252357219,
                0.821,0.854899090548952,-0.714168577577182,0.527434937241704,-0.179837661138698,
                0.822,0.85848292394138,-0.713381456605337,0.528482063851328,-0.181334502594193,
                0.823,0.862204133025555,-0.712762898779466,0.528259876653027,-0.181334589057541,
                0.824,0.866247561824707,-0.714429076756836,0.53420924135782,-0.185596635079939,
                0.825,0.869918843896603,-0.713009840492522,0.531659442626592,-0.183772601073549,
                0.826,0.873973113912963,-0.712686077821086,0.52891084670099,-0.180943995814730,
                0.827,0.877760307734411,-0.713025597064013,0.530979191081103,-0.182065681901441,
                0.828,0.88163050766031,-0.713153548232094,0.530860631377293,-0.180995165860731,
                0.829,0.885673762613449,-0.715473702660687,0.538653006916029,-0.186571267397545,
                0.83,0.889222242246055,-0.71298174413149,0.535122611385748,-0.184825156837302,
                0.831,0.893257244861519,-0.713913994486615,0.537844183591351,-0.186178930832156,
                0.832,0.897109738578492,-0.711123491285612,0.531597108216527,-0.182004663387897,
                0.833,0.900925862059747,-0.709692418969472,0.529306996890415,-0.180433810572184,
                0.834,0.905137831792564,-0.711821647070261,0.536057125356153,-0.185019734526646,
                0.835,0.909089565085252,-0.711068642588433,0.535245738917129,-0.184582113309463,
                0.836,0.913710964165962,-0.71571499678981,0.545764684761788,-0.190881324042598,
                0.837,0.917984315354936,-0.719511857901347,0.557040909537139,-0.198591319895604,
                0.838,0.922104903404986,-0.72171763033274,0.564595287775663,-0.203872799504525,
                0.839,0.925973913219306,-0.720036536385932,0.561249346492042,-0.201576794940889,
                0.84,0.929613279658258,-0.718688930944614,0.558584825768841,-0.199018309449936,
                0.841,0.93357026803085,-0.719438393434315,0.562457355862008,-0.201928918418368,
                0.842,0.937557911182812,-0.719347297666648,0.563314298559109,-0.202220754675732,
                0.843,0.942256053287836,-0.720445945468906,0.562526565521653,-0.200368214512986,
                0.844,0.946101659494692,-0.716279110654594,0.554102192204941,-0.195451814139483,
                0.845,0.949913081521164,-0.713202630672682,0.548203919621151,-0.1917459697601,
                0.846,0.95433679085253,-0.713755571758164,0.549053766416212,-0.191952601086246,
                0.847,0.958542974575374,-0.712813091141484,0.548222783495712,-0.191610672458328,
                0.848,0.962549298908851,-0.710574031200137,0.543205291070596,-0.188220983299089,
                0.849,0.967021439577237,-0.714218663114785,0.553685950972615,-0.195162497565667,
                0.85,0.9713115173349,-0.713366856898135,0.550659910039217,-0.192510179924523,
                0.851,0.975281137370803,-0.712037741082657,0.550493227788276,-0.193223825876286,
                0.852,0.97933421365381,-0.710550286766553,0.549350881142381,-0.192789932095184,
                0.853,0.983585259798318,-0.710138409916598,0.548660599214448,-0.191633909976505,
                0.854,0.987381120014674,-0.706969688453241,0.544586038175236,-0.190095108688918,
                0.855,0.99201373774293,-0.708960098836955,0.549476170451998,-0.192903631048551,
                0.856,0.996538280757852,-0.709537652771377,0.551924093750869,-0.194529621504174,
                0.857,1.00042015214414,-0.705133088592633,0.544525394270705,-0.190478443153623,
                0.858,1.00505840844183,-0.706328717447902,0.54688421080988,-0.191397936662870,
                0.859,1.00976546482002,-0.707508466560722,0.550127513668764,-0.193246612462374,
                0.86,1.01401573222036,-0.705539287806911,0.546447663806669,-0.190586565010881,
                0.861,1.01865400884611,-0.706878853876318,0.55000465803264,-0.192276096190376,
                0.862,1.02320146919304,-0.707185662116716,0.552970299521717,-0.194554991615677,
                0.863,1.02789010514515,-0.709315669663813,0.558544862300951,-0.197944710829106,
                0.864,1.03202524261461,-0.707489623728893,0.557596517811652,-0.198147176100111,
                0.865,1.03708057531935,-0.71127431552697,0.565357264135348,-0.201918413797363,
                0.866,1.04168969877273,-0.711317586990935,0.565997727943783,-0.202093850970345,
                0.867,1.04645325373254,-0.70894697111644,0.558401714595128,-0.19635797155027,
                0.868,1.05138409026968,-0.709339603424555,0.558522864765545,-0.195851552035926,
                0.869,1.05623356066278,-0.70830020132392,0.555024160781962,-0.192794414176760,
                0.87,1.06030587131798,-0.703447993762041,0.544745510731785,-0.186045881750560,
                0.871,1.06486411190622,-0.703604021350659,0.546706349326054,-0.186769954809306,
                0.872,1.07006511203870,-0.708907941014705,0.559596448967853,-0.194612474305083,
                0.873,1.07447557054390,-0.705174544826326,0.551305873541175,-0.189059543390724,
                0.874,1.07863748608068,-0.69887790548074,0.536474958204344,-0.179088007158164,
                0.875,1.08331461120185,-0.698414308605183,0.536881614223363,-0.179620387921538,
                0.876,1.08849500427660,-0.703638699847707,0.552394646395201,-0.190018486217080,
                0.877,1.09331703990623,-0.703761380631318,0.555425725622858,-0.192905490875829,
                0.878,1.09819567260735,-0.701801132412724,0.550645824813438,-0.189495575273478,
                0.879,1.10342009534341,-0.705377019422199,0.558782546502755,-0.193694565335589,
                0.88,1.10833913256683,-0.707317757983607,0.565499825805987,-0.198216650667784,
                0.881,1.11291150791487,-0.704174108997507,0.558661330536884,-0.193095764731886,
                0.882,1.11785355333062,-0.702987594634447,0.556731677436851,-0.191828887872350,
                0.883,1.12253712322489,-0.701526443080892,0.557192660185486,-0.193046468885899,
                0.884,1.12758540854593,-0.703003382742177,0.561928213314749,-0.196091047871264,
                0.885,1.13367881675373,-0.710340077435755,0.577952849502548,-0.205380577230657,
                0.886,1.13902627729622,-0.711681476749982,0.57981967641822,-0.205313379445690,
                0.887,1.14399971078530,-0.707347785921669,0.569439530437773,-0.197907707927939,
                0.888,1.14911255817202,-0.706606063241697,0.567415043500001,-0.195845385265326,
                0.889,1.15435693802761,-0.705861652186462,0.565286321046413,-0.193665023199273,
                0.89,1.16001950437010,-0.708489150201572,0.571605143395901,-0.196997586484023,
                0.891,1.16558062931633,-0.712148264786003,0.581322979949031,-0.203138786082912,
                0.892,1.17139593650381,-0.715877459898695,0.590269368337159,-0.208436080861628,
                0.893,1.17604895219366,-0.709385513093506,0.577834325690239,-0.201128695352500,
                0.894,1.18171446299985,-0.710977439003176,0.58260783074802,-0.204397245459562,
                0.895,1.18710295447942,-0.710484826492901,0.583347406004599,-0.205201097254467,
                0.896,1.19290058428561,-0.714429438039935,0.593829876275486,-0.211822139470965,
                0.897,1.19879569715856,-0.718213992323182,0.604350390128027,-0.218554882107192,
                0.898,1.20473865407073,-0.72146253986994,0.613409705790306,-0.224429327215096,
                0.899,1.21035957300690,-0.718154535967463,0.605694660347145,-0.219598882157694,
                0.9,1.21608301231077,-0.716499242945864,0.600306270553003,-0.215151021829784,
                0.901,1.22117376904607,-0.716952881256689,0.607099128612221,-0.220328155890390,
                0.902,1.22691562870285,-0.717794418605897,0.610412291740057,-0.222418899700034,
                0.903,1.23321127990507,-0.722746419294981,0.623516865215899,-0.230871891589532,
                0.904,1.23922678519266,-0.72491113856102,0.630411801634308,-0.235202814863888,
                0.905,1.24539698537027,-0.726294679978712,0.632355234509744,-0.235358710572143,
                0.906,1.25149331707958,-0.727810589308311,0.637069561723694,-0.238048440236376,
                0.907,1.25733499564801,-0.722942453698833,0.621616341596597,-0.226001513772101,
                0.908,1.26338084776582,-0.720687416692307,0.615931083333235,-0.2216183634086,
                0.909,1.26992324020671,-0.721102781220662,0.616940812406302,-0.222323930085835,
                0.91,1.27561639977365,-0.718278659314349,0.612989683395051,-0.219868251348633,
                0.911,1.28182709114975,-0.718240160852018,0.612145912271567,-0.218650506630061,
                0.912,1.28875179509563,-0.725354056409103,0.62886474438134,-0.228497040441010,
                0.913,1.2941262336839,-0.718840769179142,0.61657278432269,-0.220787294278687,
                0.914,1.29993805610373,-0.715063485303068,0.609378662954471,-0.215894625901133,
                0.915,1.30642939019556,-0.716435593134532,0.614975046748286,-0.220055846391116,
                0.916,1.31363903736424,-0.723880611625803,0.632098772390006,-0.229966677877647,
                0.917,1.31997857052896,-0.722323348501756,0.630984113606849,-0.229882227476503,
                0.918,1.32613146652117,-0.717886738841713,0.621280515612924,-0.223099764701217,
                0.919,1.33293700326728,-0.717698096215123,0.62177285585507,-0.223342106343193,
                0.92,1.33974392749008,-0.71992618530471,0.630618263848414,-0.229598025861426,
                0.921,1.34632425565595,-0.7205105437088,0.631831277070027,-0.229247412322185,
                0.922,1.35366879979773,-0.727899506725597,0.650009112199773,-0.240210859063966,
                0.923,1.36026354552976,-0.727410721029859,0.650397603718826,-0.239585334475468,
                0.924,1.36740800267188,-0.729168471196541,0.653748005910387,-0.239980896303907,
                0.925,1.37426233799561,-0.726212525927901,0.645982318792613,-0.233796291393764,
                0.926,1.38170116408065,-0.726065456307685,0.644757650626693,-0.233045381420541,
                0.927,1.38840100531305,-0.723109352411882,0.642175558099457,-0.232911911699693,
                0.928,1.39575091697750,-0.723852200285588,0.644235653222377,-0.233254986363789,
                0.929,1.40314737513544,-0.727128762490613,0.654989806675414,-0.239943334126147,
                0.93,1.41048236627274,-0.727154881619514,0.656812001164519,-0.240615569746816,
                0.931,1.41789550725395,-0.729203045568894,0.66352478881953,-0.244425471184459,
                0.932,1.42522526907201,-0.727709561920979,0.661926312650044,-0.242680360485752,
                0.933,1.43252904798597,-0.726726019235372,0.662959857641229,-0.244290768257529,
                0.934,1.43947996432927,-0.720768227708071,0.655283674951151,-0.240941258461736,
                0.935,1.44768938154597,-0.722889781868843,0.659780067652234,-0.243206659795486,
                0.936,1.45509134183665,-0.721076580813172,0.659252580352612,-0.242597440520134,
                0.937,1.4618156236858,-0.710808999852194,0.640323817382015,-0.231585240874202,
                0.938,1.4695880067523,-0.7073592721791,0.635321345044866,-0.228822247906827,
                0.939,1.47682910998299,-0.699859354451554,0.621479053793837,-0.220499612115909,
                0.94,1.48608007738593,-0.708584326686589,0.641253078626755,-0.231338308912633,
                0.941,1.49503532305523,-0.707006757267068,0.637106901700657,-0.228145922419532,
                0.942,1.50524722776299,-0.723642373965293,0.680543900691888,-0.257459004653904,
                0.943,1.51375335098784,-0.719050781645726,0.670146802214997,-0.249503694414578,
                0.944,1.52240997527123,-0.721527192827333,0.683124908479328,-0.259506364695227,
                0.945,1.53161936086803,-0.719176518952054,0.677121520352203,-0.254892062899382,
                0.946,1.54097409999639,-0.722955662604641,0.687862370843709,-0.261910321389339,
                0.947,1.55066162787495,-0.727536916630389,0.697748189072037,-0.267018402477823,
                0.948,1.55964323720581,-0.724591019514834,0.693911377032714,-0.264546987166201,
                0.949,1.56840277064208,-0.715538988506349,0.675468418200685,-0.253980496812426,
                0.95,1.57740598855837,-0.712476345354233,0.672860089756886,-0.252040408032537,
                0.951,1.58756469738823,-0.717926417278285,0.690879040844283,-0.264822005779699,
                0.952,1.59806386227459,-0.722606903696733,0.704424222379605,-0.273878906534255,
                0.953,1.60761724304468,-0.716186263633032,0.691178513069775,-0.265435272173814,
                0.954,1.61775864987906,-0.714172625926246,0.691326813103576,-0.267136279517533,
                0.955,1.62728796273953,-0.712486482727972,0.692187202366299,-0.268015376212938,
                0.956,1.63814775884034,-0.711908615501632,0.690624311273415,-0.265190885054158,
                0.957,1.64841037545878,-0.7101586782303,0.69428368742677,-0.270171152232618,
                0.958,1.65911161729729,-0.705688728686392,0.686077336532827,-0.265547504301728,
                0.959,1.67119920352256,-0.714248467278848,0.706259022857916,-0.277946027113978,
                0.96,1.68450969142771,-0.724999888815788,0.726680493407227,-0.288755645563576,
                0.961,1.69566258617525,-0.717248484451869,0.70962040674482,-0.276635747553592,
                0.962,1.70757550591262,-0.718996728094277,0.715302700772183,-0.279240043146746,
                0.963,1.72008187306366,-0.718547595384487,0.716641926425597,-0.28073375260201,
                0.964,1.73321436137037,-0.719030778162595,0.724002840454544,-0.288683805955351,
                0.965,1.74667170672122,-0.721687937119621,0.726538426116441,-0.287894068405383,
                0.966,1.75826990325213,-0.711234478730962,0.711032349836705,-0.279382008300976,
                0.967,1.77161672732817,-0.712844326485623,0.720202521449733,-0.286365923647479,
                0.968,1.78613735899467,-0.725907302539923,0.755718690122554,-0.308522801724354,
                0.969,1.79931983033775,-0.723034069466052,0.756827754655753,-0.309866326281926,
                0.97,1.81401984303883,-0.725743174486003,0.766370719107317,-0.317053796114616,
                0.971,1.82950757466766,-0.728209906522424,0.764440626663773,-0.311621818974351,
                0.972,1.84424019300149,-0.728899075758481,0.769868470051961,-0.314877478208305,
                0.973,1.85974660351513,-0.72599796253136,0.765350752286401,-0.311569388809676,
                0.974,1.87382041423005,-0.708558050147651,0.737341724162205,-0.298127166735081,
                0.975,1.88781633964274,-0.684472137148605,0.690675975726021,-0.270249740839062,
                0.976,1.90463517064792,-0.68151823851346,0.683278622319939,-0.26282908460584,
                0.977,1.92328037280988,-0.68736856693601,0.707677207760572,-0.281040080663777,
                0.978,1.94442536757071,-0.705443953424085,0.747176499631204,-0.303464204004552,
                0.979,1.96331170604305,-0.701355896888863,0.750908004200033,-0.309567588802734,
                0.98,1.98260323358401,-0.70527168667032,0.769632251995386,-0.322727931890946,
                0.981,2.00448960598959,-0.710327557105105,0.789851190475994,-0.337961943440371,
                0.982,2.02656006834055,-0.70135977661482,0.76803708981418,-0.322014773554381,
                0.983,2.05099808366925,-0.712051655343036,0.798967326621969,-0.343214954806933,
                0.984,2.07401314203151,-0.7030105099693,0.78273036673443,-0.332334167766153,
                0.985,2.09891046651502,-0.705442458931281,0.804930444735776,-0.350566585423938,
                0.986,2.12644356684745,-0.71538868897656,0.833518759604086,-0.368584784199108,
                0.987,2.15661236774642,-0.720589775886525,0.842909756045138,-0.371369549846183,
                0.988,2.18702726219444,-0.715165513472954,0.839802389194196,-0.370697144901928,
                0.989,2.21799612050171,-0.695025310209443,0.818989056844036,-0.367706323238887,
                0.99,2.25481685026135,-0.69395415477958,0.808842910446052,-0.35566869694947,
                0.991,2.29439305338754,-0.674197207751405,0.758786441908098,-0.321758964047798,
                0.992,2.33779193131495,-0.682628954408745,0.795066207209728,-0.350091398641706,
                0.993,2.38271368501742,-0.644672855487216,0.697140401800127,-0.276650146204881,
                0.994,2.43933445844919,-0.653506444830143,0.7261717636759,-0.300419024302512,
                0.995,2.51155730374491,-0.70868208903182,0.836521850018959,-0.36308612832522,
                0.996,2.59246906320489,-0.736948486391968,0.90254072614004,-0.404023407909106,
                0.997,2.69365419978271,-0.755403474087408,0.949699190756197,-0.430122270730455,
                0.998,2.82650133603303,-0.777030470816962,0.996042472115599,-0.451187365134249,
                0.999,3.03945854876171,-0.622483876879379,0.661601237979549,-0.264395668809877,
                0.99925,3.14453163573064,-0.78947434774842,1.04248192408939,-0.507505813929403,
                0.9995,3.27380880851537,-0.898546088840112,1.35208726986021,-0.74076192408041,
                0.99975,3.45037647527545,-0.85928326616613,1.57938414080673,-1.01369015376128)
 coeffs_ct <- c(0.00025,-3.78978412764071,-3.20823533164629,4.04730002355913,-2.05241166163882,
                5e-04,-3.57957091409786,-3.37062234623562,4.18482485268731,-2.04204168699408,
                0.00075,-3.45641784193387,-3.39500365790673,4.07817269905982,-1.92756430711754,
                0.001,-3.37197087696978,-3.35627452689927,3.91229197626477,-1.80653135859701,
                0.002,-3.17040568594564,-3.33626406914196,3.7495159598112,-1.66831423378873,
                0.003,-3.05443598227613,-3.25513543806639,3.5314751904476,-1.53161978626978,
                0.004,-2.9611933234642,-3.26566659445964,3.56230069027682,-1.56902515420670,
                0.005,-2.89030478020193,-3.25926653940953,3.51665191385015,-1.54022383140702,
                0.006,-2.82725975692624,-3.26408460103747,3.49352972241266,-1.5238054883178,
                0.007,-2.76797874933441,-3.31894681389186,3.6034564933774,-1.59790880492111,
                0.008,-2.71721504573220,-3.34534341858503,3.63586124070177,-1.61638964674146,
                0.009,-2.67335618592060,-3.35508762278466,3.62236626121992,-1.59965924923848,
                0.01,-2.63534779167991,-3.35812123703413,3.61747456733932,-1.59874293199982,
                0.011,-2.60084966643879,-3.37184463896285,3.64768985411621,-1.61999731482231,
                0.012,-2.56590281758995,-3.39033724863732,3.65818523922206,-1.61842284238640,
                0.013,-2.53397728339619,-3.40871333403937,3.67624930858241,-1.62307317110094,
                0.014,-2.50730401390596,-3.40465109295347,3.66030430019777,-1.61465073211265,
                0.015,-2.48079261027888,-3.40450998399818,3.65286439568616,-1.61141970511942,
                0.016,-2.45598616645510,-3.40551055693918,3.63922838027912,-1.60064247400471,
                0.017,-2.43082467703317,-3.41370111686134,3.64681427052889,-1.60654500511132,
                0.018,-2.40822889567736,-3.41969762062298,3.6550650812198,-1.61312812645198,
                0.019,-2.38791187457953,-3.41881750561671,3.64957468784263,-1.61170646671029,
                0.02,-2.36462826736118,-3.44302302323826,3.69057821617256,-1.63569056765452,
                0.021,-2.34359180920842,-3.45409793424307,3.70229824435318,-1.64088352857395,
                0.022,-2.32317506785593,-3.4670460057322,3.71768231759498,-1.64822901381444,
                0.023,-2.30461912775350,-3.46148624448735,3.69028853667014,-1.62948554872212,
                0.024,-2.28569306746622,-3.47524824167980,3.71185142323419,-1.64111539149067,
                0.025,-2.26937910293245,-3.46688526395229,3.68663518467994,-1.62674933621686,
                0.026,-2.2524090498452,-3.47206217285434,3.69019657028179,-1.62922161997976,
                0.027,-2.23612943530276,-3.47692372513364,3.6942489630045,-1.63153951673626,
                0.028,-2.22256890323353,-3.46402134561667,3.65914581131051,-1.60924286154993,
                0.029,-2.20695791791507,-3.46647296540061,3.65354144347162,-1.60400113257188,
                0.03,-2.19252324062359,-3.46320869296725,3.63919615243967,-1.59609994495544,
                0.031,-2.18034913303160,-3.44451486047885,3.59604012002829,-1.57221156029636,
                0.032,-2.16594038693516,-3.45821854265464,3.62185108200651,-1.58784303165547,
                0.033,-2.15158720676042,-3.46572809674695,3.62890130173875,-1.59159643356251,
                0.034,-2.13950711544811,-3.46245049249192,3.62051577308596,-1.58772187993700,
                0.035,-2.12633644240796,-3.46034753303024,3.61001983048411,-1.58241382326710,
                0.036,-2.11361635995665,-3.46713277157724,3.61833116391975,-1.58594079566347,
                0.037,-2.10061649840776,-3.46890444265924,3.61366881675647,-1.58310489643546,
                0.038,-2.08822908234159,-3.47500733466954,3.61971734952522,-1.58614386572838,
                0.039,-2.07581031490260,-3.47915170368091,3.61756711274197,-1.58298649078981,
                0.04,-2.06478728982969,-3.47857511044847,3.61548397460455,-1.58375733410575,
                0.041,-2.05320316919027,-3.48121860919926,3.61408246463813,-1.58206710993174,
                0.042,-2.04198095961354,-3.48023857666512,3.60274677406480,-1.57334956104005,
                0.043,-2.03051490095486,-3.48488805373684,3.60733241894645,-1.57606699109377,
                0.044,-2.01978805542053,-3.48897536971257,3.60984385514494,-1.57686721074809,
                0.045,-2.00969435722686,-3.48582968375486,3.59693459227820,-1.56855831989881,
                0.046,-1.99938044734368,-3.48599388965453,3.59166102347024,-1.56512331071694,
                0.047,-1.98936854182810,-3.48727411002187,3.58886151899152,-1.56277448229365,
                0.048,-1.97946716197815,-3.49034749557286,3.59283521644371,-1.56559980760569,
                0.049,-1.96880319196224,-3.49435735638820,3.59140060656684,-1.56252363475081,
                0.05,-1.95882711653507,-3.49714611794387,3.59394210199629,-1.56514147726352,
                0.051,-1.9488160302488,-3.50453447660413,3.60468249369075,-1.57069916193283,
                0.052,-1.9390140885604,-3.50431058996178,3.59577113160323,-1.56450017824098,
                0.053,-1.93042788742286,-3.50294414758472,3.59184765387814,-1.56317010260309,
                0.054,-1.9224075524665,-3.50050659448228,3.58655896900364,-1.56100603414680,
                0.055,-1.91286175903987,-3.50725304926478,3.59807368615424,-1.56879769762595,
                0.056,-1.90407786527241,-3.50193072736634,3.57869939692199,-1.55645343317070,
                0.057,-1.89378858655262,-3.51293996685976,3.59391953782593,-1.56415263404119,
                0.058,-1.88377564145454,-3.527784022646,3.62179352345751,-1.58117047718117,
                0.059,-1.87525723921556,-3.53031214630151,3.62242493539344,-1.58105138103714,
                0.06,-1.86717488316082,-3.52750561026472,3.60920685016152,-1.57178767715725,
                0.061,-1.85927623004704,-3.52281753681867,3.5922461172346,-1.56001775780875,
                0.062,-1.85070176665781,-3.52615386908037,3.59646600377308,-1.56338038770033,
                0.063,-1.84201948225015,-3.53037771896844,3.59929549517934,-1.56414348401354,
                0.064,-1.83438639026155,-3.53322384128457,3.60455054988223,-1.56745485772594,
                0.065,-1.82729289311986,-3.52837766902744,3.58924213504375,-1.55773831419773,
                0.066,-1.81955901005780,-3.53053097408684,3.59183716853575,-1.56010821759206,
                0.067,-1.81153039342839,-3.53531976182980,3.59821834774658,-1.56304694868498,
                0.068,-1.80408986834727,-3.53402662770848,3.58990895279878,-1.55758297082863,
                0.069,-1.79681449422266,-3.53446657930784,3.5881317845725,-1.55704206931666,
                0.07,-1.78964560213756,-3.53660756951779,3.59133024093446,-1.55913355263508,
                0.071,-1.78181008614769,-3.53953211627406,3.59038366917734,-1.55699748148553,
                0.072,-1.77461318268542,-3.53988107149449,3.58704437725648,-1.55511012371409,
                0.073,-1.76796140946732,-3.54036106825801,3.58800406193005,-1.55650451284310,
                0.074,-1.76125498749273,-3.53721939049724,3.57840689777443,-1.55126467842489,
                0.075,-1.75434498787407,-3.53722964021949,3.57413208968153,-1.54831002244005,
                0.076,-1.74684395567667,-3.54221419212396,3.57991329204303,-1.55094961154348,
                0.077,-1.74024410096876,-3.54161637815162,3.57677981164088,-1.54925458268876,
                0.078,-1.7329378383175,-3.54830133470605,3.58692304205729,-1.55445005881947,
                0.079,-1.72658010088185,-3.54866719379799,3.58685558661444,-1.55544738675723,
                0.08,-1.71973060212110,-3.55100370959440,3.59026577406198,-1.55858575975322,
                0.081,-1.71293931596212,-3.55433675571582,3.59354627909589,-1.560014544446,
                0.082,-1.70696176574383,-3.54928551873145,3.57881465050506,-1.55090305479540,
                0.083,-1.70091962119641,-3.55003970846986,3.57933852112361,-1.55145342868728,
                0.084,-1.69472308909617,-3.54528047019661,3.56420741153806,-1.54220339870623,
                0.085,-1.68833914025617,-3.54681179198873,3.56645918848675,-1.54459924993765,
                0.086,-1.68196613265689,-3.54600170941476,3.56129869382292,-1.54192769148412,
                0.087,-1.67563171896711,-3.54715048250913,3.56165038427042,-1.54263054549753,
                0.088,-1.66943834072272,-3.54674948898646,3.55867633311726,-1.54137699276360,
                0.089,-1.66326280542135,-3.54749763705478,3.55635309988316,-1.53957109649759,
                0.09,-1.65749938190920,-3.54532330839271,3.54922429440516,-1.53581511475308,
                0.091,-1.6515621430031,-3.54532098431658,3.54787885495207,-1.53583853499856,
                0.092,-1.64504039148643,-3.55107195873042,3.55902398748878,-1.54321367871207,
                0.093,-1.63966871231305,-3.54904169115726,3.55278455817938,-1.53961034347016,
                0.094,-1.63351601180481,-3.55057960274464,3.55080708541101,-1.53775509086970,
                0.095,-1.62787076371243,-3.55198363081309,3.55358756415687,-1.54068285576103,
                0.096,-1.62212207316649,-3.55304172041018,3.55222395919094,-1.53901571242119,
                0.097,-1.61587872442211,-3.55887782201436,3.56159833581324,-1.5442573973801,
                0.098,-1.60969588079742,-3.56424988203964,3.57137397424182,-1.55081045387212,
                0.099,-1.60398126430324,-3.56439935590167,3.56727123635812,-1.54770350711821,
                0.1,-1.59757122958744,-3.57257618803882,3.58204786099081,-1.55632293559432,
                0.101,-1.59211212096117,-3.57217384356903,3.57868292231209,-1.55451787739291,
                0.102,-1.58630597615237,-3.57198091588415,3.57230417859458,-1.54975873272457,
                0.103,-1.58067275089643,-3.57561819610119,3.57897661566994,-1.55413708020418,
                0.104,-1.57520255059259,-3.57752311092898,3.58275163849512,-1.55763362718922,
                0.105,-1.5699010267011,-3.57734912592007,3.58033005612774,-1.55649795940826,
                0.106,-1.56405529800801,-3.58173916710221,3.58734404761219,-1.56108505661783,
                0.107,-1.55860307507229,-3.58291723842287,3.58732576146586,-1.5613825594213,
                0.108,-1.55304396211545,-3.58469798317945,3.58776200733597,-1.56150502860273,
                0.109,-1.54716158280245,-3.59080533873920,3.59727790911508,-1.56657594240547,
                0.11,-1.54229317934544,-3.58744909986267,3.58533383287463,-1.55831904481674,
                0.111,-1.53729727958449,-3.58581836205418,3.58016534486557,-1.55567776583386,
                0.112,-1.53215291453314,-3.5858617354564,3.5776298979937,-1.55422440919352,
                0.113,-1.52687514459771,-3.58726746017232,3.57865423177671,-1.55552339415314,
                0.114,-1.52224277186467,-3.58541902099623,3.57348528562739,-1.55258478665625,
                0.115,-1.51724500507078,-3.58559319407687,3.57380226567991,-1.55406292254268,
                0.116,-1.51210374036313,-3.58835093143056,3.57906233189525,-1.55816687358530,
                0.117,-1.50686772152876,-3.59234851491927,3.58544897582788,-1.56196008161396,
                0.118,-1.50161590563579,-3.59677760858831,3.59424315642861,-1.56834069252925,
                0.119,-1.49651777226816,-3.59883742881495,3.59585333844034,-1.5690329683279,
                0.12,-1.49197036114105,-3.59548664691484,3.58827368657107,-1.56571207031590,
                0.121,-1.48670929437744,-3.59990234880246,3.59595893485689,-1.57067745169597,
                0.122,-1.48145028344993,-3.60279118618914,3.59799810740180,-1.57133624627360,
                0.123,-1.47700100416075,-3.60005959837952,3.5907621173314,-1.56745669128925,
                0.124,-1.47170171172798,-3.60364264153394,3.59651236115959,-1.57135883267737,
                0.125,-1.46666513913877,-3.60603535538259,3.59866188741918,-1.57211125875839,
                0.126,-1.46130890732288,-3.61018646697119,3.60364292579465,-1.57455014826883,
                0.127,-1.45644783096553,-3.61078110037765,3.60114497602107,-1.57276377510746,
                0.128,-1.45131749911043,-3.61474923754241,3.60588661545278,-1.57489011728387,
                0.129,-1.44612687946939,-3.61955727866982,3.61366784665675,-1.57930985853572,
                0.13,-1.44118949576684,-3.62171213852456,3.61489497204215,-1.57987500909489,
                0.131,-1.43641262902009,-3.6237197211214,3.61682981499295,-1.58104200317802,
                0.132,-1.43206183639180,-3.62303008009778,3.61424080664666,-1.58017566235887,
                0.133,-1.42768972190412,-3.62232424277564,3.61230859860844,-1.579949937076,
                0.134,-1.42310541026479,-3.62266189974173,3.60970034167961,-1.57783259206434,
                0.135,-1.41879222423103,-3.6212101213862,3.60507084522533,-1.57538128721038,
                0.136,-1.41439030205345,-3.62054255740019,3.60205541393662,-1.57392993335977,
                0.137,-1.40992525532084,-3.6211606195563,3.60172172356308,-1.57425462608792,
                0.138,-1.40508032595162,-3.62507748497444,3.60787449505004,-1.57785769988113,
                0.139,-1.40009212268231,-3.62833325679972,3.61177667254703,-1.58011641758734,
                0.14,-1.39490073125079,-3.63599400960887,3.62706162241497,-1.5897969153095,
                0.141,-1.39028828966870,-3.63763880177071,3.62876828738659,-1.59128313058652,
                0.142,-1.38556660831134,-3.64078041317161,3.63281075453080,-1.59365615498779,
                0.143,-1.38154088096818,-3.63819626917442,3.62378598225053,-1.58738238598061,
                0.144,-1.37738738584858,-3.63741085187774,3.62057095307014,-1.58580764512589,
                0.145,-1.37300563031930,-3.63803147689008,3.61897827873653,-1.58454364002832,
                0.146,-1.36900561713491,-3.63639084350702,3.61334761480565,-1.58143121081524,
                0.147,-1.36517190130988,-3.63485199083902,3.60987236775149,-1.58020319664209,
                0.148,-1.36095481609251,-3.63524851512287,3.60993897862565,-1.58076269284358,
                0.149,-1.35709483708270,-3.63496653148477,3.60956403253883,-1.58159654611384,
                0.15,-1.35313950735832,-3.63260378618106,3.60261889626882,-1.57770031176964,
                0.151,-1.34897114125728,-3.63449053653007,3.60674781585916,-1.58134164417166,
                0.152,-1.34474559709646,-3.63642828118224,3.60896083096191,-1.58276803291553,
                0.153,-1.33967043183968,-3.64361397521973,3.61950789699534,-1.58798976733986,
                0.154,-1.33547037938397,-3.64361803158382,3.61658969251788,-1.58614113919595,
                0.155,-1.33112512765646,-3.64705304415421,3.62309026164256,-1.59029112200726,
                0.156,-1.32684576967032,-3.64958088520279,3.62588415580880,-1.59169003083156,
                0.157,-1.32272815750790,-3.65071751069339,3.62549625610595,-1.59090992550222,
                0.158,-1.31891770176882,-3.65022228570605,3.62405567428804,-1.59067730175865,
                0.159,-1.31536554077754,-3.64604836758401,3.61315170110753,-1.58415581184401,
                0.16,-1.31144657031910,-3.64682560196175,3.61273437274792,-1.58374131211003,
                0.161,-1.30723701233295,-3.64713373166022,3.61167304117685,-1.58377936175774,
                0.162,-1.30288614679962,-3.64889428649949,3.61281690380101,-1.58425504607688,
                0.163,-1.29913386373421,-3.64620700353989,3.60493595366635,-1.57969122229302,
                0.164,-1.29494821217840,-3.64758227537235,3.60502900715381,-1.57948113662071,
                0.165,-1.29053194450169,-3.64852904886914,3.60301806456624,-1.57784846732713,
                0.166,-1.28661668775265,-3.65030028979424,3.60579335124528,-1.57990615978524,
                0.167,-1.28277411197573,-3.6502018025092,3.60288166607138,-1.57775057983444,
                0.168,-1.27874335262378,-3.64978900541541,3.59892587043569,-1.57530859351019,
                0.169,-1.27490434305279,-3.64840501543284,3.59234843043634,-1.57055978481834,
                0.17,-1.27069544020079,-3.65184367041095,3.59734490773374,-1.57359522882113,
                0.171,-1.26690331968122,-3.6523653147763,3.59818163564184,-1.57495835374914,
                0.172,-1.26312402781638,-3.65013008686725,3.59014870367994,-1.56990373578654,
                0.173,-1.25853644319225,-3.65534008168059,3.59842476552140,-1.57468951878395,
                0.174,-1.255035246838,-3.65412229948485,3.59510756504548,-1.57318704587877,
                0.175,-1.25153600594183,-3.65222877325245,3.58925384933042,-1.56969799435492,
                0.176,-1.24740718143917,-3.65558143701381,3.59465979074723,-1.57303233858011,
                0.177,-1.24377083203362,-3.65466848151121,3.59205836633783,-1.57199787521076,
                0.178,-1.24022233930569,-3.6524441405208,3.58525315754249,-1.56803515015828,
                0.179,-1.23610518469521,-3.65550931829817,3.58992004498252,-1.57091660732952,
                0.18,-1.23214675370319,-3.65528626349619,3.58583121616825,-1.56813886366327,
                0.181,-1.22824009270803,-3.65574519745533,3.58404025140571,-1.56680963281495,
                0.182,-1.22427352207115,-3.65755768761942,3.58595760488381,-1.56812360304104,
                0.183,-1.22036911714565,-3.65878830556942,3.58630291278505,-1.56823809395562,
                0.184,-1.21706825585522,-3.65558815324469,3.57797646663928,-1.56340944167899,
                0.185,-1.21345849519563,-3.65471034328657,3.57386235981090,-1.56116672934181,
                0.186,-1.20967470490953,-3.65823262063544,3.58130481136102,-1.56623343028356,
                0.187,-1.20555411825787,-3.66337268470793,3.59120938212998,-1.57253115490668,
                0.188,-1.20208625740471,-3.66309294099385,3.58888348990874,-1.57109940107552,
                0.189,-1.19880909872218,-3.66099353198995,3.58404919592997,-1.56876642840093,
                0.19,-1.19513600575594,-3.66242188199303,3.58618768583082,-1.57036905833118,
                0.191,-1.19182320579785,-3.66146330465908,3.58430607504464,-1.57010838740840,
                0.192,-1.18795777716289,-3.66412061218165,3.58727704656507,-1.57156302775121,
                0.193,-1.18436810520676,-3.66432257316307,3.58596777648498,-1.57069767199061,
                0.194,-1.18051785178209,-3.6660894049044,3.58733648402745,-1.57134701358191,
                0.195,-1.17640647623497,-3.67151829970111,3.59701366766049,-1.57713319331097,
                0.196,-1.17281700387543,-3.67134282446416,3.59520400720233,-1.57656811899341,
                0.197,-1.16891310626704,-3.67438085586399,3.59876255658667,-1.57840593428851,
                0.198,-1.16546627296646,-3.67298817363904,3.59450617282856,-1.57621452410510,
                0.199,-1.16190925698582,-3.67361630755466,3.5937576600839,-1.57575705917146,
                0.2,-1.15853069516762,-3.67269502106012,3.58881336406927,-1.57225840637783,
                0.201,-1.15500837179334,-3.67310177823424,3.58887387387739,-1.57274219494366,
                0.202,-1.15201103045763,-3.67184311395966,3.58635103413661,-1.57188498060498,
                0.203,-1.14864380949937,-3.67192076424667,3.58547831175773,-1.57174169776926,
                0.204,-1.14485414870064,-3.67373967088531,3.5855634986363,-1.57093874449826,
                0.205,-1.14126892924436,-3.67529498851828,3.58691865977504,-1.57154483092954,
                0.206,-1.13729529768640,-3.68032967618346,3.59662333441793,-1.57776685687711,
                0.207,-1.13372401805670,-3.68108532879637,3.59481709199105,-1.57608218957567,
                0.208,-1.13031628796527,-3.68228524381184,3.59572180448438,-1.57639119011754,
                0.209,-1.12678606658228,-3.68391918739779,3.59782598129640,-1.57807820210058,
                0.21,-1.12322867141398,-3.68521640538804,3.59865227309561,-1.578472623082,
                0.211,-1.11926713053212,-3.68936201879515,3.60447977382846,-1.58156333350801,
                0.212,-1.11557264930348,-3.69314108483045,3.60998148490386,-1.58418308399247,
                0.213,-1.11199833788528,-3.69497058306733,3.61244975892495,-1.58589911672703,
                0.214,-1.10852470042736,-3.69663167912943,3.61503337317786,-1.58766514499810,
                0.215,-1.10516543162704,-3.69726475830523,3.61550004198241,-1.58842812601277,
                0.216,-1.10191538955196,-3.69665759226094,3.61210997760328,-1.58616675600672,
                0.217,-1.09880969802017,-3.69639842758355,3.61071763457026,-1.58553910771636,
                0.218,-1.09538686469958,-3.69633346885758,3.60820621774789,-1.58399646633666,
                0.219,-1.09179128828312,-3.69746866279426,3.60722125318272,-1.58293204642012,
                0.22,-1.08892031782452,-3.69488756136685,3.60104138027206,-1.57966309029774,
                0.221,-1.08550683237930,-3.69676024884975,3.60369670494886,-1.58130454913455,
                0.222,-1.08240536051742,-3.69653329068369,3.60337812763625,-1.58182277158198,
                0.223,-1.07931408051940,-3.69528571362694,3.59906647257494,-1.57916838960837,
                0.224,-1.07572074021325,-3.69817971737854,3.60344209887735,-1.58193458006910,
                0.225,-1.07193803181066,-3.70168937504218,3.60768653477369,-1.58405889788706,
                0.226,-1.068961711226,-3.70017783226215,3.60348758307983,-1.58197395886401,
                0.227,-1.06568269743190,-3.70047849010469,3.60308629419498,-1.58210359427469,
                0.228,-1.06276244461927,-3.69909016500906,3.60030168990093,-1.58128374220844,
                0.229,-1.05948431621868,-3.70032194236284,3.60124409753117,-1.58187843462126,
                0.23,-1.05613238214562,-3.70065294838262,3.59986284362591,-1.58104031406537,
                0.231,-1.05316519052728,-3.6988577776412,3.59497532752382,-1.57846420678327,
                0.232,-1.04994377379699,-3.69960258098352,3.59485750037019,-1.57813065966165,
                0.233,-1.04700762475351,-3.69642076268432,3.58494805227383,-1.57162697500036,
                0.234,-1.04359219838051,-3.69803888856646,3.58673102369264,-1.57296750129940,
                0.235,-1.04019538246931,-3.69981854245656,3.58960083146773,-1.57517839039696,
                0.236,-1.03642423531605,-3.70574759134583,3.60116672731208,-1.58246536576543,
                0.237,-1.03313915065391,-3.70807008398857,3.60487255804191,-1.58482139759842,
                0.238,-1.02968391211447,-3.71105076376811,3.60914159554176,-1.58710299725057,
                0.239,-1.02643482409721,-3.70953249998946,3.60259064901885,-1.58289982997955,
                0.24,-1.02381627943979,-3.70539912520442,3.59365949582003,-1.57836734782092,
                0.241,-1.02050968861658,-3.70675654298556,3.59426768317367,-1.57862297989421,
                0.242,-1.01734470666593,-3.70782325174596,3.59608388699656,-1.58021880188429,
                0.243,-1.0140241932671,-3.70892517464313,3.5962312662954,-1.58011411353606,
                0.244,-1.01084375338041,-3.71016772178844,3.59774794360385,-1.58096460407836,
                0.245,-1.00747933730965,-3.71273747813214,3.60135824864729,-1.58299863320095,
                0.246,-1.00436337943996,-3.71176974402966,3.59654338932326,-1.57979516085102,
                0.247,-1.00121397157832,-3.71262269673505,3.59698819282528,-1.58013153634132,
                0.248,-0.998096677026035,-3.71327007636583,3.59774029684385,-1.58117664323110,
                0.249,-0.994968943783099,-3.713793281965,3.59708230959847,-1.5807292236861,
                0.25,-0.991845870649857,-3.71453740624237,3.59728541316591,-1.58116462231847,
                0.251,-0.988464239593486,-3.71704770358818,3.60021867065018,-1.58260153865263,
                0.252,-0.98572403139245,-3.71318887600807,3.58905994890432,-1.57552761426487,
                0.253,-0.9829108937282,-3.71218410438366,3.58624304695118,-1.57417535441743,
                0.254,-0.980059266732717,-3.71131654185993,3.58326466352388,-1.57244625460447,
                0.255,-0.976991182355854,-3.71237605285547,3.58432188317312,-1.57330647988775,
                0.256,-0.973698275250127,-3.71405626230695,3.58646385435557,-1.57491286571788,
                0.257,-0.97068148035550,-3.71356499871803,3.58355279791377,-1.57339343993538,
                0.258,-0.967503450855309,-3.71582206383622,3.58721406533336,-1.57599382940083,
                0.259,-0.964449259864957,-3.71587898736119,3.58534176115799,-1.57478530918265,
                0.26,-0.961544511288723,-3.71623754457087,3.5848203827351,-1.57460752057892,
                0.261,-0.95801499701721,-3.71998208200242,3.59085422913752,-1.57838833792023,
                0.262,-0.954850918292223,-3.72166590815818,3.59297397921274,-1.57954552267912,
                0.263,-0.951768123057254,-3.72057306538967,3.58782861646573,-1.57630546124648,
                0.264,-0.948621764007859,-3.72079480898253,3.58562001022577,-1.57456301288204,
                0.265,-0.945640166064357,-3.72169068322083,3.58570145952426,-1.57431751251459,
                0.266,-0.942281579790584,-3.72415447348957,3.58867008499265,-1.57584018873509,
                0.267,-0.939654740686544,-3.7228600136523,3.58623286249356,-1.57526078997951,
                0.268,-0.936243322933489,-3.72691826778961,3.59336586888625,-1.57985257483781,
                0.269,-0.933385435075931,-3.72525905846578,3.58788610791579,-1.57659662642501,
                0.27,-0.930664739928609,-3.72284832777675,3.58061307172872,-1.57205773868006,
                0.271,-0.927574817734219,-3.72473624533079,3.58373331672692,-1.57411873754830,
                0.272,-0.924316980727318,-3.72697444109372,3.58683206869307,-1.57604374422081,
                0.273,-0.920917204365643,-3.73077446907136,3.59285726073761,-1.57957331656074,
                0.274,-0.917795765094703,-3.73231029013171,3.59380039859297,-1.57982068465593,
                0.275,-0.9148351965795,-3.7323645809583,3.59287163544198,-1.57964923109382,
                0.276,-0.911488283141843,-3.73460406103259,3.59546824337148,-1.58108948638842,
                0.277,-0.90796095890562,-3.74002283241296,3.60562181892902,-1.5872044204149,
                0.278,-0.904960599043757,-3.73788637896445,3.59763119174676,-1.58195687234751,
                0.279,-0.901797151587144,-3.74046535450159,3.60189856949649,-1.58461214977793,
                0.28,-0.898909497246351,-3.73995575355374,3.60040448308016,-1.58463634154190,
                0.281,-0.896169861910861,-3.73979150943271,3.59950072710697,-1.58445492471058,
                0.282,-0.893553020946652,-3.73792827903676,3.59402952184165,-1.58105398214804,
                0.283,-0.890386291350285,-3.74002687965715,3.59638367477934,-1.58218380435785,
                0.284,-0.88740717825096,-3.74047126933616,3.59532151691003,-1.58166212510625,
                0.285,-0.88452794309533,-3.73948652483761,3.59110226803116,-1.57903034875105,
                0.286,-0.881339380030080,-3.74080852052442,3.59085312721515,-1.57848545792733,
                0.287,-0.878822441179985,-3.73760138262286,3.58239471116857,-1.57328926111009,
                0.288,-0.875653269439243,-3.73954789310960,3.58486197877726,-1.57485990693921,
                0.289,-0.872435569747613,-3.74164068755878,3.58712561944942,-1.57612533003898,
                0.29,-0.869517133817933,-3.74172654862691,3.5853429869364,-1.57520954587272,
                0.291,-0.86627078724074,-3.74464101064536,3.59017039849074,-1.57835861118685,
                0.292,-0.863434445106713,-3.74484132913954,3.59028413952453,-1.57917233219877,
                0.293,-0.860674253545625,-3.74437799818071,3.58735420731265,-1.57734887351459,
                0.294,-0.857633238038425,-3.74468245496303,3.58582406541867,-1.57650163129574,
                0.295,-0.854769937582836,-3.74495620853009,3.58412713669163,-1.57509301592008,
                0.296,-0.851927756620268,-3.74607675815145,3.585565930358,-1.57607427553209,
                0.297,-0.849549703491266,-3.74499963081186,3.58358227512636,-1.57538335717046,
                0.298,-0.846836841361857,-3.74317527587411,3.57841676505143,-1.57272021092987,
                0.299,-0.844060918993792,-3.74319563897564,3.57716421177825,-1.57202355238819,
                0.3,-0.84113982623721,-3.74364976313238,3.57696404397733,-1.57224757759806,
                0.301,-0.838307167154505,-3.74400765704915,3.57663826819582,-1.57243251517213,
                0.302,-0.835654393418753,-3.74312883598363,3.57370085584208,-1.57075464259490,
                0.303,-0.833111734854667,-3.74136774889355,3.56857811338561,-1.56774681620791,
                0.304,-0.830046863850798,-3.74361649051515,3.57287562158291,-1.57101303544248,
                0.305,-0.827472369759893,-3.74225365257457,3.56871199678928,-1.56851195148739,
                0.306,-0.824677313031853,-3.74198601775746,3.56688293511787,-1.56780233609636,
                0.307,-0.82171211218236,-3.74449731938372,3.57124405605023,-1.57033658713994,
                0.308,-0.818874318431477,-3.74471061564478,3.5703995055006,-1.56996545934491,
                0.309,-0.816124181527337,-3.74475441861164,3.56956985223693,-1.56985179574701,
                0.31,-0.813510750353264,-3.74415145358047,3.56739594582535,-1.5687530801606,
                0.311,-0.810571897845367,-3.74553954104068,3.56862553563529,-1.56941146189224,
                0.312,-0.807991037895409,-3.74486127539473,3.56613848342528,-1.56805239177384,
                0.313,-0.804926319867139,-3.74724571421949,3.56944115968075,-1.57003973515949,
                0.314,-0.801952785731023,-3.74996003838874,3.57580989589898,-1.57481008646960,
                0.315,-0.798988527927984,-3.75215854154051,3.57890297888284,-1.57670386267518,
                0.316,-0.796351559012443,-3.75149678446220,3.57693367206401,-1.57595238203053,
                0.317,-0.793535533246267,-3.75241150678798,3.57895989504644,-1.57799169053968,
                0.318,-0.79095113486755,-3.75014520436132,3.57196992684559,-1.57361139125425,
                0.319,-0.788398087108831,-3.75027627855565,3.57186629761576,-1.57389168223874,
                0.32,-0.785483486768771,-3.75166423936274,3.57353305735022,-1.57515261072799,
                0.321,-0.782323762263466,-3.75460932686964,3.57743787837879,-1.57722558470331,
                0.322,-0.779339017723037,-3.7551733566177,3.57648592650896,-1.57681628351929,
                0.323,-0.77669254559902,-3.75398896720925,3.57272626474923,-1.57485729581174,
                0.324,-0.773968504317278,-3.75384172387055,3.57129035734926,-1.57422063412753,
                0.325,-0.771232958135012,-3.75387762186707,3.56898520828293,-1.57255823755710,
                0.326,-0.768375521975246,-3.75564296646635,3.57215609571954,-1.57492592091361,
                0.327,-0.765629297676903,-3.75754327193393,3.57630663624029,-1.57811954834754,
                0.328,-0.762914431858977,-3.75840102436667,3.57736536015439,-1.57912020228509,
                0.329,-0.760021697170246,-3.75925790492288,3.57769486236315,-1.57940713016258,
                0.33,-0.757239044665326,-3.76043012899221,3.57825953846965,-1.57937584514576,
                0.331,-0.754308263784493,-3.76223045822841,3.58054410775842,-1.58092286718714,
                0.332,-0.751579295237222,-3.76219939203273,3.57969969597834,-1.58094556068016,
                0.333,-0.748873213331698,-3.76208845410864,3.57769982536294,-1.57983690639577,
                0.334,-0.745718733167594,-3.76577201322903,3.58345708646563,-1.58306637426639,
                0.335,-0.743283463306261,-3.76402088923198,3.57887668502232,-1.58067606182856,
                0.336,-0.740450672012572,-3.76553957372627,3.58096437446610,-1.58211884741308,
                0.337,-0.737622985817161,-3.76681926215494,3.58154441678224,-1.58223515221393,
                0.338,-0.734771691830514,-3.76926610274034,3.58615266594530,-1.58545485517688,
                0.339,-0.73179967045229,-3.77103476926004,3.58782704621557,-1.58628160640754,
                0.34,-0.72944916292945,-3.76865125446769,3.58134417171424,-1.58248054494269,
                0.341,-0.726787975274064,-3.76876670906759,3.58085933494475,-1.58269898176483,
                0.342,-0.724110836685295,-3.768935491681,3.58017547015257,-1.58256730555064,
                0.343,-0.721434685243712,-3.76960103785799,3.57993966681419,-1.58216522532167,
                0.344,-0.718526629678873,-3.77181008172797,3.58318849795125,-1.58415852993618,
                0.345,-0.715821003711652,-3.77196984072819,3.58142220211547,-1.5828081870645,
                0.346,-0.713227192702967,-3.77272451799087,3.58222071576437,-1.58340144580994,
                0.347,-0.71021416658226,-3.77553028942928,3.58564720052686,-1.58520386881669,
                0.348,-0.707789877616297,-3.77444499542848,3.58268228020435,-1.58388409186038,
                0.349,-0.705065606022228,-3.77502491919375,3.58155402526514,-1.58280463043700,
                0.35,-0.702397859768155,-3.77468147875483,3.57900435461954,-1.58133260838971,
                0.351,-0.699650722183737,-3.77562464715639,3.57947392543920,-1.58158691149033,
                0.352,-0.696765107873277,-3.77588010830527,3.5764497194901,-1.57898748915904,
                0.353,-0.694146626808937,-3.77636967905891,3.57595223432494,-1.57868770215343,
                0.354,-0.691445367775631,-3.77673053410724,3.57512002558051,-1.57823764275256,
                0.355,-0.689057578685796,-3.77498844533496,3.57022239250777,-1.57538003268896,
                0.356,-0.686387487006833,-3.77470872827131,3.56767034275981,-1.57379656416355,
                0.357,-0.683660668569919,-3.77442750353711,3.56427366233752,-1.57133822175406,
                0.358,-0.680916411188081,-3.77537560678373,3.56450952064846,-1.57148866189956,
                0.359,-0.678141825558352,-3.7760335247574,3.56437848878190,-1.57159548322662,
                0.36,-0.675311001636371,-3.77835849342119,3.56871578455224,-1.57471791940401,
                0.361,-0.672487319564125,-3.77906127538183,3.56766461438834,-1.57384273793039,
                0.362,-0.669838914712224,-3.78058447250352,3.56996228657087,-1.57550184758492,
                0.363,-0.667237251349799,-3.78078843071773,3.56911217294335,-1.57508508688087,
                0.364,-0.66406368294281,-3.78513716054876,3.57670773157690,-1.57960969454820,
                0.365,-0.661335037791531,-3.78567595413699,3.57456271416619,-1.57732825554898,
                0.366,-0.658839933985192,-3.78444288212282,3.57027351641641,-1.57474228463152,
                0.367,-0.656054535437365,-3.78559251211170,3.57048841845877,-1.57469871617969,
                0.368,-0.653655815090014,-3.78372633068929,3.56477325632400,-1.57117358063693,
                0.369,-0.650904081026722,-3.78512520606239,3.56639339596761,-1.57234614766031,
                0.37,-0.648054128136394,-3.78673796384893,3.56778586035673,-1.57303148216709,
                0.371,-0.645266944747893,-3.78992343961146,3.57394642991498,-1.57695812527307,
                0.372,-0.642750313091421,-3.78989225511213,3.57285153405603,-1.57643253697551,
                0.373,-0.64001202820502,-3.79116097546793,3.57302535546079,-1.57587799625618,
                0.374,-0.637332575937909,-3.7911676063366,3.57023216333908,-1.57366206101617,
                0.375,-0.634932067424707,-3.78993987928947,3.56736904612539,-1.57260000856128,
                0.376,-0.632126912712956,-3.79235050837031,3.57090991272647,-1.57470585045808,
                0.377,-0.629921851312922,-3.79011701403185,3.56592520115644,-1.57225115491690,
                0.378,-0.62720415333235,-3.79069941621469,3.5657795840627,-1.57247433571474,
                0.379,-0.624404667722516,-3.791948828033,3.56651354488434,-1.57264083324385,
                0.38,-0.621569866222012,-3.79349381666997,3.56729091091685,-1.57267363486348,
                0.381,-0.619101919277635,-3.793101982267,3.56506577344252,-1.57153457202005,
                0.382,-0.616005999172937,-3.79791280325326,3.5737232553848,-1.57661776681750,
                0.383,-0.613209660465253,-3.80045199836584,3.57861700132050,-1.57996862846062,
                0.384,-0.610400884928362,-3.80197730072308,3.57946620464444,-1.58002148510304,
                0.385,-0.60770538297862,-3.8034746829991,3.5815132950955,-1.58146300912356,
                0.386,-0.605040358350472,-3.80528458700075,3.58506247483342,-1.58419035768567,
                0.387,-0.60255174761584,-3.805683532701,3.58600223358648,-1.58554062078970,
                0.388,-0.599658819751339,-3.80900100902935,3.59133612629587,-1.58860509741667,
                0.389,-0.597046066657834,-3.80970570609452,3.59076225023743,-1.58790325186384,
                0.39,-0.594175982936385,-3.81292681368915,3.59576840490972,-1.59070651959392,
                0.391,-0.591795437351911,-3.81167482855718,3.59117814002375,-1.58768317376067,
                0.392,-0.589276521740525,-3.81169499455036,3.5901815205704,-1.58733321266529,
                0.393,-0.586456445152517,-3.81458385302699,3.59567465890052,-1.59096367662218,
                0.394,-0.583942262176817,-3.81441537988045,3.59400299340784,-1.58997138538763,
                0.395,-0.58154926001284,-3.81317298391666,3.58991356482315,-1.58761168458430,
                0.396,-0.578832558321664,-3.8135618368146,3.58852556592208,-1.58674111415075,
                0.397,-0.576025227927009,-3.81514893391532,3.59076083477006,-1.58859414679582,
                0.398,-0.573390640050321,-3.81561602259668,3.58997259503489,-1.58813948008651,
                0.399,-0.570837323486439,-3.81581488366837,3.58888212456784,-1.58733757069311,
                0.4,-0.568329646276025,-3.8163787979299,3.58860115001389,-1.58705156145513,
                0.401,-0.565625856279882,-3.81704285248753,3.58783484681363,-1.58652079438503,
                0.402,-0.563177276108952,-3.81593828202927,3.58508696269619,-1.58551407060185,
                0.403,-0.560583356527969,-3.81629391343476,3.58371155391444,-1.58431941866821,
                0.404,-0.558145639415261,-3.81616769567386,3.58313930950300,-1.58440859892940,
                0.405,-0.555839566169195,-3.81478739176461,3.57963674909459,-1.58278171449554,
                0.406,-0.553076933253471,-3.81619151259485,3.58097937586103,-1.58360415471231,
                0.407,-0.550478720237033,-3.81655823780655,3.58027428525217,-1.58338193201422,
                0.408,-0.547623904297565,-3.81987496129048,3.58602996973693,-1.58683970714702,
                0.409,-0.54513421777169,-3.82008341317758,3.58557135957367,-1.58682208684479,
                0.41,-0.542463864091653,-3.82091141176396,3.58508717163815,-1.58621146918108,
                0.411,-0.539930787766086,-3.82135059050214,3.58618584672602,-1.58772061642763,
                0.412,-0.53694380768913,-3.82443267437425,3.58981790300307,-1.58939667728795,
                0.413,-0.534085088394988,-3.82711888523446,3.59434052655698,-1.59228111788211,
                0.414,-0.531695751575027,-3.82740390652219,3.59412320681626,-1.59224897834463,
                0.415,-0.529015930071056,-3.82921927653907,3.59603385672054,-1.59305977656249,
                0.416,-0.526717731244366,-3.82741656569389,3.59056208484380,-1.58966054437938,
                0.417,-0.524102174136032,-3.82948393284729,3.59419926211461,-1.59204934515863,
                0.418,-0.521292165502983,-3.83204161024407,3.59792512198863,-1.59415613537559,
                0.419,-0.518702857245969,-3.83257826197208,3.59750520951009,-1.59382306214161,
                0.42,-0.516167605096383,-3.83190867404902,3.5942034962553,-1.59168300283293,
                0.421,-0.513725751388036,-3.83194698597079,3.59343332354535,-1.59138734094352,
                0.422,-0.511222061440922,-3.83190357910989,3.59219457689021,-1.59088219395654,
                0.423,-0.508655437312448,-3.83321052448785,3.59397669007241,-1.59190849813772,
                0.424,-0.506167018241098,-3.83366025040671,3.59346124341405,-1.59154872352149,
                0.425,-0.503650744192084,-3.83426345184935,3.59330234778074,-1.59133532170672,
                0.426,-0.501007095782485,-3.83507540255683,3.59339272138860,-1.59130778595467,
                0.427,-0.498216981763703,-3.83673551317982,3.59501722472497,-1.59207256126315,
                0.428,-0.495655143345463,-3.83652136421298,3.59249512362576,-1.59047447462090,
                0.429,-0.4929655257414,-3.83897061733117,3.59611347576565,-1.59256634706875,
                0.43,-0.490534946268041,-3.83873749420726,3.59376766236269,-1.59088831211697,
                0.431,-0.488164608195777,-3.83782388645828,3.59094003257209,-1.58946001898822,
                0.432,-0.485437081165382,-3.83904535882916,3.59142794452427,-1.58954813186381,
                0.433,-0.482948522579562,-3.83949246324153,3.59064255903995,-1.58882208764754,
                0.434,-0.480237840928177,-3.84190755487162,3.59438475966092,-1.59103464554296,
                0.435,-0.4776022801139,-3.84333238030975,3.59648573410691,-1.59268789849662,
                0.436,-0.474723751736894,-3.84620630221481,3.60100757269293,-1.59557093817102,
                0.437,-0.472196614490467,-3.84734004068798,3.60268013122545,-1.59681987594164,
                0.438,-0.469929493302054,-3.84690609851075,3.60189896560408,-1.59707478782457,
                0.439,-0.467299844808068,-3.8480421351316,3.60281181440524,-1.59754085073478,
                0.44,-0.464620348609723,-3.84944777610809,3.60469521110294,-1.59900021640582,
                0.441,-0.462072484721564,-3.84999442102398,3.60491023124138,-1.59954442162249,
                0.442,-0.45964216028977,-3.84936486993406,3.60241778240191,-1.59828252931515,
                0.443,-0.456957199789342,-3.85092651838683,3.60386460536494,-1.59896550493870,
                0.444,-0.454289316424066,-3.85299731156588,3.60688797940563,-1.60073944133330,
                0.445,-0.451615502487907,-3.85433499178615,3.6083942389403,-1.60186997440145,
                0.446,-0.448895976726177,-3.85585313487066,3.60941587087825,-1.60228316928585,
                0.447,-0.446544851413446,-3.85560384765130,3.60816862167918,-1.60177836672084,
                0.448,-0.444131989713139,-3.85543074769044,3.60714526678749,-1.6014341465418,
                0.449,-0.441504567127796,-3.85694180817861,3.60980232202377,-1.60356118320773,
                0.45,-0.438686728147755,-3.85945895648211,3.61305390706564,-1.60541325823972,
                0.451,-0.436141510423196,-3.86032510440866,3.61331843455352,-1.60546940081164,
                0.452,-0.433631616247697,-3.85978317222553,3.60988141447742,-1.60324317108336,
                0.453,-0.431212275719863,-3.8594934792628,3.60850160210886,-1.60279858803357,
                0.454,-0.428445175223968,-3.86184130124273,3.61175574016724,-1.60453315404825,
                0.455,-0.42590905314004,-3.86228424724492,3.61111102074812,-1.60419240787385,
                0.456,-0.423386845935181,-3.86235584219738,3.610434479822,-1.60429745751889,
                0.457,-0.420734493210983,-3.86404587413212,3.61230559967787,-1.60532092494100,
                0.458,-0.418263141671470,-3.86509515690033,3.61365347555745,-1.60624085489200,
                0.459,-0.415616922846757,-3.86663564868608,3.61528505926103,-1.60725367530742,
                0.46,-0.412823485720136,-3.86895622536112,3.61820442877609,-1.60877398333289,
                0.461,-0.410661258776916,-3.86626734421036,3.61071525243993,-1.60411369555484,
                0.462,-0.408154949546701,-3.86724668123058,3.61217892994199,-1.60545797939925,
                0.463,-0.40581040082568,-3.8666048766661,3.61018985033508,-1.60471053581337,
                0.464,-0.403235637421027,-3.86901946701014,3.61402758997899,-1.60690232140840,
                0.465,-0.401130843007604,-3.86673591434005,3.60901550072656,-1.60458818905491,
                0.466,-0.398436015315092,-3.86924671391527,3.61362411248533,-1.60776295181507,
                0.467,-0.395805181706527,-3.86999107399434,3.61336817095706,-1.60766541166049,
                0.468,-0.393045569621634,-3.87305704814744,3.61932199757017,-1.61165302377354,
                0.469,-0.390544497689406,-3.87323843100976,3.61886103528675,-1.61169185842893,
                0.47,-0.388071187864087,-3.87336323637708,3.61722380446906,-1.61055148959771,
                0.471,-0.385589406616797,-3.8745172341869,3.61893876058865,-1.61192278587290,
                0.472,-0.382958062237721,-3.87563623861036,3.61944010049971,-1.61207498636159,
                0.473,-0.380486799281158,-3.87532928732698,3.61720833525295,-1.61066064961497,
                0.474,-0.378275973802952,-3.87373009320602,3.61282931303063,-1.60816602288602,
                0.475,-0.375492407416626,-3.87600026895614,3.61565069519341,-1.60977012104422,
                0.476,-0.373375479019334,-3.87397020683640,3.61010716410805,-1.60661411263443,
                0.477,-0.370938711954631,-3.87412590737531,3.60990291594641,-1.60704501682505,
                0.478,-0.368327595754575,-3.87608527354954,3.61317847888635,-1.60931432955449,
                0.479,-0.366123826761826,-3.87458965094269,3.60861467545515,-1.60655524259459,
                0.48,-0.363790368515178,-3.87426112541141,3.6068726694225,-1.60560202477445,
                0.481,-0.361187852819402,-3.87582606722675,3.6094033007977,-1.60742533398299,
                0.482,-0.358620837540354,-3.87704107602202,3.6103841002106,-1.60801592129702,
                0.483,-0.355927587189329,-3.87789233881263,3.60994104523034,-1.60753211794648,
                0.484,-0.353399167171939,-3.87864900634969,3.60994688335476,-1.60749168710797,
                0.485,-0.350646014849946,-3.88104521413084,3.61361933079087,-1.60974706743575,
                0.486,-0.348145221616065,-3.88116907177949,3.61349035894664,-1.61029176100502,
                0.487,-0.345630685075981,-3.88229302463643,3.61536803753092,-1.61181232377554,
                0.488,-0.343131139318423,-3.88237604910219,3.61397724139754,-1.61098164757319,
                0.489,-0.340670247957346,-3.88246567546201,3.61321167094188,-1.6107743927333,
                0.49,-0.338026461869356,-3.88424974703105,3.61579541213533,-1.61244092671179,
                0.491,-0.335465665200474,-3.88650672450813,3.62012382354830,-1.61538220558804,
                0.492,-0.332946283953313,-3.88704553584545,3.61963720878812,-1.61507790213892,
                0.493,-0.330209712659514,-3.88832823035988,3.61970193872104,-1.61481473002524,
                0.494,-0.32761060057977,-3.88974033931453,3.62178685403423,-1.61623645481641,
                0.495,-0.325317082541889,-3.88982576913455,3.6217092979457,-1.61669131887972,
                0.496,-0.322997160226607,-3.88858088821813,3.61755456060637,-1.61421560844549,
                0.497,-0.320243970379449,-3.89134464861654,3.62257152743195,-1.61761347182263,
                0.498,-0.317885422192471,-3.89012134703295,3.61793354250208,-1.61464028104909,
                0.499,-0.315429636545669,-3.89032309484483,3.61676766399542,-1.61387744052190,
                0.5,-0.313164529287126,-3.88930586167676,3.61350605438768,-1.61207574264810,
                0.501,-0.310530437471946,-3.89090261845617,3.61532853914547,-1.61317140207974,
                0.502,-0.308017343527175,-3.89066631479042,3.61266438219682,-1.61139494522107,
                0.503,-0.305494370225709,-3.89139336248369,3.61298525848211,-1.61180968595751,
                0.504,-0.302753280497,-3.89304593014134,3.61494054859375,-1.61323960406908,
                0.505,-0.300323778899788,-3.89368145683367,3.61572986368159,-1.61408286512440,
                0.506,-0.297828048175178,-3.89412560118478,3.61522479568613,-1.61385881904375,
                0.507,-0.295342716116397,-3.89495350300135,3.61611868958164,-1.61482710662187,
                0.508,-0.292792625134292,-3.89538770007982,3.61552179576712,-1.6145041190162,
                0.509,-0.290474059009582,-3.89469296508578,3.61296190474274,-1.61309499377914,
                0.51,-0.288107601062283,-3.89364222354199,3.60927626672055,-1.61108369149206,
                0.511,-0.285242921501376,-3.89663205512698,3.61392639031061,-1.61393053347950,
                0.512,-0.282679505885769,-3.89871145902033,3.61833266016341,-1.61719527938789,
                0.513,-0.280016049892742,-3.90136211078932,3.62355470097339,-1.62081608271536,
                0.514,-0.277377066303406,-3.90313612931088,3.62637100084067,-1.62277939762886,
                0.515,-0.274999928295015,-3.90345744976699,3.62632112559781,-1.6229270139417,
                0.516,-0.272302165579552,-3.90598595004846,3.62997735309233,-1.62492536813347,
                0.517,-0.269857700645591,-3.90613528080899,3.6291994500881,-1.62457696857632,
                0.518,-0.267427955128817,-3.90646863355761,3.62875845927405,-1.62437990505198,
                0.519,-0.264918765572547,-3.90666972522178,3.62804743512073,-1.62418246820702,
                0.52,-0.262142406102521,-3.90815592078152,3.62822435340624,-1.62371925611663,
                0.521,-0.259691674453017,-3.90889147494533,3.62874896328696,-1.62412693566741,
                0.522,-0.257478767793105,-3.90675392636023,3.62290391174559,-1.62080140457753,
                0.523,-0.254845145916345,-3.90813955219265,3.62422075635541,-1.62153197072596,
                0.524,-0.252475572166829,-3.90726727249587,3.62082945310642,-1.61955297978752,
                0.525,-0.249962795521159,-3.90854535972348,3.62297128431627,-1.62114591406954,
                0.526,-0.2475974707616,-3.90836820083428,3.62203108043003,-1.62089628538310,
                0.527,-0.244873111944389,-3.90962396804623,3.62234547988816,-1.62090211268070,
                0.528,-0.242156717817925,-3.91067165336208,3.62196898986443,-1.62033029436538,
                0.529,-0.239596347277282,-3.91104358573913,3.62068931540432,-1.61942090704183,
                0.53,-0.237289721827293,-3.91046716903033,3.61884036445918,-1.6186323525031,
                0.531,-0.234907612586529,-3.91066972878006,3.61816379762162,-1.61825550079417,
                0.532,-0.232496476239410,-3.91068243483744,3.61689089985434,-1.61750422029595,
                0.533,-0.230070652130374,-3.91152878982494,3.61821586633395,-1.61873098748790,
                0.534,-0.227468575448456,-3.91264021947308,3.61968356645313,-1.61993902138211,
                0.535,-0.224969604899011,-3.91340089355991,3.62041329298200,-1.62062198505956,
                0.536,-0.222297673137179,-3.91566746602589,3.62389656083060,-1.62292133584726,
                0.537,-0.219816788393953,-3.91602244374617,3.62302801742375,-1.62235460211998,
                0.538,-0.217115920474515,-3.91706942129668,3.62313230720455,-1.62235227822994,
                0.539,-0.214447051047561,-3.91855116710123,3.62443134440724,-1.62312735673761,
                0.54,-0.211828861112248,-3.91928511686257,3.62319774902639,-1.62181558263259,
                0.541,-0.209312868847448,-3.91996963293075,3.62320547490001,-1.62190434212290,
                0.542,-0.206771232255502,-3.92027159237556,3.6221378735176,-1.62133925029595,
                0.543,-0.204385779325367,-3.91934542857145,3.61890365989586,-1.6197110319759,
                0.544,-0.202071469007499,-3.91786127822357,3.6152369657184,-1.61811386819266,
                0.545,-0.199676867454997,-3.91806409706279,3.61531727535131,-1.61881591520064,
                0.546,-0.19706307146305,-3.91874215231652,3.61545228847500,-1.61904244832138,
                0.547,-0.194437829094894,-3.91969504711675,3.61589086701669,-1.61943861692926,
                0.548,-0.191563493729181,-3.92257606986737,3.62031508852569,-1.62204672515663,
                0.549,-0.189131910773012,-3.92264390384262,3.62000130497706,-1.62251773254447,
                0.55,-0.186389352545821,-3.92484104526048,3.62250196094921,-1.62375775541685,
                0.551,-0.183483555600798,-3.92850400194515,3.62960606314388,-1.62848517985030,
                0.552,-0.181240345125545,-3.92642646856340,3.62394135816346,-1.62523332548940,
                0.553,-0.178735589354889,-3.9275594832911,3.62607095380528,-1.62716247308183,
                0.554,-0.175975157064629,-3.93041352602968,3.63055188055008,-1.62994602275312,
                0.555,-0.173486165338194,-3.93032879018613,3.62913972860008,-1.62929402369376,
                0.556,-0.171212928447782,-3.92960191717919,3.62751081010765,-1.62900617486185,
                0.557,-0.168815724346776,-3.92919717942357,3.62569924982858,-1.62808731584233,
                0.558,-0.166096687066349,-3.93154778405576,3.62890299965225,-1.62992009969676,
                0.559,-0.163653307922340,-3.93170953136934,3.627385454724,-1.62867131429379,
                0.56,-0.160931767616550,-3.9321605770862,3.62542552819988,-1.62703622780669,
                0.561,-0.158569692429690,-3.93173706068408,3.62398330384609,-1.62656849899933,
                0.562,-0.156293655546689,-3.93120226340109,3.62287470688216,-1.62657771247138,
                0.563,-0.153782293349286,-3.93126824031731,3.6212476028273,-1.62563815701366,
                0.564,-0.151243559979741,-3.93210566779021,3.6213168961451,-1.62554056583189,
                0.565,-0.148411945899056,-3.93369346999460,3.62204192101322,-1.62571384006306,
                0.566,-0.146046718889414,-3.93300638410359,3.62010130925175,-1.62492881284975,
                0.567,-0.143629879997270,-3.93262169520144,3.61840590516921,-1.62431148993405,
                0.568,-0.140962551315525,-3.9339167405087,3.61971549000433,-1.62519728499695,
                0.569,-0.138424540204146,-3.93455892144483,3.61951711952233,-1.62497368777580,
                0.57,-0.136060779129866,-3.9337779874868,3.61665466454712,-1.62325560563434,
                0.571,-0.133560175242574,-3.93483423659905,3.61826869723018,-1.62447562182234,
                0.572,-0.130928825074170,-3.93700232812807,3.62168189541094,-1.62670659200121,
                0.573,-0.128542706217448,-3.93626157994844,3.61931098132991,-1.62571926366576,
                0.574,-0.125776016834022,-3.93961431181171,3.62544942993267,-1.62946541278842,
                0.575,-0.123333570319116,-3.93858901849334,3.62206562621927,-1.62775043975480,
                0.576,-0.120921746879523,-3.93850658914084,3.6211894243515,-1.62753454071273,
                0.577,-0.118205904827404,-3.94000576589744,3.62267150960189,-1.62853764673612,
                0.578,-0.115427169618193,-3.94254804862516,3.62679434423145,-1.63119660636620,
                0.579,-0.112894229003064,-3.94247059640172,3.6251923497385,-1.63046711993071,
                0.58,-0.110324354882536,-3.94376173537647,3.62719610403341,-1.63198817281744,
                0.581,-0.107652323533668,-3.94683606481641,3.63386346845211,-1.63660278337073,
                0.582,-0.105139054208999,-3.94680242391582,3.63219682348675,-1.63575863613815,
                0.583,-0.102739043684876,-3.94550707685383,3.62798693937109,-1.63329261791882,
                0.584,-0.100066314030604,-3.94716591578717,3.63003708917596,-1.63449583614396,
                0.585,-0.0973499383964803,-3.94926197945608,3.63326635455863,-1.63658581332679,
                0.586,-0.0947248185376793,-3.95048287319734,3.63501592567269,-1.63796043595803,
                0.587,-0.0921102986863506,-3.9514764998904,3.63529795095752,-1.63796854024560,
                0.588,-0.0895428867280616,-3.95218260168342,3.63518284839868,-1.63785280416057,
                0.589,-0.0869872815842066,-3.95241106453545,3.63479225097947,-1.63805313725784,
                0.59,-0.0843575842648304,-3.95258834504693,3.63257973522726,-1.63614994292629,
                0.591,-0.0816455940068923,-3.95442032260616,3.63487305374073,-1.63752066957506,
                0.592,-0.079175622870696,-3.95449164361046,3.63388966865113,-1.63715427114754,
                0.593,-0.076542446715584,-3.95491079913774,3.63265475510068,-1.63616024644309,
                0.594,-0.0739384201476672,-3.9551769674515,3.63135199656635,-1.63530505287975,
                0.595,-0.0713130757314856,-3.95664574531472,3.63475736542337,-1.63822949219747,
                0.596,-0.0690830145469788,-3.95465934185769,3.63033521313139,-1.63620859063257,
                0.597,-0.0665971773525233,-3.95440855794641,3.62870407997714,-1.63557693228425,
                0.598,-0.0639705258944172,-3.95413897163676,3.62608768613478,-1.63403932735928,
                0.599,-0.0613124455136681,-3.95463754108762,3.62532310451891,-1.63349172230304,
                0.6,-0.0586319582175037,-3.95578165094446,3.62754338681094,-1.63560937391591,
                0.601,-0.0557764055424261,-3.95772054343140,3.63000100154261,-1.63718241081571,
                0.602,-0.0531294372451502,-3.95919154789171,3.63208506549587,-1.63870688987513,
                0.603,-0.0506692973212741,-3.95833664485453,3.62995921728573,-1.63821064201586,
                0.604,-0.0479805865098128,-3.95894233276866,3.6290369536324,-1.63758021580545,
                0.605,-0.0453260848655687,-3.96066861020838,3.63162551689094,-1.63927432201875,
                0.606,-0.0427783322304022,-3.96152344577134,3.6330416165376,-1.64062789093297,
                0.607,-0.0400982298066168,-3.96280378774621,3.63387381263612,-1.64101798060052,
                0.608,-0.0377952094993744,-3.96098370687795,3.62915327804351,-1.63852073821007,
                0.609,-0.0353259569240706,-3.96053365872551,3.62744855256360,-1.63791052410552,
                0.61,-0.0328278169615568,-3.96053826638798,3.62706745240191,-1.63822972562502,
                0.611,-0.0302749038291987,-3.96035093288158,3.62513473316464,-1.63720946708715,
                0.612,-0.0279017182581606,-3.95890618566358,3.62043768749497,-1.63424763046003,
                0.613,-0.0249940300382992,-3.96160046180114,3.62414877568016,-1.63634340281794,
                0.614,-0.0221438050907973,-3.96418885062318,3.62861721923335,-1.63935781198586,
                0.615,-0.0194202362579867,-3.96578369425927,3.63029899108743,-1.64028666061517,
                0.616,-0.0168197857787622,-3.96723123607604,3.63280440305105,-1.64204960457501,
                0.617,-0.0143079554925680,-3.96760740578463,3.63292837088884,-1.64228629076925,
                0.618,-0.0117024079655977,-3.96823625275121,3.63210209957984,-1.64150303193587,
                0.619,-0.00892012049818524,-3.96998635144199,3.63415609685403,-1.64277659095818,
                0.62,-0.00646882026736584,-3.97043943174044,3.63456984404714,-1.64332673454044,
                0.621,-0.00379701040109294,-3.97110172409696,3.63422250521225,-1.6431929998698,
                0.622,-0.000665960661056468,-3.97533053095352,3.64042348782038,-1.6465898541974,
                0.623,0.00198314591732017,-3.97584801749866,3.63958559421247,-1.64588728986334,
                0.624,0.00478478507436179,-3.97775526219195,3.64192579393909,-1.64727767227416,
                0.625,0.00746651109536662,-3.97913316433609,3.64418097339843,-1.64902352955281,
                0.626,0.0101277575168801,-3.98054033419874,3.64646167392403,-1.65080944001743,
                0.627,0.0128682574933519,-3.98202887087901,3.64824334979177,-1.65196473578577,
                0.628,0.0154859145396538,-3.98266096756383,3.64823562299981,-1.65201517070796,
                0.629,0.0183356968185324,-3.98432035847466,3.65003735283907,-1.65323148516051,
                0.63,0.0208795929798303,-3.98357954975395,3.64677643279636,-1.65131167538047,
                0.631,0.0230768756210406,-3.98020017289494,3.63910168848807,-1.64714951952673,
                0.632,0.0259358550732564,-3.98130662140621,3.6393403747038,-1.64723865078391,
                0.633,0.0284876001585880,-3.98107553461127,3.63786544837626,-1.64678577632971,
                0.634,0.0308520199846713,-3.97994797053091,3.63498476512142,-1.64547976003245,
                0.635,0.0335847918419032,-3.98051802698278,3.63439963343238,-1.64515532035497,
                0.636,0.0364117635534362,-3.98412322331972,3.64127117428797,-1.64950567370965,
                0.637,0.0392496224039894,-3.98501774852793,3.64089652686171,-1.64916257147332,
                0.638,0.0420393188097983,-3.98688599564193,3.64377721254766,-1.65113677020849,
                0.639,0.0445417903908882,-3.98664237944954,3.64238504312714,-1.65041782774655,
                0.64,0.0472800037742715,-3.98647921513986,3.63982026752426,-1.64879891062280,
                0.641,0.0499513471758749,-3.98726988024424,3.63994294467348,-1.64877659006641,
                0.642,0.0528273407528042,-3.98852825715105,3.64038466031695,-1.64901924342963,
                0.643,0.0553520971525819,-3.98876845047183,3.64079479144965,-1.64978619794816,
                0.644,0.0582055231195491,-3.99137501388115,3.64534197556615,-1.65277963028825,
                0.645,0.0611522932646966,-3.99447775327373,3.65063205499114,-1.65622135630818,
                0.646,0.0636159419968111,-3.99389497107312,3.64868823055355,-1.65536208170334,
                0.647,0.0664517898244417,-3.9948283403497,3.64888677193041,-1.65547284671200,
                0.648,0.0689403169255921,-3.99379762154951,3.64589985063322,-1.65394993927399,
                0.649,0.0715558893471382,-3.99339946062797,3.64314499673891,-1.65206418563406,
                0.65,0.0744731741237813,-3.99592988479154,3.64781322839099,-1.65535943324628,
                0.651,0.077324942157492,-3.99817221637123,3.65193893409821,-1.65818613981886,
                0.652,0.0800033194164449,-3.99741541139177,3.64831713135946,-1.65603968755172,
                0.653,0.0827688137665049,-3.99878797697026,3.64997897786693,-1.65732896089031,
                0.654,0.085800374805916,-4.00138336797437,3.65264366353991,-1.65848035691494,
                0.655,0.0884919340101932,-4.00097156888815,3.64983392968039,-1.65680349216174,
                0.656,0.0910725133412112,-4.00048341593188,3.647082365316,-1.6551213910614,
                0.657,0.0941353028559456,-4.00452127836436,3.65398186670999,-1.65915991631749,
                0.658,0.0966142279086006,-4.00305878488799,3.65001067321187,-1.65716843009901,
                0.659,0.0995108196274669,-4.00454418863663,3.65145498227056,-1.65794231657094,
                0.66,0.102066567640529,-4.00392635742054,3.64893215164602,-1.65645276891527,
                0.661,0.104983219529998,-4.0055391194906,3.65151489699324,-1.65848028089189,
                0.662,0.107792357317499,-4.00625303752516,3.65144406784991,-1.65858768988596,
                0.663,0.110474187589705,-4.00724882691327,3.65300827049159,-1.65993706692846,
                0.664,0.113272006945740,-4.00839769441901,3.654347372069,-1.66112296487192,
                0.665,0.115876979436991,-4.00790026061824,3.65181036765894,-1.65962601034352,
                0.666,0.118316854343624,-4.00557278011735,3.64539676087216,-1.65590960864144,
                0.667,0.121108659418050,-4.0070334523654,3.64738110136170,-1.65740614482068,
                0.668,0.124018362151472,-4.0091736837471,3.65071326451130,-1.65955749945046,
                0.669,0.126570341440286,-4.0083108475892,3.64857306989567,-1.65914064389276,
                0.67,0.129404280937327,-4.00886827420852,3.64776408816311,-1.65868432123706,
                0.671,0.132278491212072,-4.00948386693476,3.64703068424468,-1.65819759907608,
                0.672,0.13500574549772,-4.00816484477552,3.64224138898069,-1.65536192773022,
                0.673,0.137610309795001,-4.0066954339396,3.63702822613496,-1.65205957995619,
                0.674,0.14037157616879,-4.00740057319948,3.63738323106631,-1.65247129428925,
                0.675,0.143030961309149,-4.00743431409264,3.63703916543732,-1.65298435786877,
                0.676,0.145600466475735,-4.00702429624493,3.63520698198147,-1.65226875484231,
                0.677,0.148126364021358,-4.00606077007855,3.63205427032834,-1.65042468037267,
                0.678,0.150671536928558,-4.00534821928375,3.62984280882022,-1.64944287367559,
                0.679,0.154223789717727,-4.01184163413044,3.64147886944861,-1.65627639818824,
                0.68,0.156780182224937,-4.00990190746053,3.63564939036083,-1.65279125702208,
                0.681,0.159464031456444,-4.01024964551828,3.63637348557853,-1.65402196150959,
                0.682,0.162013784106891,-4.00920099181839,3.63253021540848,-1.65149593065807,
                0.683,0.164677915069849,-4.00977083964611,3.63395559350989,-1.65319752538006,
                0.684,0.167511383410534,-4.01024145684316,3.63318029583062,-1.65276238570997,
                0.685,0.170382706409384,-4.01078315612577,3.63203960204746,-1.65182388979565,
                0.686,0.173129089293392,-4.00994079729988,3.62949524299491,-1.65101967340007,
                0.687,0.176099319388519,-4.01148904716446,3.63105261823674,-1.65183480928539,
                0.688,0.178608129365716,-4.01026548195927,3.62748350470822,-1.64989206443437,
                0.689,0.181616717916302,-4.01099519710651,3.62734397875144,-1.65009026762688,
                0.69,0.184517861604009,-4.01336605413799,3.632355462953,-1.65393139232393,
                0.691,0.187260393636051,-4.01395079107074,3.63230706711237,-1.65383311008892,
                0.692,0.190252583481563,-4.01609677822453,3.63491046807129,-1.65507356274506,
                0.693,0.193160591405129,-4.01615773799099,3.63257682451801,-1.65334799572495,
                0.694,0.195768342619636,-4.01509597434669,3.62884379854413,-1.65116370449867,
                0.695,0.198666805422917,-4.01536165336537,3.62640337470184,-1.64914508951085,
                0.696,0.201390505197093,-4.01460493336435,3.62330944490406,-1.64741652444356,
                0.697,0.204069627156648,-4.01420746376267,3.62070051428367,-1.6458165377543,
                0.698,0.206858650187918,-4.01410901559618,3.61926723513935,-1.64530024818447,
                0.699,0.209704720229161,-4.01303865798137,3.61433530145922,-1.64211587691341,
                0.7,0.212619758984302,-4.01332460982238,3.61252748116971,-1.64091455228484,
                0.701,0.215475360200129,-4.0140270086681,3.61256689966334,-1.64105127274224,
                0.702,0.218769251361509,-4.01799681650452,3.61888674430097,-1.64476428345039,
                0.703,0.222214557612771,-4.02282277927972,3.62643789359712,-1.64888155880152,
                0.704,0.225128476522744,-4.0233724081618,3.62655034433631,-1.64929955672856,
                0.705,0.228007234574815,-4.02460510759639,3.62889352398139,-1.65131485744319,
                0.706,0.231260948963588,-4.02816393600481,3.63508778686218,-1.65517170683026,
                0.707,0.233993434212584,-4.02637825043803,3.62992355744408,-1.65237315895756,
                0.708,0.23677333196622,-4.0267877931962,3.63004833960449,-1.65281744099539,
                0.709,0.239546220367503,-4.02686847505655,3.62924166163602,-1.65286167251185,
                0.71,0.242567223925786,-4.0286555672917,3.63117969302076,-1.6537795366587,
                0.711,0.245340529729067,-4.02846587493922,3.62973676232393,-1.65316455576713,
                0.712,0.248231605571864,-4.02805343571056,3.62734708045577,-1.65198802506611,
                0.713,0.251189630141253,-4.02979852812198,3.63034975636089,-1.65418844531202,
                0.714,0.254126703339341,-4.02975806891529,3.62870028968494,-1.65352484492069,
                0.715,0.256971033964544,-4.02961015030143,3.6271799334121,-1.65296413716608,
                0.716,0.259835600706587,-4.02931232003894,3.62496717717473,-1.65174621037064,
                0.717,0.262399810361798,-4.02694942192708,3.61923410310756,-1.64889463320163,
                0.718,0.265220789121004,-4.02648518911399,3.61719510502435,-1.64813831890224,
                0.719,0.268195278015465,-4.02695723427868,3.61708632541438,-1.64849130185095,
                0.72,0.271337169727924,-4.02888085583504,3.61987601448441,-1.65046755060143,
                0.721,0.274174382457909,-4.02973696748933,3.62140068327499,-1.65186243562754,
                0.722,0.277214907214673,-4.03103795492000,3.62222695685266,-1.65229340486658,
                0.723,0.280390139876274,-4.03368841190879,3.62572518440002,-1.65407870195622,
                0.724,0.283218360160247,-4.03182584768485,3.61940997603830,-1.65008915789627,
                0.725,0.286313332717455,-4.03297583060265,3.62033065513327,-1.65080097072512,
                0.726,0.289388017782733,-4.03410767691518,3.62134751368320,-1.65152377614217,
                0.727,0.292427221538936,-4.03665875298975,3.62579563920553,-1.65434019865366,
                0.728,0.295405160834921,-4.03754104465601,3.62697712852267,-1.65564520016867,
                0.729,0.298399229425745,-4.03800334985168,3.62531118994748,-1.65413456245167,
                0.73,0.301771324334771,-4.04126208641258,3.6303769270183,-1.65726978413413,
                0.731,0.304599018719603,-4.04148220973654,3.63019288687503,-1.65751871325813,
                0.732,0.307861978590943,-4.04477708123696,3.63632353214083,-1.66146103139636,
                0.733,0.310831379531875,-4.04572226733921,3.63639313408725,-1.66123449862653,
                0.734,0.314304232826188,-4.05020960328249,3.64543880081414,-1.66729537621830,
                0.735,0.317452455434408,-4.05279470050699,3.64963072520091,-1.66995008962260,
                0.736,0.320311553602378,-4.05361436606083,3.65037696572322,-1.67060603716582,
                0.737,0.323537412345677,-4.05467528167255,3.65054108446305,-1.67079172096633,
                0.738,0.326283508525916,-4.05441842682918,3.65040428414037,-1.67152081941169,
                0.739,0.329525485499598,-4.05583632822612,3.65097266774654,-1.6717340855738,
                0.74,0.332570476084105,-4.05633945400761,3.65119212827666,-1.67207165836419,
                0.741,0.335386865943790,-4.05526741443018,3.64755621758792,-1.67007421992592,
                0.742,0.338265739426457,-4.05524551024284,3.64606894496661,-1.66919624465338,
                0.743,0.341001062661635,-4.05232239168614,3.63797116906615,-1.66437908227128,
                0.744,0.343821495034485,-4.05027122203217,3.63230900206547,-1.66129822940296,
                0.745,0.347056447135452,-4.05165137245717,3.6338999106679,-1.66250239943437,
                0.746,0.350234937421522,-4.05303705732836,3.63514725501902,-1.66335410480114,
                0.747,0.353512434243489,-4.05486048888656,3.63678416401830,-1.66427495079176,
                0.748,0.356485673026434,-4.05448174834346,3.63449369915658,-1.66323912481706,
                0.749,0.359945251038978,-4.05676675202988,3.63583804192669,-1.66329032923815,
                0.75,0.362662562240686,-4.05519846045301,3.63318295100052,-1.66278048312918,
                0.751,0.36594357865513,-4.05730110562177,3.6364138704584,-1.66508211468132,
                0.752,0.36894960520203,-4.05645807427895,3.63316226721607,-1.66327841248729,
                0.753,0.371841463617956,-4.05536110333235,3.63061728467048,-1.66251905260446,
                0.754,0.375249466385005,-4.05802829211603,3.63544663413999,-1.66590327833488,
                0.755,0.378436897879028,-4.05839207406872,3.63312517545486,-1.66377048319457,
                0.756,0.381679517832969,-4.0591861764678,3.63384962277599,-1.66486901323736,
                0.757,0.384911351752026,-4.06054539836798,3.63474657354962,-1.66535376811235,
                0.758,0.388251342529934,-4.06160362938896,3.63531916759648,-1.66590391796738,
                0.759,0.391766410345613,-4.06578185552651,3.64243646049805,-1.67014184079659,
                0.76,0.395228118270015,-4.06855678195532,3.64649709210638,-1.67253819976677,
                0.761,0.398035197582145,-4.06545143711918,3.63825876432667,-1.66763741199927,
                0.762,0.401195824700922,-4.06475843265527,3.63530177140573,-1.66616244682987,
                0.763,0.404376821720868,-4.06587106649254,3.63620682781686,-1.66669385659233,
                0.764,0.407207721262352,-4.06348366211286,3.62959103544397,-1.66287026049325,
                0.765,0.41057876406643,-4.06539279595764,3.63167176285795,-1.66422037610073,
                0.766,0.414015082152607,-4.06602028444658,3.63109098420367,-1.66419749624045,
                0.767,0.417541365861259,-4.06888791316408,3.63499360437471,-1.66633455062889,
                0.768,0.420844199081544,-4.07012193642187,3.6359440792134,-1.66699870534654,
                0.769,0.424176736837612,-4.07148269040447,3.63913830199024,-1.66994016332782,
                0.77,0.427472456198318,-4.07260716290089,3.64012777104130,-1.67070938259796,
                0.771,0.431073962594993,-4.07495276422273,3.64264621329005,-1.67207768541211,
                0.772,0.434543043767325,-4.07779532571992,3.6476966348234,-1.67537770484927,
                0.773,0.437923534378941,-4.07881981167794,3.64795069359138,-1.67564147086599,
                0.774,0.441938970788193,-4.08388694903672,3.65514367917654,-1.67962726110174,
                0.775,0.445365938218248,-4.08558335490501,3.65746867466318,-1.68114941737959,
                0.776,0.448733245163857,-4.08603722430602,3.65680604485996,-1.68110716078797,
                0.777,0.452007036508417,-4.08580838854923,3.65564448989134,-1.68120847993041,
                0.778,0.455381065767447,-4.08555383138826,3.65354689594698,-1.68034173132984,
                0.779,0.459160257869281,-4.09079087974189,3.66363415587412,-1.68672735222246,
                0.78,0.463011706039793,-4.09534481878575,3.67162396794101,-1.69191260128957,
                0.781,0.466125694069051,-4.0943651996748,3.66988953539014,-1.69212885204399,
                0.782,0.46975727085462,-4.09587592996055,3.66990270701241,-1.69151143587593,
                0.783,0.472966260954476,-4.09324431555695,3.66191177515599,-1.6867698910411,
                0.784,0.475935411900016,-4.09047861822377,3.65490525308234,-1.68304342496553,
                0.785,0.479487610881034,-4.09181688501071,3.65594183223511,-1.68380510724810,
                0.786,0.482835352104067,-4.09216586299945,3.65544684238863,-1.68387528410467,
                0.787,0.486524933427781,-4.09443291585528,3.65859325219556,-1.68580550486953,
                0.788,0.489820270373121,-4.09326130167434,3.65289080444581,-1.68171398654770,
                0.789,0.493372172030762,-4.09459455107837,3.65394094767485,-1.68238398400222,
                0.79,0.496708169963747,-4.09382264763109,3.6511219941089,-1.68121121955259,
                0.791,0.499727932063645,-4.09249363317117,3.64783900458884,-1.67985294771723,
                0.792,0.502978099804572,-4.09194615856522,3.64584785001656,-1.67913161392034,
                0.793,0.506326913521483,-4.09146884761268,3.64394252103461,-1.67860497325014,
                0.794,0.510165419285464,-4.09472184223512,3.64849270464846,-1.68120722967955,
                0.795,0.513720796033811,-4.09630764174715,3.65023543132397,-1.68225024224349,
                0.796,0.517080169605127,-4.09578478334503,3.64727840431159,-1.68041971761119,
                0.797,0.520439536635133,-4.09440523219976,3.64213006734639,-1.67731814152055,
                0.798,0.523874620651223,-4.09441218579054,3.64202094831859,-1.67818268153642,
                0.799,0.527816028867724,-4.09923159965109,3.65191665082269,-1.68484301307328,
                0.8,0.53167444145198,-4.10390558710927,3.66075695016868,-1.69032229096278,
                0.801,0.535148865620123,-4.10401952364965,3.65972086307676,-1.68995681260857,
                0.802,0.538272285716664,-4.10175822912284,3.65495666273535,-1.68805273982182,
                0.803,0.54209797438019,-4.10306004883884,3.65469331877713,-1.68757833242311,
                0.804,0.545555808637337,-4.10388815891540,3.65678436766270,-1.68996279352735,
                0.805,0.54890189752637,-4.10292722989611,3.65348011247846,-1.68824593982901,
                0.806,0.552545947705714,-4.10577339311793,3.66032360171156,-1.69346086847138,
                0.807,0.556251134646524,-4.10667037665123,3.65958984070258,-1.69279856903136,
                0.808,0.560048110306439,-4.1080604217431,3.65930952210996,-1.69193955685941,
                0.809,0.563719826308736,-4.10845040415113,3.65835810368528,-1.69160415936217,
                0.81,0.567596829915814,-4.11061414973578,3.66225174799730,-1.69476921519567,
                0.811,0.57127637453304,-4.11138555754679,3.66138900004663,-1.69399840008038,
                0.812,0.574855109462861,-4.11018814446111,3.65624269228725,-1.69077951495007,
                0.813,0.57878710282284,-4.11196761328273,3.65788497320641,-1.69190532997734,
                0.814,0.582800326131280,-4.11518321858287,3.66339392561897,-1.69554322735435,
                0.815,0.586643959640628,-4.1159475017431,3.66258432621493,-1.69488844008704,
                0.816,0.589955457153435,-4.11222636149422,3.65166431478995,-1.68800689968366,
                0.817,0.593499756515238,-4.11063734931246,3.64616613208229,-1.68476906105493,
                0.818,0.596935794157762,-4.1095788399473,3.64362587215349,-1.68415701307293,
                0.819,0.600804145218953,-4.11264916142605,3.64952080869894,-1.68814232788065,
                0.82,0.604629104283215,-4.11262342319998,3.64697210899915,-1.68669261128542,
                0.821,0.608382259332302,-4.1135667129787,3.64755709256734,-1.68715262716651,
                0.822,0.611954446948585,-4.11265992863855,3.64488535755283,-1.68616318358673,
                0.823,0.615402264026618,-4.11173854699933,3.64247653358363,-1.68536286137793,
                0.824,0.619303995061207,-4.11291539232054,3.64357549627307,-1.68639669538403,
                0.825,0.623054145177973,-4.11346692166287,3.64257354538556,-1.68576177729128,
                0.826,0.626917576610109,-4.11478251002026,3.64456996380077,-1.6875991859089,
                0.827,0.630708365850949,-4.11391142070792,3.64053906939592,-1.68520810784593,
                0.828,0.634121574944646,-4.11159786837475,3.63424150830582,-1.68159751961324,
                0.829,0.637815215158205,-4.1118021898017,3.63311317770872,-1.68082845467686,
                0.83,0.641413193204167,-4.11072585173765,3.63084240441968,-1.68032551761458,
                0.831,0.64533142863497,-4.11258879017304,3.63450486107872,-1.68317664893702,
                0.832,0.649276306690794,-4.11302923630565,3.63336220744220,-1.68263222220149,
                0.833,0.653389712535679,-4.11622894507776,3.63934134496082,-1.68708127366091,
                0.834,0.657660116293499,-4.12037647044834,3.64592277061582,-1.69074683582106,
                0.835,0.661438213265383,-4.11987598187565,3.64413394101203,-1.69029513526987,
                0.836,0.66532866524888,-4.11831199751066,3.6388119450646,-1.68705934650292,
                0.837,0.669428235414825,-4.11835493200572,3.63690663369442,-1.68626355729757,
                0.838,0.67355381768301,-4.12091197365807,3.6414284057098,-1.68951691196182,
                0.839,0.67742070409378,-4.12113215621159,3.64142094859799,-1.68983260335765,
                0.84,0.681324156543389,-4.12223660463988,3.64380389497904,-1.69204372299251,
                0.841,0.685078194515752,-4.12012311678616,3.63701361951976,-1.68773550945478,
                0.842,0.688744841704484,-4.11755590796209,3.63045658903144,-1.68431509449598,
                0.843,0.692758893248523,-4.11842582017904,3.63221422828788,-1.68632740651237,
                0.844,0.697514396130264,-4.12450991019141,3.64246035228164,-1.69224311330671,
                0.845,0.70157717559798,-4.12451020712559,3.64053379412441,-1.69110534402800,
                0.846,0.70628850269424,-4.12998135836618,3.64912759869161,-1.69554792154413,
                0.847,0.710617518312313,-4.13139419587149,3.65048163937030,-1.69673165337580,
                0.848,0.714492049598613,-4.12947950160149,3.64326132430638,-1.69165873606002,
                0.849,0.718544004432383,-4.12781454818213,3.63779912560994,-1.68872716819697,
                0.85,0.722811191643216,-4.12888164637999,3.63731165426656,-1.68820466772410,
                0.851,0.727264838304056,-4.13114137350901,3.64047855879031,-1.69018308242248,
                0.852,0.732223383515312,-4.13725516766823,3.65119118646552,-1.69674975428292,
                0.853,0.736436262163908,-4.13723359785383,3.64938057119996,-1.69591915515364,
                0.854,0.740469485583872,-4.13586807474527,3.64592539332364,-1.69455616523922,
                0.855,0.744991030525642,-4.13676296811673,3.6440901709914,-1.69289363543506,
                0.856,0.749350971111229,-4.13638903518408,3.64046785176257,-1.69055563272043,
                0.857,0.753806907029022,-4.13973636776524,3.64569008642036,-1.69349232501636,
                0.858,0.757783499279764,-4.13641824273515,3.63609030844556,-1.68732084237334,
                0.859,0.762428181094861,-4.13753408972759,3.63526325961185,-1.68658151985858,
                0.86,0.766822909529621,-4.13689457498849,3.63084773377512,-1.68361358336391,
                0.861,0.771530855164802,-4.13890818258276,3.63216461477146,-1.6842961738966,
                0.862,0.77608496347095,-4.14008255761135,3.63463258840534,-1.68709633159927,
                0.863,0.781497011656059,-4.14718803572859,3.64578398349129,-1.69328513101548,
                0.864,0.786278209288257,-4.14832527168748,3.64630924394851,-1.69401919390834,
                0.865,0.79077311397414,-4.14855388556684,3.64461808994988,-1.69262875681698,
                0.866,0.79516936117282,-4.14945688512492,3.64699671615161,-1.69529385614895,
                0.867,0.799647406554354,-4.14813262436602,3.6403976875537,-1.69060899994410,
                0.868,0.803959525703886,-4.14519268029578,3.6307024110884,-1.68414261841367,
                0.869,0.809213400492442,-4.15101862550632,3.6414063892449,-1.69076540343919,
                0.87,0.814347483478781,-4.15392749625535,3.64361096815828,-1.69117319369526,
                0.871,0.818842319999072,-4.15328502621999,3.64047869861143,-1.68938164153362,
                0.872,0.823701189949729,-4.15399298737023,3.64026669713184,-1.68958839627199,
                0.873,0.828728332121535,-4.15482625041083,3.63665393778088,-1.68587021960049,
                0.874,0.833684463964878,-4.15586305118571,3.63543948486368,-1.68496800536245,
                0.875,0.839465087080931,-4.16533604140599,3.65376038108591,-1.69618945025607,
                0.876,0.844175308160447,-4.16535601783901,3.651213954685,-1.69447213270127,
                0.877,0.848665536761373,-4.16360007111809,3.64631572381085,-1.69187968743939,
                0.878,0.853342477656969,-4.16342158332975,3.64465696447642,-1.69169759382903,
                0.879,0.858272066532483,-4.16599793034751,3.65048529869406,-1.69635480086222,
                0.88,0.862639085367948,-4.16486974299545,3.64814989185264,-1.69546259868308,
                0.881,0.867512435386794,-4.16606459377281,3.65004792845392,-1.69731416485456,
                0.882,0.872603831511072,-4.16836814522285,3.65231724533203,-1.69835982039796,
                0.883,0.878332816513715,-4.17465783202109,3.66245770924013,-1.70405631662069,
                0.884,0.883889381388308,-4.17973913751369,3.67149340936868,-1.70975212766462,
                0.885,0.889073019304857,-4.18005574469606,3.66860363683530,-1.70748031897591,
                0.886,0.894637468997383,-4.18405011787004,3.67473070680974,-1.71126824632854,
                0.887,0.899788043617345,-4.18551345972435,3.67565190348226,-1.71179316595557,
                0.888,0.904586698048942,-4.18279126268363,3.66692246408708,-1.70631930483384,
                0.889,0.909756103674878,-4.18133361184857,3.66039955058781,-1.70209271737385,
                0.89,0.914999279859368,-4.18119299654649,3.65554049511712,-1.69794178966569,
                0.891,0.920328630241533,-4.18315802648176,3.65635662228531,-1.69797209241874,
                0.892,0.925501776760443,-4.1807333675218,3.64803970513597,-1.69259825284413,
                0.893,0.931112001773453,-4.18272493119359,3.6472432012851,-1.69140202657657,
                0.894,0.936439341990093,-4.18441159281499,3.64816407330345,-1.69117874476256,
                0.895,0.941202279716706,-4.17935367771253,3.63557545250437,-1.68366355822605,
                0.896,0.946309818439999,-4.17673586979375,3.62595633694007,-1.67687197399371,
                0.897,0.951373356629674,-4.17533158678873,3.62218578365302,-1.67521376541201,
                0.898,0.95753226798921,-4.18146706441167,3.63141237362267,-1.67971976674894,
                0.899,0.963840342021206,-4.18763611984942,3.64217474314530,-1.68657118967815,
                0.9,0.969976540242773,-4.19157304611285,3.64509935840123,-1.68694578249643,
                0.901,0.97525845402004,-4.19007880386169,3.64126305535635,-1.6853167481406,
                0.902,0.981479839114347,-4.19682296401613,3.6525138287191,-1.69189796092837,
                0.903,0.987624531464742,-4.19959328887511,3.65540208447216,-1.69341787168932,
                0.904,0.99346473788373,-4.19846446730931,3.64801659436264,-1.68778898730470,
                0.905,0.999653257748778,-4.20056283001938,3.64845993292159,-1.68768762935301,
                0.906,1.00613666094386,-4.20478848760344,3.65285524446281,-1.68951374256954,
                0.907,1.01295871331626,-4.21371930153779,3.66914909761342,-1.69900041871874,
                0.908,1.01888925547811,-4.21439168315297,3.66833839981065,-1.69828329511023,
                0.909,1.02405437585782,-4.20900090045333,3.65447527753099,-1.68993092322330,
                0.91,1.03048231860914,-4.21276500525058,3.65932806865205,-1.69270502129079,
                0.911,1.03716956957494,-4.21957572233056,3.67174316411941,-1.70003736238138,
                0.912,1.04306185922122,-4.21659346040694,3.66027833020041,-1.69218699822400,
                0.913,1.04976479168839,-4.2204418524173,3.66562523610378,-1.69561932622268,
                0.914,1.05562174020948,-4.22017057511922,3.66387260891138,-1.69497468721938,
                0.915,1.06175260055169,-4.22112659997540,3.66287000817218,-1.69385273066728,
                0.916,1.06789991678678,-4.21782943204952,3.64947498833398,-1.68392823731792,
                0.917,1.07440079532998,-4.21584401375711,3.63772812753515,-1.67482983483684,
                0.918,1.08138566938123,-4.22144491438022,3.64708260387579,-1.68097176193263,
                0.919,1.08901571418996,-4.23013098010691,3.66069470676286,-1.68838427279711,
                0.92,1.09559697435470,-4.23468280731958,3.66889051887402,-1.69348698992183,
                0.921,1.10224333178833,-4.23478625884461,3.66441047274932,-1.68992403791112,
                0.922,1.10810548354611,-4.22846333196003,3.64971183148067,-1.68220786028486,
                0.923,1.11551474134846,-4.23142545833913,3.64879806422829,-1.67980455777005,
                0.924,1.12303158989555,-4.24016792676802,3.66451224289757,-1.68873915509976,
                0.925,1.13016029734230,-4.24134282464701,3.66231508444865,-1.68633324371906,
                0.926,1.13734528649259,-4.24348575137579,3.66202085885455,-1.68508552945949,
                0.927,1.14484096469920,-4.24589927207435,3.66220491862185,-1.68474010691931,
                0.928,1.15233055747203,-4.24904969922946,3.66377753491297,-1.68450630624912,
                0.929,1.15957645429140,-4.25134582631382,3.66495079977572,-1.68423977972337,
                0.93,1.16686238966524,-4.2531751080282,3.66598006847969,-1.68457752964466,
                0.931,1.17429283997644,-4.24994505711936,3.65165622923231,-1.67468686654178,
                0.932,1.18138719010938,-4.24725311784374,3.64090443508976,-1.66724073688675,
                0.933,1.18895408480501,-4.24764729181944,3.63713362415879,-1.66428736778616,
                0.934,1.19686208179560,-4.25304989099033,3.64593291797356,-1.66947538154412,
                0.935,1.20436219730830,-4.25245608807136,3.63919535986537,-1.66390620797331,
                0.936,1.21188665216602,-4.2526084780694,3.6369319781411,-1.66219038983389,
                0.937,1.21990742641084,-4.25416781995241,3.63574267975778,-1.66074070450557,
                0.938,1.22735879493967,-4.25013880912693,3.62267406828752,-1.65214495196491,
                0.939,1.23557484716390,-4.2494469822056,3.61392302917323,-1.64470135928816,
                0.94,1.24362544050649,-4.25186194135745,3.61723701908032,-1.64675082980283,
                0.941,1.25154123898643,-4.25028192625071,3.60940896209128,-1.64042093665558,
                0.942,1.25958057556572,-4.2496834419017,3.60644815487515,-1.63908588251067,
                0.943,1.26735297376234,-4.24301996429983,3.58956351328499,-1.62898354606393,
                0.944,1.27587875418115,-4.24063174891364,3.57799913533851,-1.62030689302482,
                0.945,1.28453471977722,-4.23549434912356,3.55734452892669,-1.60544337742361,
                0.946,1.29399469609062,-4.23889427434466,3.55785965188695,-1.60401636445517,
                0.947,1.30331512100884,-4.23928078047433,3.55228186426290,-1.59932840993854,
                0.948,1.31221357494921,-4.23636389501219,3.54212271369769,-1.59361464575932,
                0.949,1.32095775557462,-4.23734008759493,3.54210552985891,-1.59323042518020,
                0.95,1.33098732294457,-4.2403883703292,3.53978884827859,-1.58880471800764,
                0.951,1.34013714843643,-4.23925693833837,3.53502370640369,-1.58695735146303,
                0.952,1.35064283308746,-4.24227758651186,3.53480906001287,-1.58608692510053,
                0.953,1.35992048501142,-4.23761345156818,3.52262005644334,-1.579441325187,
                0.954,1.37015688985939,-4.23792656650827,3.51419674781938,-1.57133979898199,
                0.955,1.38112600227273,-4.244816881954,3.52651928855000,-1.57916379457499,
                0.956,1.39240048515364,-4.24843925173038,3.52722885696919,-1.57815984868139,
                0.957,1.40337092035274,-4.25439561787324,3.53619344048054,-1.58214269495212,
                0.958,1.41533487171869,-4.26266578900453,3.54683086823867,-1.58621312199977,
                0.959,1.42652437978271,-4.26580068603005,3.54878315497245,-1.58602410175725,
                0.96,1.43910435968959,-4.27465228366461,3.56014337121651,-1.59081240270856,
                0.961,1.44960663644312,-4.27341138239219,3.55801794194063,-1.59025125666518,
                0.962,1.46222715133827,-4.27995691068408,3.56733717789648,-1.59542921773813,
                0.963,1.47405085530046,-4.27738768388006,3.55611902714453,-1.58794880004848,
                0.964,1.48608702525895,-4.27400576313425,3.53824364268693,-1.57356186018667,
                0.965,1.4987362642616,-4.27509397043287,3.53191949521587,-1.56643815283755,
                0.966,1.51181343341075,-4.28357597993508,3.54796805837511,-1.5760003583852,
                0.967,1.52456018868295,-4.2769669994428,3.52692342983974,-1.56336799835828,
                0.968,1.53804919816634,-4.28215650855338,3.53433907706212,-1.5665278746765,
                0.969,1.55274046337729,-4.2901274829837,3.54583175495824,-1.57272378002843,
                0.97,1.56493763808565,-4.27716709422707,3.51129647968913,-1.54983618747264,
                0.971,1.58006086455337,-4.28157445672379,3.50932486789101,-1.54512146937984,
                0.972,1.594359229104,-4.27460784408418,3.48568428983034,-1.52915970277203,
                0.973,1.61003134168339,-4.27463040199757,3.47762653353840,-1.52281670768685,
                0.974,1.62674914492416,-4.27441848586651,3.46423473373235,-1.51074444834641,
                0.975,1.64439198095384,-4.28574578859019,3.47777891818512,-1.51458088897525,
                0.976,1.65991304858054,-4.27913011546233,3.46007997429315,-1.50283420524537,
                0.977,1.67803905660233,-4.27394923903332,3.43549192547953,-1.48502694678899,
                0.978,1.69763569534921,-4.28421976523284,3.44569650533361,-1.48779589739978,
                0.979,1.71621819389915,-4.27665162885681,3.41316288726304,-1.46258551433815,
                0.98,1.73880227644063,-4.29451035073313,3.44318134745213,-1.48089770160515,
                0.981,1.75955808759478,-4.29319110450795,3.42038393000961,-1.45870128115561,
                0.982,1.78009047657644,-4.27915607016476,3.37241668038065,-1.42293329569058,
                0.983,1.80326758668517,-4.28523550377287,3.38469442303006,-1.43336587702191,
                0.984,1.82530307587333,-4.26129504068871,3.32333985654891,-1.39501155055626,
                0.985,1.85430517201287,-4.27998538386709,3.34036593144971,-1.39953152485970,
                0.986,1.88069518965295,-4.27991897775931,3.32180105554352,-1.38139094252936,
                0.987,1.91254685466517,-4.31751437191922,3.39901249913303,-1.42915858038688,
                0.988,1.94528098069581,-4.33969728136701,3.42988342530575,-1.43993222372418,
                0.989,1.97806482209067,-4.34862743300756,3.43860971481185,-1.43899578286885,
                0.99,2.01319915198427,-4.35115116071872,3.4346489122946,-1.43581189864958,
                0.991,2.05127512449288,-4.34373632317636,3.39261823572262,-1.40184625835533,
                0.992,2.09654585043313,-4.36468271208351,3.39621676940168,-1.38594062806235,
                0.993,2.13833583352580,-4.3343719779922,3.33010690672014,-1.34828320349695,
                0.994,2.19720120176985,-4.39174666677393,3.44197727229641,-1.41126377323276,
                0.995,2.25611583972567,-4.36891603890258,3.36496380547627,-1.35250379056674,
                0.996,2.31952732883032,-4.2904007633479,3.20340873798628,-1.26397562951065,
                0.997,2.41564290708261,-4.26835430845816,3.0680858719057,-1.15154697202919,
                0.998,2.56609197069315,-4.49388344623851,3.50461774030133,-1.38669462983466,
                0.999,2.76299769897078,-4.40063470752084,3.29091539117092,-1.28094961717536,
                0.99925,2.84212480196315,-4.36262175913823,3.1853362386858,-1.21278982575716,
                0.9995,2.96209088168725,-4.41734900978279,3.36397764849639,-1.36511997376670,
                0.99975,3.10061360956949,-4.0412121862155,2.54139076254710,-0.867306115295149)

