% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BchronCalibrate.R
\name{BchronCalibrate}
\alias{BchronCalibrate}
\title{Fast radiocarbon calibration}
\usage{
BchronCalibrate(
  ages,
  ageSds,
  calCurves = rep("intcal20", length(ages)),
  ids = NULL,
  positions = NULL,
  pathToCalCurves = system.file("data", package = "Bchron"),
  allowOutside = FALSE,
  eps = 1e-05,
  dfs = rep(100, length(ages))
)
}
\arguments{
\item{ages}{A vector of ages provided in years before 1950.}

\item{ageSds}{A vector of 1-sigma values for the ages given above}

\item{calCurves}{A vector of values containing either \code{intcal20}, \code{shcal20}, \code{marine20}, or \code{normal} (older calibration curves are supposed such as intcal13). Should be the same length the number of ages supplied. Non-standard calibration curves can be used provided they are supplied in the same format as those previously mentioned and are placed in the same directory. Normal indicates a normally-distributed (non-14C) age.}

\item{ids}{ID names for each age}

\item{positions}{Position values (e.g. depths) for each age}

\item{pathToCalCurves}{File path to where the calibration curves are located. Defaults to the system directory where the 3 standard calibration curves are stored.}

\item{allowOutside}{Whether to allow calibrations to run outside the range of the calibration curve. By default this is turned off as calibrations outside of the range of the calibration curve can cause severe issues with probability ranges of calibrated dates}

\item{eps}{Cut-off point for density calculation. A value of eps>0 removes ages from the output which have negligible probability density}

\item{dfs}{Degrees-of-freedom values for the t-distribution associated with the calibration calculation. A large value indicates Gaussian distributions assumed for the 14C ages}
}
\value{
A list of lists where each element corresponds to a single age. Each element contains:
 \item{ages}{The original age supplied}
 \item{ageSds}{The original age standard deviation supplied}
 \item{positions}{The position of the age (usually the depth)}
 \item{calCurves}{The calibration curve used for that age}
 \item{ageGrid}{A grid of age values over which the density was created}
 \item{densities}{A vector of probability values indicating the probability value for each element in \code{ageGrid}}
 \item{ageLab}{The label given to the age variable}
 \item{positionLab}{The label given to the position variable}
}
\description{
A fast function for calibrating large numbers of radiocarbon dates involving multiple calibration curves
}
\details{
This function provides a direct numerical integration strategy for computing calibrated radiocarbon ages. The steps for each 14C age are approximately as follows:
1) Create a grid of ages covering the range of the calibration curve
2) Calculate the probability of each age according to the 14C age, the standard deviation supplied and the calibration curve
3) Normalise the probabilities so that they sum to 1
4) Remove any probabilities that are less than the value given for eps
Multiple calibration curves can be specified so that each 14C age can have a different curve. For ages that are not 14C, use the 'normal' calibration curve which treats the ages as normally distributed with given standard deviation
}
\examples{
# Calibrate a single age
ages1 <- BchronCalibrate(
  ages = 11553,
  ageSds = 230,
  calCurves = "intcal20",
  ids = "Date-1"
)
summary(ages1)
plot(ages1)

# Or plot with Calibration curve
plot(ages1, includeCal = TRUE)

# Calibrate multiple ages with different calibration curves
ages2 <- BchronCalibrate(
  ages = c(3445, 11553, 7456),
  ageSds = c(50, 230, 110),
  calCurves = c("intcal20", "intcal20", "shcal20")
)
summary(ages2)
plot(ages2)

# Calibrate multiple ages with multiple calibration curves and including depth
ages3 <- BchronCalibrate(
  ages = c(3445, 11553),
  ageSds = c(50, 230),
  positions = c(100, 150),
  calCurves = c("intcal20", "normal")
)
summary(ages3)
plot(ages3, withPositions = TRUE)
}
\seealso{
\code{\link{Bchronology}}, \code{\link{BchronRSL}}, \code{\link{BchronDensity}}, \code{\link{BchronDensityFast}}, \code{\link{createCalCurve}}
}
