% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get.boundary.R
\name{get.boundary}
\alias{get.boundary}
\title{Generate the optimal dose escalation and deescalation boundaries for conducting the trial.}
\usage{
get.boundary(
  target,
  ncohort,
  cohortsize,
  n.earlystop = 100,
  p.saf = 0.6 * target,
  p.tox = 1.4 * target,
  cutoff.eli = 0.95,
  extrasafe = FALSE,
  offset = 0.05
)
}
\arguments{
\item{target}{the target DLT rate}

\item{ncohort}{the total number of cohorts}

\item{cohortsize}{the cohort size}

\item{n.earlystop}{the early stopping parameter. If the number of patients treated at
the current dose reaches \code{n.earlystop}, stop the trial
and select the MTD based on the observed data. The default
value \code{n.earlystop=100} essentially turns off the type
of early stopping.}

\item{p.saf}{the highest toxicity probability that is deemed subtherapeutic
(i.e., below the MTD) such that dose escalation should be made.
The default value is \code{p.saf = 0.6 * target}.}

\item{p.tox}{the lowest toxicity probability that is deemed overly toxic such
that deescalation is required. The default value is
\code{p.tox=1.4*target}.}

\item{cutoff.eli}{the cutoff to eliminate an overly toxic dose for safety.
We recommend the default value (\code{cutoff.eli=0.95}) for general use.}

\item{extrasafe}{set \code{extrasafe=TRUE} to impose a more strict stopping rule for extra safety,
expressed as the stopping boundary value in the result .}

\item{offset}{a small positive number (between 0 and 0.5) to control how strict
the stopping rule is when \code{extrasafe=TRUE}. A larger value leads
to a more strict stopping rule. The default value
(\code{offset=0.05}) generally works well.}
}
\value{
\code{get.boundary()} returns a list object, including the dose escalation and de-escalation
         boundaries \code{$lambda_e} and \code{$lambda_d} and the corresponding decision tables
         \code{$boundary_tab} and \code{$full_boundary_tab}. If \code{extrasafe=TRUE}, the function also returns
         a (more strict) safety stopping boundary \code{$stop_boundary}.
}
\description{
Use this function to generate the optimal dose escalation and deescalation boundaries for conducting the trial.
}
\details{
The dose escalation and deescalation boundaries are all we need to run a
         phase I trial when using the BOIN design. The decision of which dose to
         administer to the next cohort of patients does not require complicated
         computations, but only a simple comparison of the observed DLT rate
         at the current dose with the dose escalation and deescalation boundaries.
         If the observed DLT rate at the current dose is smaller than or equal
         to the escalation boundary, we escalate the dose; if the observed toxicity
         rate at the current dose is greater than or equal to the deescalation boundary,
         we deescalate the dose; otherwise, we retain the current dose. The dose
         escalation and deescalation boundaries are chosen to minimize the probability
         of assigning patients to subtherapeutic or overly toxic doses, thereby
         optimizing patient ethics. \code{get.boundary()} also outputs the elimination
         boundary, which is used to avoid treating patients at overly toxic doses based
         on the following Bayesian safety rule: if \eqn{Pr(p_j > \phi | m_j , n_j ) > 0.95} and
         \eqn{n_j \ge 3}, dose levels \eqn{j} and higher are eliminated from the trial, where \eqn{p_j} is
         the toxicity probability of dose level \eqn{j}, \eqn{\phi} is the target DLT rate,
         and \eqn{m_j} and \eqn{n_j} are the number of toxicities and patients treated at dose level \eqn{j}.
         The trial is terminated if the lowest dose is eliminated.


         The BOIN design has two built-in stopping rules: (1) stop the trial if the lowest dose is eliminated
         due to toxicity, and no dose should be selected as the MTD; and (2) stop the trial
         and select the MTD if the number of patients treated at the current dose reaches
         \code{n.earlystop}. The first stopping rule is a safety rule to protect patients
         from the case in which all doses are overly toxic. The rationale for the second
         stopping rule is that when there is a large number (i.e., \code{n.earlystop})
         of patients assigned to a dose, it means that the dose-finding algorithm has
         approximately converged. Thus, we can stop the trial early and select the MTD
         to save the sample size and reduce the trial duration. For some applications,
         investigators may prefer a more strict safety stopping rule than rule (1) for
         extra safety when the lowest dose is overly toxic. This can be achieved by
         setting \code{extrasafe=TRUE}, which imposes the following more strict safety
         stopping rule: stop the trial if (i) the number of patients treated at the
         lowest dose >=3, and (ii) \eqn{Pr(toxicity\ rate\ of\ the\ lowest\ dose > \code{target} | data)
         > \code{cutoff.eli}-\code{offset}}. As a tradeoff, the strong stopping rule will decrease the
         MTD selection percentage when the lowest dose actually is the MTD.
}
\note{
We should avoid setting the values of \code{p.saf} and \code{p.tox} very close to the
      \code{target}. This is because the small sample sizes of typical phase I trials prevent us from
      differentiating the target DLT rate from the rates close to it. In addition,
      in most clinical applications, the target DLT rate is often a rough guess,
      and finding a dose level with a DLT rate reasonably close to the target rate
      will still be of interest to the investigator. The default values provided by
      \code{get.boundary()} are generally reasonable for most clinical applications.
}
\examples{

## get the dose escalation and deescalation boundaries for BOIN design with
## the target DLT rate of 0.3, maximum sample size of 30, and cohort size of 3
bound <- get.boundary(target=0.3, ncohort=10, cohortsize=3)
summary(bound) # get the descriptive summary of the boundary
plot(bound)    # plot the flowchart of the design with boundaries

}
\references{
Liu S. and Yuan, Y. (2015). Bayesian Optimal Interval Designs for Phase I
            Clinical Trials, \emph{Journal of the Royal Statistical Society: Series C}, 64, 507-523.

            Yuan Y., Hess K.R., Hilsenbeck S.G. and Gilbert M.R. (2016). Bayesian Optimal Interval Design: A
       Simple and Well-performing Design for Phase I Oncology Trials, \emph{Clinical Cancer Research}, 22, 4291-4301.
}
\seealso{
Tutorial: \url{http://odin.mdacc.tmc.edu/~yyuan/Software/BOIN/BOIN2.6_tutorial.pdf}

         Paper: \url{http://odin.mdacc.tmc.edu/~yyuan/Software/BOIN/paper.pdf}
}
\author{
Suyu Liu and Ying Yuan
}
